/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.lazy.primitive;

import com.gs.collections.api.CharIterable;
import com.gs.collections.api.LazyCharIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.bag.primitive.MutableCharBag;
import com.gs.collections.api.block.function.primitive.CharToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.CharPredicate;
import com.gs.collections.api.block.procedure.primitive.CharProcedure;
import com.gs.collections.api.list.primitive.MutableCharList;
import com.gs.collections.api.set.primitive.MutableCharSet;
import com.gs.collections.impl.bag.mutable.primitive.CharHashBag;
import com.gs.collections.impl.block.factory.primitive.CharPredicates;
import com.gs.collections.impl.factory.primitive.CharSets;
import com.gs.collections.impl.list.mutable.primitive.CharArrayList;
import com.gs.collections.impl.set.mutable.primitive.CharHashSet;
import com.gs.collections.impl.utility.internal.primitive.CharIterableIterate;
import com.gs.collections.impl.utility.primitive.LazyCharIterate;

import java.util.NoSuchElementException;

/**
 * This file was automatically generated from template file abstractLazyPrimitiveIterable.stg.
 *
 * @since 5.0
 */
public abstract class AbstractLazyCharIterable implements LazyCharIterable
{
    public int size()
    {
        return this.count(CharPredicates.alwaysTrue());
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public boolean isEmpty()
    {
        return CharIterableIterate.isEmpty(this);
    }

    public boolean notEmpty()
    {
        return CharIterableIterate.notEmpty(this);
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        CharIterableIterate.appendString(this, appendable, start, separator, end);
    }

    public boolean contains(char value)
    {
        return this.anySatisfy(CharPredicates.equal(value));
    }

    public boolean containsAll(char... source)
    {
        return this.containsAll(CharSets.immutable.of(source));
    }

    public boolean containsAll(CharIterable source)
    {
        return source.allSatisfy(new CharPredicate()
        {
            public boolean accept(char value)
            {
                return AbstractLazyCharIterable.this.contains(value);
            }
        });
    }

    public LazyCharIterable select(CharPredicate predicate)
    {
        return LazyCharIterate.select(this, predicate);
    }

    public LazyCharIterable reject(CharPredicate predicate)
    {
        return LazyCharIterate.select(this, CharPredicates.not(predicate));
    }

    public <V> LazyIterable<V> collect(CharToObjectFunction<? extends V> function)
    {
        return LazyCharIterate.collect(this, function);
    }

    public char detectIfNone(CharPredicate predicate, char ifNone)
    {
        return CharIterableIterate.detectIfNone(this, predicate, ifNone);
    }

    public int count(CharPredicate predicate)
    {
        return CharIterableIterate.count(this, predicate);
    }

    public boolean anySatisfy(CharPredicate predicate)
    {
        return CharIterableIterate.anySatisfy(this, predicate);
    }

    public boolean allSatisfy(CharPredicate predicate)
    {
        return CharIterableIterate.allSatisfy(this, predicate);
    }

    public boolean noneSatisfy(CharPredicate predicate)
    {
        return CharIterableIterate.noneSatisfy(this, predicate);
    }

    public <T> T injectInto(T injectedValue, ObjectCharToObjectFunction<? super T, ? extends T> function)
    {
        return CharIterableIterate.injectInto(this, injectedValue, function);
    }

    public char[] toArray()
    {
        return this.toList().toArray();
    }

    public MutableCharList toList()
    {
        final MutableCharList list = new CharArrayList();
        this.forEach(new CharProcedure()
        {
            public void value(char each)
            {
                list.add(each);
            }
        });
        return list;
    }

    public MutableCharSet toSet()
    {
        final MutableCharSet set = new CharHashSet();
        this.forEach(new CharProcedure()
        {
            public void value(char each)
            {
                set.add(each);
            }
        });
        return set;
    }

    public MutableCharBag toBag()
    {
        final MutableCharBag bag = new CharHashBag();
        this.forEach(new CharProcedure()
        {
            public void value(char each)
            {
                bag.add(each);
            }
        });
        return bag;
    }

    public long sum()
    {
        CharSumProcedure procedure = new CharSumProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public char max()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        CharMaxProcedure procedure = new CharMaxProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public char maxIfEmpty(char ifEmpty)
    {
        if (this.isEmpty())
        {
            return ifEmpty;
        }
        return this.max();
    }

    public char min()
    {
        if (this.isEmpty())
        {
            throw new NoSuchElementException();
        }
        CharMinProcedure procedure = new CharMinProcedure();
        this.forEach(procedure);
        return procedure.getValue();
    }

    public char minIfEmpty(char ifEmpty)
    {
        if (this.isEmpty())
        {
            return ifEmpty;
        }
        return this.min();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        char[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            char first = sortedArray[middleIndex];
            char second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public char[] toSortedArray()
    {
        return this.toSortedList().toArray();
    }

    public MutableCharList toSortedList()
    {
        return CharArrayList.newList(this).sortThis();
    }

    private static final class CharMaxProcedure implements CharProcedure
    {
        private boolean visitedOnce;
        private char max;

        public void value(char each)
        {
            if (this.visitedOnce)
            {
                if (this.max < each)
                {
                    this.max = each;
                }
            }
            else
            {
                this.max = each;
                this.visitedOnce = true;
            }
        }

        public char getValue()
        {
            return this.max;
        }
    }

    private static final class CharMinProcedure implements CharProcedure
    {
        private boolean visitedOnce;
        private char min;

        public void value(char each)
        {
            if (this.visitedOnce)
            {
                if (each < this.min)
                {
                    this.min = each;
                }
            }
            else
            {
                this.min = each;
                this.visitedOnce = true;
            }
        }

        public char getValue()
        {
            return this.min;
        }
    }

    private static final class CharSumProcedure implements CharProcedure
    {
        private char sum = 0;

        public void value(char each)
        {
            this.sum += each;
        }

        public char getValue()
        {
            return this.sum;
        }
    }

    public LazyCharIterable asLazy()
    {
        return this;
    }
}
