/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.internal.primitive;

import java.util.Collection;

import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectLongToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.block.procedure.primitive.LongProcedure;
import com.gs.collections.api.collection.primitive.MutableLongCollection;
import com.gs.collections.api.iterator.LongIterator;

/**
 * The LongIteratorIterate class provides implementations of the various iteration patterns for use with the {@link LongIterator}.
 * This file was automatically generated from template file primitiveIteratorIterate.stg.
 *
 * @since 5.0
 */
public final class LongIteratorIterate
{
    private LongIteratorIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    public static void forEach(LongIterator iterator, LongProcedure procedure)
    {
        while (iterator.hasNext())
        {
            procedure.value(iterator.next());
        }
    }

    public static <R extends MutableLongCollection> R select(
            LongIterator iterator,
            LongPredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            long item = iterator.next();
            if (predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <R extends MutableLongCollection> R reject(
            LongIterator iterator,
            LongPredicate predicate,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            long item = iterator.next();
            if (!predicate.accept(item))
            {
                targetCollection.add(item);
            }
        }
        return targetCollection;
    }

    public static <V, R extends Collection<V>> R collect(
            LongIterator iterator,
            LongToObjectFunction<? extends V> function,
            R targetCollection)
    {
        while (iterator.hasNext())
        {
            long item = iterator.next();
            targetCollection.add(function.valueOf(item));
        }
        return targetCollection;
    }

    public static long detectIfNone(LongIterator iterator, LongPredicate predicate, long ifNone)
    {
        while (iterator.hasNext())
        {
            long item = iterator.next();
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public static int count(LongIterator iterator, LongPredicate predicate)
    {
        int count = 0;
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                count++;
            }
        }
        return count;
    }

    public static boolean anySatisfy(LongIterator iterator, LongPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return true;
            }
        }
        return false;
    }

    public static boolean allSatisfy(LongIterator iterator, LongPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (!predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static boolean noneSatisfy(LongIterator iterator, LongPredicate predicate)
    {
        while (iterator.hasNext())
        {
            if (predicate.accept(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public static <T> T injectInto(LongIterator iterator, T injectedValue, ObjectLongToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        while (iterator.hasNext())
        {
            result = function.valueOf(result, iterator.next());
        }
        return result;
    }


    public static long sum(LongIterator iterator)
    {
        long sum = 0L;
        while (iterator.hasNext())
        {
            sum += iterator.next();
        }
        return sum;
    }

    public static long max(LongIterator iterator)
    {
        long max = iterator.next();
        while (iterator.hasNext())
        {
            long next = iterator.next();
            if (max < next)
            {
                max = next;
            }
        }
        return max;
    }

    public static long min(LongIterator iterator)
    {
        long min = iterator.next();
        while (iterator.hasNext())
        {
            long next = iterator.next();
            if (next < min)
            {
                min = next;
            }
        }
        return min;
    }
}
