/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.ProducerFencedException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.processor.StreamPartitioner;
import org.apache.kafka.streams.processor.internals.RecordCollector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RecordCollectorImpl
implements RecordCollector {
    private static final int MAX_SEND_ATTEMPTS = 3;
    private static final long SEND_RETRY_BACKOFF = 100L;
    private static final Logger log = LoggerFactory.getLogger(RecordCollectorImpl.class);
    private final Producer<byte[], byte[]> producer;
    private final Map<TopicPartition, Long> offsets;
    private final String logPrefix;
    private volatile Exception sendException;

    public RecordCollectorImpl(Producer<byte[], byte[]> producer, String streamTaskId) {
        this.producer = producer;
        this.offsets = new HashMap<TopicPartition, Long>();
        this.logPrefix = String.format("task [%s]", streamTaskId);
    }

    @Override
    public <K, V> void send(String topic, K key, V value, Long timestamp, Serializer<K> keySerializer, Serializer<V> valueSerializer, StreamPartitioner<? super K, ? super V> partitioner) {
        Integer partition = null;
        if (partitioner != null) {
            List partitions = this.producer.partitionsFor(topic);
            if (partitions.size() > 0) {
                partition = partitioner.partition(key, value, partitions.size());
            } else {
                throw new StreamsException("Could not get partition information for topic '" + topic + "'." + " This can happen if the topic does not exist.");
            }
        }
        this.send(topic, key, value, partition, timestamp, keySerializer, valueSerializer);
    }

    @Override
    public <K, V> void send(final String topic, K key, V value, Integer partition, Long timestamp, Serializer<K> keySerializer, Serializer<V> valueSerializer) {
        this.checkForException();
        byte[] keyBytes = keySerializer.serialize(topic, key);
        byte[] valBytes = valueSerializer.serialize(topic, value);
        ProducerRecord serializedRecord = new ProducerRecord(topic, partition, timestamp, (Object)keyBytes, (Object)valBytes);
        for (int attempt = 1; attempt <= 3; ++attempt) {
            try {
                this.producer.send(serializedRecord, new Callback(){

                    public void onCompletion(RecordMetadata metadata, Exception exception) {
                        if (exception == null) {
                            if (RecordCollectorImpl.this.sendException != null) {
                                return;
                            }
                            TopicPartition tp = new TopicPartition(metadata.topic(), metadata.partition());
                            RecordCollectorImpl.this.offsets.put(tp, metadata.offset());
                        } else if (RecordCollectorImpl.this.sendException == null) {
                            RecordCollectorImpl.this.sendException = exception;
                            if (RecordCollectorImpl.this.sendException instanceof ProducerFencedException) {
                                log.error("{} Error sending record to topic {}. No more offsets will be recorded for this task and it will be closed as it is a zombie.", new Object[]{RecordCollectorImpl.this.logPrefix, topic, exception});
                            } else {
                                log.error("{} Error sending record to topic {}. No more offsets will be recorded for this task and the exception will eventually be thrown", new Object[]{RecordCollectorImpl.this.logPrefix, topic, exception});
                            }
                        }
                    }
                });
                return;
            }
            catch (TimeoutException e) {
                if (attempt == 3) {
                    throw new StreamsException(String.format("%s Failed to send record to topic %s after %d attempts", this.logPrefix, topic, attempt));
                }
                log.warn("{} Timeout exception caught when sending record to topic {} attempt {}", new Object[]{this.logPrefix, topic, attempt});
                Utils.sleep((long)100L);
                continue;
            }
        }
    }

    private void checkForException() {
        if (this.sendException != null) {
            if (this.sendException instanceof ProducerFencedException) {
                throw (ProducerFencedException)this.sendException;
            }
            throw new StreamsException(String.format("%s exception caught when producing", this.logPrefix), this.sendException);
        }
    }

    @Override
    public void flush() {
        log.debug("{} Flushing producer", (Object)this.logPrefix);
        this.producer.flush();
        this.checkForException();
    }

    @Override
    public void close() {
        log.debug("{} Closing producer", (Object)this.logPrefix);
        this.producer.close();
        this.checkForException();
    }

    @Override
    public Map<TopicPartition, Long> offsets() {
        return this.offsets;
    }

    Producer<byte[], byte[]> producer() {
        return this.producer;
    }
}

