package org.tecgraf.tdk.cache;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.geotools.data.DefaultQuery;
import org.geotools.data.FeatureReader;
import org.geotools.data.FeatureStore;
import org.geotools.data.Query;
import org.geotools.data.Transaction;
import org.geotools.factory.CommonFactoryFinder;
import org.geotools.factory.GeoTools;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureCollections;
import org.geotools.feature.FeatureIterator;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.feature.simple.SimpleFeatureImpl;
import org.opengis.feature.FeatureFactory;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.filter.Filter;
import org.opengis.filter.FilterFactory2;
import org.opengis.filter.identity.FeatureId;

public class SimpleCachingFeatureStore extends SimpleCachingFeatureSource implements CachingFeatureStore<SimpleFeatureType,SimpleFeature> {

	private FeatureStore<SimpleFeatureType,SimpleFeature> _wrapped;
	private FeatureCacher<SimpleFeatureType, SimpleFeature> _cacher;
	private FilterFactory2 filterFactory = CommonFactoryFinder.getFilterFactory2(GeoTools.getDefaultHints());
	
	public SimpleCachingFeatureStore(
			FeatureStore<SimpleFeatureType, SimpleFeature> original, FeatureCacher<SimpleFeatureType, SimpleFeature> cacher)
	{
		super(original, cacher);
		_wrapped = original;
		_cacher = cacher;
	}

	@Override
	public List<FeatureId> addFeatures(
			FeatureCollection<SimpleFeatureType, SimpleFeature> collection)
			throws IOException {
		
		List<FeatureId> featureIds =  _wrapped.addFeatures(collection);
		
		if(collection.size() == 0 ) return featureIds;
		
		SimpleFeatureBuilder builder = new SimpleFeatureBuilder(collection.getSchema());
		
		
		FeatureCollection<SimpleFeatureType, SimpleFeature> newFeaturesCollection = FeatureCollections.newCollection();
		
		FeatureIterator<SimpleFeature> featureIterator = collection.features();
		try
		{
			int index = 0 ; 
			while(featureIterator.hasNext())
			{
				
				SimpleFeature simpleFeature = builder.buildFeature(featureIds.get(index).getID(), featureIterator.next().getAttributes().toArray());
				newFeaturesCollection.add(simpleFeature);
				index++;
			}
		}finally
		{
			featureIterator.close();
		}
		
		Query query = new DefaultQuery(collection.getSchema().getTypeName(),filterFactory.id(new HashSet<FeatureId>(featureIds)));
		_cacher.add(newFeaturesCollection, query);
		
		return featureIds;
	}

	@Override
	public Transaction getTransaction() {
		return _wrapped.getTransaction();
	}

	@Override
	public void modifyFeatures(AttributeDescriptor[] type, Object[] value,
			Filter filter) throws IOException {
		
		_wrapped.modifyFeatures(type, value, filter);
		
		FeatureCollection<SimpleFeatureType,SimpleFeature> featuresToUpdate = getFeatures(filter);
		FeatureIterator<SimpleFeature> featureIterator = featuresToUpdate.features();
		try
		{
			while(featureIterator.hasNext())
			{
				SimpleFeature featureToUpdate = featureIterator.next();
				int index = 0;
				for(AttributeDescriptor attributeDescriptor : type)
				{
					featureToUpdate.setAttribute(attributeDescriptor.getLocalName(), value[index]);
					index++;
				}
				
				_cacher.update(featureToUpdate);
			}
		}
		finally
		{
			featureIterator.close();
		}
		
		
		
	}

	@Override
	public void modifyFeatures(AttributeDescriptor type, Object value,
			Filter filter) throws IOException {
		modifyFeatures(new AttributeDescriptor[]{type}, new Object[]{value}, filter);
		
	}

	@Override
	public void removeFeatures(Filter filter) throws IOException 
	{
		_wrapped.removeFeatures(filter);
		
		FeatureCollection<SimpleFeatureType,SimpleFeature> collection = getFeatures(filter);
		FeatureIterator<SimpleFeature> featureIterator = collection.features();
		try
		{
			while(featureIterator.hasNext())
			{
				_cacher.remove(featureIterator.next());
			}
			
		}
		finally
		{
			featureIterator.close();
		}
		
	}

	@Override
	public void setFeatures(
			FeatureReader<SimpleFeatureType, SimpleFeature> reader)
			throws IOException 
	{
		//_wrapped.setFeatures(reader);
		_wrapped.removeFeatures(Filter.INCLUDE);
		_cacher.clear();
		FeatureCollection<SimpleFeatureType,SimpleFeature> featureCollection = FeatureCollections.newCollection();
		
		while(reader.hasNext())
		{
			featureCollection.add(reader.next());
		}
		
		addFeatures(featureCollection);
	}

	@Override
	public void setTransaction(Transaction transaction) {
		_wrapped.setTransaction(transaction);
	}

}
