/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2009(c).
 * 
 * file: AbstractDataStoreFactoryDecorator.java
 * created: Dec 17, 2009
 */
package org.tecgraf.tdk.data;

import java.awt.RenderingHints.Key;
import java.io.IOException;
import java.io.Serializable;
import java.util.Map;

import org.geotools.data.DataStore;
import org.geotools.data.DataStoreFactorySpi;
import org.tecgraf.tdk.cache.SimpleCachingDataStore;

/**
 * Abstract base class to create DatastoreFactories decorators, to implement a existing 
 * DataStore with cache.
 * @author fabiomano
 * @since gt-caching-ds 1.0
 */
public abstract class AbstractDataStoreFactoryDecorator implements DataStoreFactorySpi
{
    public static final Param CACHE_TYPE = new Param("cache type", CachingType.class,"cache type",true,CachingType.SIMPLECACHING);
    
    protected static final String CACHED_SUFIX = " - cached";
    
    protected DataStoreFactorySpi _target;

    
    public AbstractDataStoreFactoryDecorator(DataStoreFactorySpi target)
    {
        if (target == null)
            throw new IllegalArgumentException("target can't be null");
        _target = target;
    }
    
    /**
     * Creates the target datastore this DataStoreFactory is decorating.
     */
    public abstract DataStore createTargetDataStore(Map<String, Serializable> params) throws IOException;
    
    /* (non-Javadoc)
     * @see org.geotools.data.DataStoreFactorySpi#createDataStore(java.util.Map)
     */
    @Override
    public DataStore createDataStore(Map<String, Serializable> params) throws IOException
    {
        if (params == null)
            throw new IllegalArgumentException("params should not be null.");

        CachingType cacheType = getCachingType(params);
        
        if (cacheType == null)
            throw new IllegalArgumentException("Could not find the caching type parameter for createDataStore.");
        
        switch (getCachingType(params))
        {
            case SIMPLECACHING:
                return new SimpleCachingDataStore(createTargetDataStore(params));
            default:
                throw new UnsupportedOperationException("Your library version does not have suport for " + getCachingType(params));
        }
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataStoreFactorySpi#createNewDataStore(java.util.Map)
     */
    @Override
    public DataStore createNewDataStore(Map<String, Serializable> params) throws IOException
    {
        throw new UnsupportedOperationException();
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccessFactory#canProcess(java.util.Map)
     */
    @Override
    public boolean canProcess(Map<String, Serializable> params)
    {
        if (params == null)
            throw new IllegalArgumentException("params should not be null.");

        //check for the caching parameter
        CachingType cachingType = getCachingType(params);
        if (cachingType == null)
            return false;
        else
        {
            return _target.canProcess(params);
        }
        
    }
    
    private CachingType getCachingType(Map<String, Serializable> params)
    {
        Object paramValue = params.get(CACHE_TYPE.key);
        CachingType cachingType;
        if (paramValue instanceof String)
            cachingType = CachingType.getCachingType((String)paramValue);
        else if (paramValue instanceof CachingType) 
            cachingType = (CachingType)paramValue;
        else
            cachingType = null;
            
        return cachingType;
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccessFactory#getDescription()
     */
    @Override
    public String getDescription()
    {
        return _target.getDescription() + CACHED_SUFIX;
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccessFactory#getDisplayName()
     */
    @Override
    public String getDisplayName()
    {
        return _target.getDisplayName() + CACHED_SUFIX;
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccessFactory#getParametersInfo()
     */
    @Override
    public Param[] getParametersInfo()
    {
        Param[] targetParams = _target.getParametersInfo();
        Param[] params = new Param[targetParams.length + 1];
        System.arraycopy(targetParams, 0, params, 0, targetParams.length);
        params[params.length-1] = CACHE_TYPE;
        return params;
    }

    /* (non-Javadoc)
     * @see org.geotools.data.DataAccessFactory#isAvailable()
     */
    @Override
    public boolean isAvailable()
    {
        return _target.isAvailable();
    }

    /* (non-Javadoc)
     * @see org.geotools.factory.Factory#getImplementationHints()
     */
    @Override
    public Map<Key, ?> getImplementationHints()
    {
        return _target.getImplementationHints();
    }

}
