/*
 * $Id$
 */
package csbase.client;

import csbase.exception.CSBaseException;
import csbase.logic.*;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ServerEntryPoint;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.core.lng.LNG;

import java.rmi.RemoteException;
import java.security.PublicKey;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;

/**
 * Extenso do {@link ServerManager} para implementao de login usando usurio
 * e senha e login por referncia. tambm o ponto de entrada do cliente para
 * obter a referncia para um determinado servio. Os campos estticos do
 * {@link ClientRemoteLocator} so populados com as referncias dos servios do
 * servidor default.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ClientServerManager extends ServerManager {

  /**
   * Constri uma instncia desta classe que compartilhar os dados dos
   * servidores monitorados atravs de {@link ServerManagerData}
   */
  public ClientServerManager() {
    this(true);
  }

  /**
   * Constri uma instncia desta classe
   * 
   * @param sharedServerData true indica que os dados dos servidores monitorados
   *        sero compartilhados entre todas as instncias dessa classe, false
   *        os dados monitorados sero independentes.
   */
  public ClientServerManager(boolean sharedServerData) {
    super(sharedServerData);
  }

  /**
   * Obtm o valor mximo da janela de backoff a partir de um valor que vem do
   * client.properties
   * 
   * @return o valor mximo da janela de backoff a partir de um valor que vem do
   *         client.properties
   */
  public static int getWindowSize() {
    try {
      Configuration conf =
        ConfigurationManager.getInstance().getConfiguration(
          csbase.client.Client.class);
      return Integer
        .parseInt(conf.getOptionalProperty("monitor.server.window"));
    }
    catch (Throwable e) {
      System.out
        .println("O valor da propriedade monitor.server.window no foi "
          + "encontrado. Usando o valor default: "
          + MonitoredServer.RELOAD_TIME);
    }

    return MonitoredServer.RELOAD_TIME;
  }

  /**
   * Implementao de monitoramento para um servidor monitorado com login por
   * referncia
   * 
   * @author Tecgraf/PUC-Rio
   */
  private final class ReferedMonitoredServer extends MonitoredServer {

    /**
     * Indica se os attributos da sesso do servidor de referncia deve ser
     * copiado
     */
    private boolean copyServerSessionAttrs;

    /**
     * Default locale a ser usado caso no seja definido
     */
    private Locale locale = Locale.getDefault();

    /**
     * Construtor
     * 
     * @param serverURI URI do servidor
     * @param copyServerSessionAttrs indicativo de cpia de atributos.
     */
    protected ReferedMonitoredServer(ServerURI serverURI,
      boolean copyServerSessionAttrs) {
      super(serverURI, Client.getInstance().getClientRemoteLocatorClass(),
        getWindowSize());
      this.copyServerSessionAttrs = copyServerSessionAttrs;
    }

    /**
     * Construtor
     * 
     * @param serverURI URI do servidor
     * @param copyServerSessionAttrs indicativo de cpia de atributos.
     * @param serviceNames nomes dos servios
     */
    protected ReferedMonitoredServer(ServerURI serverURI,
      boolean copyServerSessionAttrs, Set<String> serviceNames) {
      super(serverURI, serviceNames, getWindowSize());
      this.copyServerSessionAttrs = copyServerSessionAttrs;
    }

    /**
     * Construtor
     * 
     * @param serverURI URI do servidor
     * @param copyServerSessionAttrs indicativo de cpia de atributos.
     * @param ignoreVersion indicativo de ignorar verso
     */
    protected ReferedMonitoredServer(ServerURI serverURI,
      boolean copyServerSessionAttrs, boolean ignoreVersion) {
      super(serverURI, Client.getInstance().getClientRemoteLocatorClass(),
        ignoreVersion, getWindowSize());
      this.copyServerSessionAttrs = copyServerSessionAttrs;
    }

    /**
     * Construtor
     * 
     * @param serverURI URI do servidor
     * @param copyServerSessionAttrs indicativo de cpia de atributos.
     * @param serviceNames nomes dos servios.
     * @param ignoreVersion indicativo de ignorar verso
     */
    protected ReferedMonitoredServer(ServerURI serverURI,
      boolean copyServerSessionAttrs, Set<String> serviceNames,
      boolean ignoreVersion) {
      super(serverURI, serviceNames, ignoreVersion, getWindowSize());
      this.copyServerSessionAttrs = copyServerSessionAttrs;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Session performLogin() throws CSBaseException, RemoteException {
      return doLoginWithReference(this.getURI(), this.delegatedLogin,
        this.locale, this.timeZone, this.copyServerSessionAttrs);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void postLogin() throws CSBaseException {
      /** nada a fazer */
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected String lng(String key) {
      return LNG.get(key);
    }

    /**
     * Ajuste de locale
     * 
     * @param locale o novo locale
     */
    protected void setLocale(Locale locale) {
      if (locale != null) {
        this.locale = locale;
      }
    }

  }

  /**
   * Implementao de monitoramento para um servidor monitorado com login usando
   * usurio e senha
   * 
   * @author Tecgraf/PUC-Rio
   */
  private final class UserAndPasswordMonitoredServer extends MonitoredServer {

    /**
     * Informaes de login
     */
    LoginInfo loginInfo = null;

    /**
     * Construtor
     * 
     * @param serverURI URI do servidor.
     * @param loginInfo informao de login.
     */
    protected UserAndPasswordMonitoredServer(ServerURI serverURI,
      LoginInfo loginInfo) {
      super(serverURI, Client.getInstance().getClientRemoteLocatorClass(),
        getWindowSize());
      this.loginInfo = loginInfo;
    }

    /**
     * Construtor
     * 
     * @param serverURI URI do servidor.
     * @param loginInfo informao de login.
     * @param serviceNames nomes dos servios.
     */
    protected UserAndPasswordMonitoredServer(ServerURI serverURI,
      LoginInfo loginInfo, Set<String> serviceNames) {
      super(serverURI, serviceNames, getWindowSize());
      this.loginInfo = loginInfo;
    }

    /**
     * Construtor
     * 
     * @param serverURI URI do servidor.
     * @param loginInfo informao de login.
     * @param ignoreVersion indicativo para ignorar verso.
     */
    protected UserAndPasswordMonitoredServer(ServerURI serverURI,
      LoginInfo loginInfo, boolean ignoreVersion) {
      super(serverURI, Client.getInstance().getClientRemoteLocatorClass(),
        ignoreVersion, getWindowSize());
      this.loginInfo = loginInfo;
    }

    /**
     * Construtor
     * 
     * @param serverURI URI do servidor.
     * @param loginInfo informao de login.
     * @param serviceNames nomes dos servios.
     * @param ignoreVersion indicativo para ignorar verso.
     */
    protected UserAndPasswordMonitoredServer(ServerURI serverURI,
      LoginInfo loginInfo, Set<String> serviceNames, boolean ignoreVersion) {
      super(serverURI, serviceNames, ignoreVersion, getWindowSize());
      this.loginInfo = loginInfo;
    }

    /**
     * {@inheritDoc}
     * 
     * Executa o login usando usurio e senha.
     */
    @Override
    public Session performLogin() throws CSBaseException, RemoteException {
      return doLoginWithUserAndPassword(getURI(), loginInfo.getLoginName(),
        loginInfo.getPassword(), delegatedLogin, loginInfo.getLocale(),
        this.timeZone);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void postLogin() throws CSBaseException {
      /** Nada a fazer */
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected String lng(String key) {
      return LNG.get(key);
    }

    /**
     * Ajuste de loginInfo
     * 
     * @param loginInfo o novo loginInfo.
     */
    protected void setLoginInfo(LoginInfo loginInfo) {
      this.loginInfo = loginInfo;
    }
  }

  /**
   * A instncia nica dessa classe
   */
  private static ClientServerManager instance = null;

  /**
   * Conjunto dos nomes do servios, alternativa para no depender do
   * {@link ClientRemoteLocator}
   */
  Set<String> serviceNames;

  /**
   * Executa o login indicando como referncia para validao do usurio o
   * servidor default.
   * 
   * @param serverURI A URI do servidor onde se deseja executar o login
   * @param delegatedLogin O login usado para delegao, null se no for o caso
   * @param locale o Locale a ser usado no login
   * @param timeZone O time zone a ser usado no login
   * @param copyServerSessionAttrs indica se o login por referncia deve copiar
   *        os attributos de sesso do servidor de referncia
   * @return A instncia da sesso no servidor ou null caso o usurio no seja
   *         validado no servidor de referncia ou ocorra algum erro.
   * @throws CSBaseException Em caso de erro durante o login
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  private Session doLoginWithReference(ServerURI serverURI,
    String delegatedLogin, Locale locale, TimeZone timeZone,
    boolean copyServerSessionAttrs) throws CSBaseException, RemoteException {
    ServerURI defaultServerURI = this.getDefaultURI();
    ServerEntryPoint server = this.getServer(serverURI);
    Session session = this.getSession(defaultServerURI);

    return server.login(defaultServerURI, (Map) serverURI.getParamsMap(),
      copyServerSessionAttrs, session.getKey(), session.getUser().getLogin(),
      delegatedLogin, locale, timeZone);
  }

  /**
   * Executa o login requisitando atravs de um dilogo o usurio e senha.
   * 
   * @param serverURI O servidor onde se deseja executar o login
   * @param loginName O nome do usurio
   * @param password A senha
   * @param delegatedLogin O usurio delegado
   * @param locale o Locale do usurio
   * @param timeZone o time zone para usar no login
   * 
   * @return A instncia da sesso ou null se o login ou senha forem invlidos.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  private Session doLoginWithUserAndPassword(ServerURI serverURI,
    String loginName, String password, String delegatedLogin, Locale locale,
    TimeZone timeZone) throws RemoteException {
    ServerEntryPoint server = this.getServer(serverURI);
    TimeZone tz = timeZone;
    if (tz == null) {
      tz = TimeZone.getDefault();
    }
    PublicKey publicKey = server.getPublicKey();
    EncryptedPassword encryptedPassword =
      LoginPasswordCipher.encrypt(password, publicKey);
    return server.login(loginName, encryptedPassword, locale, tz,
      delegatedLogin, (Map) serverURI.getParamsMap());
  }

  /**
   * Executa um login no servidor remoto identificado pela URI. O servidor ser
   * colocado na monitorao se no estiver. O login ser feito usando o
   * servidor default como referncia.
   * 
   * 
   * @param serverURI A URI do servidor
   * @param delegatedLogin O login do usurio delegado
   * @param copyServerSessionAttrs indica se o login por referncia deve copiar
   *        os atributos de sesso do servidor de referncia
   * @param ignoreVersion true indica que a verificao de verso entre cliente
   *        servidor ser ignorada
   * @throws CSBaseException Se houver falha tentando durante o login.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   * @return true se login de sucesso ou false se login/senha invlido
   */
  public final boolean loginByReference(ServerURI serverURI,
    String delegatedLogin, boolean copyServerSessionAttrs, boolean ignoreVersion)
    throws CSBaseException, RemoteException {
    return this.loginByReference(serverURI, delegatedLogin, null,
      copyServerSessionAttrs, ignoreVersion);
  }

  /**
   * Executa um login no servidor remoto identificado pela URI. O servidor ser
   * colocado na monitorao se no estiver. O login ser feito usando o
   * servidor default como referncia.
   * 
   * 
   * @param serverURI A URI do servidor
   * @param delegatedLogin O login do usurio delegado
   * @param timeZone O time zone para ser usado no login
   * @param copyServerSessionAttrs indica se o login por referncia deve copiar
   *        os atributos de sesso do servidor de referncia
   * @param ignoreVersion true indica que a verificao de verso entre cliente
   *        servidor ser ignorada
   * @throws CSBaseException Se houver falha tentando durante o login.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   * @return true se login de sucesso ou false se login/senha invlido
   */
  public final boolean loginByReference(ServerURI serverURI,
    String delegatedLogin, TimeZone timeZone, boolean copyServerSessionAttrs,
    boolean ignoreVersion) throws CSBaseException, RemoteException {
    return this.loginByReference(serverURI, delegatedLogin, null, timeZone,
      copyServerSessionAttrs, ignoreVersion);
  }

  /**
   * Executa um login no servidor remoto identificado pela URI. O servidor ser
   * colocado na monitorao se no estiver. O login ser feito usando o
   * servidor default como referncia.
   * 
   * 
   * @param serverURI A URI do servidor
   * @param delegatedLogin O login do usurio delegado
   * @param locale O locale para ser usado no login
   * @param timeZone O time zone para ser usado no login
   * @param copyServerSessionAttrs indica se o login por referncia deve copiar
   *        os atributos de sesso do servidor de referncia
   * @param ignoreVersion true indica que a verificao de verso entre cliente
   *        servidor ser ignorada
   * @throws CSBaseException Se houver falha tentando durante o login.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   * @return true se login de sucesso ou false se login/senha invlido
   */
  public final boolean loginByReference(ServerURI serverURI,
    String delegatedLogin, Locale locale, TimeZone timeZone,
    boolean copyServerSessionAttrs, boolean ignoreVersion)
    throws CSBaseException, RemoteException {

    ReferedMonitoredServer monitoredServer;
    if (this.serviceNames == null) {
      monitoredServer =
        new ReferedMonitoredServer(serverURI, copyServerSessionAttrs);
    }
    else {
      monitoredServer =
        new ReferedMonitoredServer(serverURI, copyServerSessionAttrs,
          serviceNames);
    }
    monitoredServer.setDelegatedLogin(delegatedLogin);
    monitoredServer.setTimeZone(timeZone);
    monitoredServer.setLocale(locale);
    this.addServer(monitoredServer);

    return this.doLogin(serverURI);
  }

  /**
   * Executa um login no servidor remoto identificado pela URI. O servidor ser
   * colocado na monitorao se no estiver. O login ser feito usando o
   * servidor default como referncia.
   * 
   * 
   * @param serverURI A URI do servidor
   * @param copyServerSessionAttrs indica se o login por referncia deve copiar
   *        os atributos de sesso do servidor de referncia
   * @throws CSBaseException Se houver falha tentando durante o login.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   * @return true se login de sucesso ou false se login/senha invlido
   */
  public final boolean loginByReference(ServerURI serverURI,
    boolean copyServerSessionAttrs) throws CSBaseException, RemoteException {
    return this
      .loginByReference(serverURI, null, copyServerSessionAttrs, false);
  }

  /**
   * Executa um login no servidor remoto identificado pela URI. O servidor ser
   * colocado na monitorao se no estiver. O login ser feito usando o
   * servidor default como referncia.
   * 
   * 
   * @param serverURI A URI do servidor
   * @param delegatedLogin O login do usurio delegado
   * @param copyServerSessionAttrs indica se o login por referncia deve copiar
   *        os atributos de sesso do servidor de referncia
   * @throws CSBaseException Se houver falha tentando durante o login.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   * @return true se login de sucesso ou false se login/senha invlido
   */
  public final boolean loginByReference(ServerURI serverURI,
    String delegatedLogin, boolean copyServerSessionAttrs)
    throws CSBaseException, RemoteException {
    return this.loginByReference(serverURI, delegatedLogin,
      copyServerSessionAttrs, false);
  }

  /**
   * Executa um login no servidor remoto identificado pela URI. O servidor ser
   * colocado na monitorao se no estiver.
   * 
   * @param serverURI A URI do servidor
   * @param loginInfo Os dados para login
   * @param ignoreVersion true indica que a verificao de verso entre cliente
   *        servidor ser ignorada
   * @return true se login de sucesso ou false se login/senha invlido
   * @throws CSBaseException Se houver falha tentando durante o login.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  public final boolean loginWithUserPassword(ServerURI serverURI,
    LoginInfo loginInfo, boolean ignoreVersion) throws CSBaseException,
    RemoteException {
    return this
      .loginWithUserPassword(serverURI, loginInfo, null, ignoreVersion);
  }

  /**
   * Executa um login no servidor remoto identificado pela URI. O servidor ser
   * colocado na monitorao se no estiver.
   * 
   * @param serverURI A URI do servidor
   * @param loginInfo Os dados para login
   * @param delegatedLogin O login do usurio delegado
   * @param ignoreVersion true indica que a verificao de verso entre cliente
   *        servidor ser ignorada
   * @return true se login de sucesso ou false se login/senha invlido
   * @throws CSBaseException Se houver falha tentando durante o login.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  public final boolean loginWithUserPassword(ServerURI serverURI,
    LoginInfo loginInfo, String delegatedLogin, boolean ignoreVersion)
    throws CSBaseException, RemoteException {
    return this.loginWithUserPassword(serverURI, loginInfo, null,
      delegatedLogin, ignoreVersion);
  }

  /**
   * Executa um login no servidor remoto identificado pela URI. O servidor ser
   * colocado na monitorao se no estiver.
   * 
   * @param serverURI A URI do servidor
   * @param loginInfo Os dados para login incluindo locale
   * @param timeZone O time zone para ser usado no login
   * @param delegatedLogin O login do usurio delegado
   * @param ignoreVersion true indica que a verificao de verso entre cliente
   *        servidor ser ignorada
   * @return true se login de sucesso ou false se login/senha invlido
   * @throws CSBaseException Se houver falha tentando durante o login.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  public final boolean loginWithUserPassword(ServerURI serverURI,
    LoginInfo loginInfo, TimeZone timeZone, String delegatedLogin,
    boolean ignoreVersion) throws CSBaseException, RemoteException {

    UserAndPasswordMonitoredServer monitoredServer;
    if (this.serviceNames == null) {
      monitoredServer =
        new UserAndPasswordMonitoredServer(serverURI, loginInfo, ignoreVersion);
    }
    else {
      monitoredServer =
        new UserAndPasswordMonitoredServer(serverURI, loginInfo, serviceNames,
          ignoreVersion);
    }

    monitoredServer.setDelegatedLogin(delegatedLogin);
    monitoredServer.setTimeZone(timeZone);
    this.addServer(monitoredServer);
    return this.doLogin(serverURI);
  }

  /**
   * 
   * Executa um login no servidor remoto identificado pela URI. O servidor ser
   * colocado na monitorao se no estiver.
   * 
   * @param serverURI A URI do servidor
   * @param loginInfo Os dados para login
   * @return true se login de sucesso ou false se login/senha invlido
   * @throws CSBaseException Se houver falha tentando durante o login.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  public final boolean loginWithUserPassword(ServerURI serverURI,
    LoginInfo loginInfo) throws CSBaseException, RemoteException {
    return this.loginWithUserPassword(serverURI, loginInfo, false);
  }

  /**
   * @return Retorna a instncia nica desta classe que compartilha os dados dos
   *         servidores monitorados
   */
  public static ClientServerManager getInstance() {
    if (instance == null) {
      instance = new ClientServerManager();
    }
    return instance;
  }

  /**
   * Define os nomes dos servios a serem recuperados do servidor aps login bem
   * sucedido. Alternativa para o uso do {@link ClientRemoteLocator}
   * 
   * @param serviceNames Conjunto com os nomes do servios a serem recuperados
   *        em um login bem sucedido
   */
  public final void setServiceNames(Set<String> serviceNames) {
    this.serviceNames = serviceNames;
  }

}
