/*
 * $Id: ShowHistoryAction.java 150777 2014-03-19 14:16:56Z oikawa $
 */
package csbase.client.algorithms;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SortOrder;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableModel;
import javax.swing.text.JTextComponent;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.kernel.ClientException;
import csbase.client.util.DateTableCellRenderer;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.algorithms.HistoryProvider;
import csbase.logic.algorithms.HistoryRecord;

/**
 * Ao de visualizao do histrico de um objeto. Exibe um dilogo com uma
 * tabela em que cada linha corresponde a um registro de histrico.
 * 
 * @author Tecgraf/PUC-Rio.
 */
public final class ShowHistoryAction extends AbstractAction {
  /**
   * Dilogo para exibio de histrico
   */
  private DesktopComponentDialog dialog;
  /** Janela pai */
  private Window ownerWindow;
  /**
   * Objeto provedor do histrico a ser exibido
   */
  private HistoryProvider historyProvider;
  // #TODO Verificar se a tabela precisa realmente ser global
  private SortableTable table;

  /**
   * Cria a ao com a janela-pai especificada, definindo sua descrio e seu
   * cone.
   * 
   * @param ownerWindow janela-pai.
   */
  public ShowHistoryAction(Window ownerWindow) {
    this(null, ownerWindow);
  }

  /**
   * Cria a ao com o provedor de histrico e a janela-pai especificada,
   * definindo sua descrio e seu cone.
   * 
   * @param provider objeto provedor do histrico.
   * @param ownerWindow janela-pai.
   */
  public ShowHistoryAction(HistoryProvider provider, Window ownerWindow) {
    this(provider);
    this.ownerWindow = ownerWindow;
  }

  /**
   * Cria a ao, definindo sua descrio e seu cone.
   * 
   * @param provider provider
   */
  public ShowHistoryAction(HistoryProvider provider) {
    super(LNG.get(ShowHistoryAction.class.getName() + ".name"),
      ApplicationImages.ICON_HISTORY_24);
    this.historyProvider = provider;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    try {
      showDialog();
    }
    catch (ClientException ex) {
      StandardErrorDialogs.showErrorDialog(this.ownerWindow, ex.getMessage());
    }
  }

  /**
   * Exibe o dilogo de histrico.
   * 
   * @throws ClientException em caso de falha.
   */
  private final void showDialog() throws ClientException {
    String path =
      (this.historyProvider == null) ? "" : this.historyProvider.getName();
    String title =
      MessageFormat.format(
        LNG.get(ShowHistoryAction.class.getName() + ".title"),
        new Object[] { path });

    this.dialog = new DesktopComponentDialog(this.ownerWindow, title);
    this.dialog.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        closeDialog();
      }
    });
    this.dialog.getContentPane().add(makeContentPane());
    final int FRAME_WIDTH = 800;
    final int FRAME_HEIGHT =
      this.dialog.getContentPane().getPreferredSize().height;
    this.dialog.setSize(FRAME_WIDTH, FRAME_HEIGHT);
    final int FRAME_INTERNAL_MARGIN = 13;// Margem interna a ser descontada
    this.table.setPreferredScrollableViewportSize(new Dimension(FRAME_WIDTH
      - FRAME_INTERNAL_MARGIN,
      this.table.getPreferredScrollableViewportSize().height));
    if (this.ownerWindow != null) {
      this.dialog.center(this.ownerWindow);
    }
    this.dialog.setVisible(true);
  }

  /**
   * Cria o contedo do dilogo.
   * 
   * @return contedo do dilogo.
   * 
   * @throws ClientException em caso de falha.
   */
  private Container makeContentPane() throws ClientException {
    JComponent panel = new JPanel(new BorderLayout());
    panel.add(makeTopPanel(), BorderLayout.NORTH);
    panel.add(new JScrollPane(makeTable()), BorderLayout.CENTER);
    panel.add(makeButtonPanel(), BorderLayout.SOUTH);
    return panel;
  }

  /**
   * Constri o painel superior, contendo o caminho para o objeto cujo histrico
   * est sendo exibido.
   * 
   * @return painel superior.
   */
  private Container makeTopPanel() {
    JLabel pathLbl =
      new JLabel(LNG.get(ShowHistoryAction.class.getName() + ".pathLabel"));
    String path =
      (this.historyProvider == null) ? "" : this.historyProvider.getName();
    JTextComponent pathText = new JTextField(path);
    pathText.setBackground(this.dialog.getBackground());
    pathText.setEditable(false);
    JComponent[][] row = { { pathLbl, pathText } };
    return GUIUtils.createBasicGridPanel(row);
  }

  /**
   * Constri a tabela que exibe o histrico.
   * 
   * @return tabela que exibe o histrico.
   * @throws ClientException em caso de falha.
   */
  private JTable makeTable() throws ClientException {
    List rows =
      (this.historyProvider == null) ? new ArrayList() : this.historyProvider
        .getHistory();
    if (rows == null) {
      throw new ClientException(LNG.get(ShowHistoryAction.class.getName()
        + ".error"));
    }
    TableModel m = new HistoryTableModel(rows);
    // #TODO Verificar se a tabela precisa realmente ser global
    this.table = new SortableTable(m);
    this.table.sort(0, SortOrder.DESCENDING);
    this.table.setDefaultRenderer(Date.class, new DateTableCellRenderer());
    this.table.adjustColumnWidth();
    return this.table;
  }

  /**
   * Constri o painel de botes para a janela.
   * 
   * @return painel de botes para a janela.
   */
  private Container makeButtonPanel() {
    JPanel buttonPanel = new JPanel();
    JButton closeButton =
      new JButton(LNG.get(ShowHistoryAction.class.getName() + ".closeButton"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        closeDialog();
      }
    });
    buttonPanel.add(closeButton);
    return buttonPanel;
  }

  /**
   * Fecha o dilogo.
   */
  private final void closeDialog() {
    this.dialog.dispose();
  }

  /**
   * Obtm o/a provedor de histrico.
   * 
   * @return provedor de histrico.
   */
  public HistoryProvider getHistoryProvider() {
    return this.historyProvider;
  }

  /**
   * Substitui o provedor de histrico.
   * 
   * @param provider novo provedor de histrico.
   */
  public void setHistoryProvider(HistoryProvider provider) {
    this.historyProvider = provider;
  }

  /**
   * Modelo para a tabela
   */
  private class HistoryTableModel extends AbstractTableModel {
    /** Lista de <code>HistoryRecord</code> */
    private final List recordList;

    /** Cabealhos para as colunas */
    private final String[] columnNames = {
        LNG.get(ShowHistoryAction.class.getName() + ".dateColumn"),
        LNG.get(ShowHistoryAction.class.getName() + ".pathColumn"),
        LNG.get(ShowHistoryAction.class.getName() + ".loginColumn"),
        LNG.get(ShowHistoryAction.class.getName() + ".nameColumn"),
        LNG.get(ShowHistoryAction.class.getName() + ".descriptionColumn") };

    /** Classes das colunas */
    private final Class[] columnClasses = { Date.class, String.class,
        String.class, String.class, String.class };

    /**
     * Cria o modelo.
     * 
     * @param recordList Lista de <code>HistoryRecord</code>.
     */
    public HistoryTableModel(List recordList) {
      this.recordList = recordList;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getRowCount() {
      return this.recordList.size();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getColumnCount() {
      return this.columnNames.length;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
      HistoryRecord rec = (HistoryRecord) this.recordList.get(rowIndex);
      switch (columnIndex) {
        case HistoryRecord.DATE_COLUMN_INDEX:
          return rec.getDate();
        case HistoryRecord.PATH_COLUMN_INDEX:
          return FileUtils.joinPath(rec.getDirPath());
        case HistoryRecord.LOGIN_COLUMN_INDEX:
          return rec.getUserLogin();
        case HistoryRecord.NAME_COLUMN_INDEX:
          return rec.getUserName();
        case HistoryRecord.DESCRIPTION_COLUMN_INDEX:
          return rec.getDescription();
        default:
          return null;
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName(int columnIndex) {
      return this.columnNames[columnIndex];
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class getColumnClass(int columnIndex) {
      return this.columnClasses[columnIndex];
    }
  }
}
