/*
 * $Id$
 */
package csbase.client.algorithms.commands.cache.events;

import java.util.Collection;
import java.util.Collections;

import csbase.client.util.event.IEvent;
import csbase.logic.CommandInfo;

/**
 * @author Tecgraf / PUC-Rio
 * 
 * Evento indicando que a cache foi atualizada.
 * 
 * @see csbase.client.algorithms.commands.cache.CommandsCache
 */
public class CacheUpdatedEvent implements IEvent {

  /**
   * Tipo do evento.
   */
  public enum Type {
    /**
     * Exceo lanada pela ao de atualizao de comandos.
     */
    exception,
    /**
     * Cache atualizada com sucesso.
     */
    updated
  }

  /**
   * Tipo do evento gerado.
   */
  private Type type;
  /**
   * Exceo que pode ser gerada durante o processo de atualizao da cache.
   */
  private Exception exception;
  /**
   * A descrio de como a exceo foi gerada.
   */
  private String exceptionDescription;
  /**
   * Status da cache aps a atualizao.
   */
  private Collection<CommandInfo> commands;

  /**
   * Construtor utilizado para criar um evento indicando que uma exceo foi
   * lanada durante a atualizao dos comandos.
   * 
   * @param e a exceo que interrompeu a atualizaao da cache.
   * @param description a descrio de como a exceo foi gerada.
   */
  public CacheUpdatedEvent(Exception e, String description) {
    this.exception = e;    
    this.type = Type.exception;
    this.exceptionDescription = description;
    this.commands = null;
  }

  /**
   * Construtor.
   * 
   * @param commands todos os comandos presentes na cache aps sua atualizao.
   */
  public CacheUpdatedEvent(Collection<CommandInfo> commands) {
    this.commands = Collections.unmodifiableCollection(commands);
    this.type = Type.updated;
  }

  /**
   * Obtm a exceo que interrompeu a atualizaao da cache.
   * 
   * @return a exceo que interrompeu a atualizaao da cache.
   */
  public Exception getException() {
    return exception;
  }

  /**
   * Obtm a descrio de como a exceo foi gerada.
   * 
   * @return a descrio de como a exceo foi gerada.
   */
  public String getExceptionDescription() {
    return exceptionDescription;
  }    
  
  /**
   * Obtm todos os comandos presentes na cache aps sua atualizao.
   * 
   * @return todos os comandos presentes na cache aps sua atualizao.
   */
  public Collection<CommandInfo> getCommands() {
    return commands;
  }
  
  /**
   * Obtm o tipo do evento gerado.
   * 
   * @return o tipo do evento gerado.
   */
  public Type getType() {
    return type;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((commands == null) ? 0 : commands.hashCode());
    result = prime * result + ((exception == null) ? 0 : exception.getClass().hashCode());
    result =
      prime
        * result
        + ((exceptionDescription == null) ? 0 : exceptionDescription.hashCode());
    result = prime * result + ((type == null) ? 0 : type.hashCode());
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    final CacheUpdatedEvent other = (CacheUpdatedEvent) obj;
    if (commands == null) {
      if (other.commands != null)
        return false;
    }
    else if (!commands.equals(other.commands))
      return false;
    if (exception == null) {
      if (other.exception != null)
        return false;
    }
    else if (!exception.getClass().equals(other.exception.getClass()))
      return false;
    if (exceptionDescription == null) {
      if (other.exceptionDescription != null)
        return false;
    }
    else if (!exceptionDescription.equals(other.exceptionDescription))
      return false;
    if (type == null) {
      if (other.type != null)
        return false;
    }
    else if (!type.equals(other.type))
      return false;
    return true;
  }
  
  
}
