package csbase.client.applicationmanager.resourcehelpers;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;

import javax.swing.AbstractListModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Dilogo que permite que o usurio escolha a aplicao.
 * 
 * @author Tecgraf
 */
public class ApplicationSelectionDialog extends JDialog {

  /**
   * Lista com as aplicaes que exibem um determinado tipo de arquivo.
   */
  final private JList list = new JList();

  /**
   * Construtor padro.
   * 
   * @param owner janela me.
   */
  public ApplicationSelectionDialog(Window owner) {
    this(owner, ApplicationManager.getInstance().getAllApplicationRegistries());
  }

  /**
   * Construtor padro.
   * 
   * @param owner janela me.
   * @param regs registros de aplicaes que devem constar na lista.
   */
  public ApplicationSelectionDialog(Window owner, List<ApplicationRegistry> regs) {
    super(owner, "", ModalityType.DOCUMENT_MODAL);

    ApplicationManager manager = ApplicationManager.getInstance();
    final Locale locale = manager.getLocale();
    Collections.sort(regs, new Comparator<ApplicationRegistry>() {
      @Override
      public int compare(ApplicationRegistry o1, ApplicationRegistry o2) {
        String n1 = o1.getApplicationName(locale);
        String n2 = o2.getApplicationName(locale);
        return n1.compareTo(n2);
      }
    });
    initListAttributes(regs);

    setLayout(new BorderLayout());

    JButton okButton = new JButton(getClassString("ok.button"));
    JButton cancelButton = new JButton(getClassString("cancel.button"));
    GUIUtils.matchPreferredSizes(okButton, cancelButton);
    okButton.setEnabled(false);

    JPanel buttonsPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
    buttonsPanel.add(cancelButton);
    buttonsPanel.add(okButton);

    JScrollPane pane = new JScrollPane(list);
    BorderUtil.setLowBorder(pane);

    add(pane, BorderLayout.CENTER);
    add(buttonsPanel, BorderLayout.SOUTH);

    setLocationRelativeTo(getOwner());
    setUpButtons(okButton, cancelButton);
    setUpListeners(okButton);

    setMinimumSize(new Dimension(400, 300));
    pack();
  }

  /**
   * Inicializa atributos da lista de aplicaes.
   * 
   * @param regs registros que devem constar na lista.
   */
  private void initListAttributes(final List<ApplicationRegistry> regs) {
    list.setModel(new AbstractListModel() {
      @Override
      public int getSize() {
        return regs.size();
      }

      @Override
      public Object getElementAt(int index) {
        return regs.get(index);
      }
    });
    list.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    list.setLayoutOrientation(JList.VERTICAL);
    list.setCellRenderer(buildCellRenderer());
  }

  /**
   * Mtodo que constroi os botes do dilogo.
   * 
   * @param okButton boto de OK
   * @param cancelButton boto de cancelamento.
   */
  private void setUpButtons(JButton okButton, JButton cancelButton) {
    okButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        ApplicationSelectionDialog.this.setVisible(false);
      }
    });

    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        ApplicationSelectionDialog.this.setVisible(false);
        list.clearSelection();
      }
    });
  }

  /**
   * Adiciona ouvintes de lista.
   * 
   * @param okButton
   */
  private void setUpListeners(final JButton okButton) {
    list.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        okButton.setEnabled(list.getSelectedValue() != null);
      }
    });

    list.addMouseListener(new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2) {
          okButton.doClick();
        }
      }
    });
  }

  /**
   * Retorna lista com as aplicaes selecionadas.
   * 
   * @return lista das aplicaes.
   */
  public List<String> getSelectedApplications() {
    if (list.getSelectedIndices() == null) {
      return null;
    }
    Object[] indices = list.getSelectedValues();
    List<String> apps = new ArrayList<String>(indices.length);
    for (Object obj : indices) {
      apps.add(obj.toString());
    }
    return apps;
  }

  /**
   * Constri o renderizador da lista que exibe todas as aplicaes.
   * 
   * @return renderizador.
   */
  private ListCellRenderer buildCellRenderer() {
    return new DefaultListCellRenderer() {
      @Override
      public Component getListCellRendererComponent(JList jlist, Object value,
        int index, boolean isSelected, boolean cellHasFocus) {
        super.getListCellRendererComponent(jlist, value, index, isSelected,
          cellHasFocus);

        ApplicationRegistry reg = (ApplicationRegistry) value;
        String id = reg.getId();

        ApplicationManager manager = ApplicationManager.getInstance();
        Icon icon = manager.getApplicationIcon(id);
        String name = manager.getApplicationName(reg);

        setIcon(icon);
        setText(name);
        setHorizontalAlignment(SwingConstants.LEADING);

        return this;
      }
    };
  }

  /**
   * Mtodo auxiliar que obtem a string internacionalizada.
   * 
   * @param key - chave da string.
   * @return string internacionalizada.
   */
  private String getClassString(String key) {
    return LNG.get(getClass().getSimpleName() + "." + key);
  }
}
