package csbase.client.applicationmanager.resourcehelpers;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.List;

import javax.swing.DefaultListCellRenderer;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Dilogo que permite que o usurio escolha a aplicao preferencial para um
 * determinado tipo de arquivo.
 * 
 * @author Tecgraf
 */
public class PreferredAppDialog extends JDialog {

  /** Lista com as aplicaes que exibem um determinado tipo de arquivo. */
  private JList appList;

  /** Check box que define se deve salvar a aplicao como preferida. */
  private JCheckBox saveCheckBox;

  /** Flag que define se o usurio pode salvar a aplicao como preferida. */
  private boolean saveOption;

  /** Boto de confirmar. */
  private JButton ok;

  /** Boto de cancelar. */
  private JButton cancel;

  /**
   * Construtor padro.
   * 
   * @param owner - janela pai.
   * @param apps - lista com os identificadores das aplicaes.
   * @param fileType - tipo do arquivo.
   * @param saveOption - oferece a opo de salvar a aplicao como preferida.
   */
  public PreferredAppDialog(Window owner, List<String> apps, String fileType,
    boolean saveOption) {
    super(owner, "", ModalityType.DOCUMENT_MODAL);
    setTitle(getString("title"));
    setLayout(new GridBagLayout());

    this.appList = new JList(apps.toArray());
    appList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    appList.setLayoutOrientation(JList.VERTICAL);
    appList.setCellRenderer(buildCellRenderer());

    this.saveCheckBox = new JCheckBox(getString("save.as.preferred"));
    this.saveOption = saveOption;

    this.ok = new JButton("Ok");
    this.ok.setEnabled(false);
    this.cancel = new JButton(LNG.get("UTIL_CANCEL"));
    GUIUtils.matchPreferredSizes(new JComponent[] { this.ok, this.cancel });

    buildInterface();

    setLocationRelativeTo(getOwner());
    setUpButtons(fileType);
    setUpListeners();

    setMinimumSize(new Dimension(300, 300));
    pack();
  }

  /** Constroi a interface do dilogo. */
  private void buildInterface() {
    JPanel panel = new JPanel(new GridBagLayout());
    panel.add(new JScrollPane(appList), new GBC(0, 0).west().both());
    if (saveOption) {
      panel.add(saveCheckBox, new GBC(0, 1).west());
    }

    JPanel buttonsPanel = new JPanel(new GridBagLayout());
    buttonsPanel.add(new JLabel(), new GBC(0, 0).horizontal());
    buttonsPanel.add(cancel, new GBC(1, 0).west().bottom(10).left(10).right(5));
    buttonsPanel.add(ok, new GBC(2, 0).west().bottom(10).left(5).right(10));

    add(panel, new GBC(0, 0).both().insets(10));
    add(buttonsPanel, new GBC(0, 1).horizontal());
  }

  /**
   * Mtodo que constroi os botes do dilogo.
   * 
   * @param fileType - tipo do arquivo a ser aberto.
   */
  private void setUpButtons(final String fileType) {
    ok.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        PreferredAppDialog.this.setVisible(false);
      }
    });

    cancel.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        PreferredAppDialog.this.setVisible(false);
        appList.clearSelection();
      }
    });

  }

  /** Adiciona ouvintes de lista. */
  private void setUpListeners() {
    appList.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        ok.setEnabled(appList.getSelectedValue() != null);
      }
    });

    appList.addMouseListener(new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2) {
          ok.doClick();
        }
      }
    });
  }

  /**
   * Retorna a aplicao selecionada.
   * 
   * @return aplicao selecionada.
   */
  public String getSelectedApp() {
    if (appList != null && appList.getSelectedValue() != null) {
      return appList.getSelectedValue().toString();
    }

    return null;
  }

  /**
   * Retorna flag que indica se  pra salvar a aplicao selecionada ou no
   * 
   * @return true se for para salvar a aplicao selecionada, false caso
   *         contrrio.
   */
  public boolean toSave() {
    return saveOption && saveCheckBox.isSelected();
  }

  /**
   * Constri o renderizador da lista que exibe todas as aplicaes.
   * 
   * @return renderizador.
   */
  private ListCellRenderer buildCellRenderer() {
    return new DefaultListCellRenderer() {
      @Override
      public Component getListCellRendererComponent(JList list, Object value,
        int index, boolean isSelected, boolean cellHasFocus) {
        super.getListCellRendererComponent(list, value, index, isSelected,
          cellHasFocus);

        String appId = value.toString();

        ApplicationManager am = ApplicationManager.getInstance();

        Icon icon = am.getApplicationIcon(appId);
        ApplicationRegistry appReg = am.getApplicationRegistry(appId);
        String name = am.getApplicationName(appReg);

        setIcon(icon);
        setText(name);
        setHorizontalAlignment(SwingConstants.LEADING);

        return this;
      }
    };
  }

  /**
   * Mtodo auxiliar que obtem a string internacionalizada.
   * 
   * @param key - chave da string.
   * @return string internacionalizada.
   */
  private String getString(String key) {
    return LNG.get(getClass().getSimpleName() + "." + key);
  }
}
