/*
 * $Id:$
 */
package csbase.client.applications;

import javax.swing.JFrame;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.StatusBar;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.desktop.CommandListener;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommandEndNotification;
import csbase.logic.CommandErrorNotification;
import csbase.logic.CommandFailedNotification;
import csbase.logic.CommandKilledNotification;
import csbase.logic.CommandLostNotification;
import csbase.logic.CommandNotification;
import csbase.logic.CommandSuccessNotification;
import csbase.logic.CommandWithNoExitCodeNotification;
import csbase.logic.CommonClientProject;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Esta classe implementa o adaptador para o caso da notificao pelo SGA do
 * trmino de um comando que foi submetido para execuo a partir de um dado de
 * uma aplicao.
 * 
 * O trmino da execuo do comando no necessariamente significa sucesso, e
 * esta classe lida com todos os possveis casos (interrupo, falha ou "comando
 * no recuperado"). Nessa implementao, nenhuma ao  realizada em relao a
 * abertura do arquivo de sada resultante da execuo do comando submetido pela
 * aplicao. Veja SGAServicee#commandLost(String, String).
 */
public class ApplicationCommandAdapter implements CommandListener {

  /**
   * Identificador do comando submetido
   */
  private Object commandId;

  /**
   * Indica se o trmino de comando deve ser tratado adaptador
   */
  private boolean handleFinishedCommand;

  /**
   * Indica se o listener deve ser removido aps receber a notificao
   */
  private boolean remove;

  /**
   * Caminho do arquivo de dados da aplicao relacionado ao comando
   */
  private String[] dataFilePath;

  /** Aplicao associada a execuo do comando */
  private ApplicationProject app;

  /**
   * Nome que identifica o que est sendo executado, e pode ser especifico de
   * cada projeto, como por ex: algoritmo, fluxo de algoritmos, simulao, entre
   * outros
   */
  private String executionName;

  /**
   * Constri um listener de comandos bsico, que no abre nenhum tipo de
   * arquivo de sada automaticamente.
   * 
   * @param commandId identificador do comando executado
   * @param dataFilePath caminho do arquivo de dados da aplicao
   * @param executionName nome da execuo, por exemplo, algoritmo ou simulao
   * @param application aplicao associada a execuo do comando
   */
  public ApplicationCommandAdapter(Object commandId, String[] dataFilePath,
    String executionName, ApplicationProject application) {
    this.commandId = commandId;
    this.dataFilePath = dataFilePath;
    this.app = application;
    this.remove = false;
    this.executionName = executionName;
    this.handleFinishedCommand = false;
  }

  /**
   * Obtm o tipo da aplicao que pode ser usada para abrir o dado de sada.
   * 
   * @return o tipo da aplicao que pode ser usada para abrir o dado de sada,
   *         ou null, caso nenhuma aplicao esteja associada ao dado de saida
   */
  public ApplicationProject getApplication() {
    return app;
  }

  /**
   * Obtm o identificador do comando executado.
   * 
   * @return o identificador do comando
   */
  public Object getCommandId() {
    return commandId;
  }

  /**
   * Obtm o caminho do arquivo de dados da aplicao que submeteu o comando
   * para execuo.
   * 
   * @return o caminho do arquivo de dados da aplicao
   */
  public String[] getDataFilePath() {
    return dataFilePath;
  }

  /**
   * Estabelece se o trmino de comando deve ser tratado ou no pelo adaptador.
   * 
   * O default  no tratar o trmino de comando. Se for estabelecido o
   * tratamento do trmino de comando, uma janela popup  exibida com uma
   * mensagem para o usurio, contendo as informaes do comando. A janela popup
   *  aberta com base no desktop do projeto, e pode acabar aparecendo mltiplas
   * janelas para o usurio, no caso do trmino de comando ser tratado tambm
   * pela aplicao que submeteu o comando para execuo.
   * 
   * @param state se true, estabelece que o trmino de comando deve ser tratado
   *        pelo adaptador, caso contrrio, o usurio receber somente uma
   *        notificao de trmino de comando no painel de notificaes
   */
  public void handleFinishedCommand(boolean state) {
    this.handleFinishedCommand = state;
  }

  /**
   * Obtm o nome do arquivo, a partir de um caminho completo do arquivo.
   * 
   * @param filePath Caminho completo do arquivo.
   * 
   * @return o nome do arquivo ou retorna null, no caso do path estar vazio
   */
  protected String getFileName(String[] filePath) {
    if (filePath.length == 0) {
      return null;
    }
    return filePath[filePath.length - 1];
  }

  /**
   * Se o listener estiver interessado, chama o mtodo para a notificao
   * especfica.
   * 
   * @param data informaes do comando
   */
  @Override
  public void notifyCommand(CommandNotification data) {
    if (!commandId.equals(data.getCommandId())) {
      return;
    }
    if (data instanceof CommandEndNotification) {
      notifyCommandFinished((CommandEndNotification) data);
      if (ApplicationManager.getInstance().isApplicationRunning(app)) {
        notifyCommandFinished();
      }
    }
    else if (data instanceof CommandSuccessNotification) {
      notifyCommandSuccess((CommandSuccessNotification) data);
      if (ApplicationManager.getInstance().isApplicationRunning(app)) {
        notifyCommandFinished();
      }
    }
    else if (data instanceof CommandErrorNotification) {
      notifyCommandError((CommandErrorNotification) data);
      if (ApplicationManager.getInstance().isApplicationRunning(app)) {
        notifyCommandFinished();
      }
    }
    else if (data instanceof CommandWithNoExitCodeNotification) {
      notifyCommandWithNoExitCode((CommandWithNoExitCodeNotification) data);
      if (ApplicationManager.getInstance().isApplicationRunning(app)) {
        notifyCommandFinished();
      }
    }
    else if (data instanceof CommandFailedNotification) {
      notifyCommandFailed((CommandFailedNotification) data);
    }
    else if (data instanceof CommandKilledNotification) {
      notifyCommandKilled((CommandKilledNotification) data);
    }
    else if (data instanceof CommandLostNotification) {
      notifyCommandLost((CommandLostNotification) data);
    }
    remove = true;
  }

  /**
   * Implementao default para uma aplicao que  notificada do trmino do
   * comando que foi submetido para execuo. Esse mtodo deve ser redefinido
   * por toda aplicao que deseja realizar alguma tarefa especfica quando o
   * comando terminar, tal como recarregar o arquivo corrente.
   */
  public void notifyCommandFinished() {
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean shouldRemove() {
    return remove;
  }

  /**
   * Verifica se o arquivo de dados faz parte do projeto corrente. Caso no
   * seja, exibe uma mensagem.
   * 
   * @return retorna true, se o arquivo de dados cuja execuo foi disparada
   *         estiver aberto ou false, caso contrrio.
   */
  protected boolean isDataFileInCurrentProject() {
    DesktopFrame desktop = DesktopFrame.getInstance();
    ClientProjectFile file = app.stringArrayToFile(dataFilePath);
    CommonClientProject proj = desktop.getProject();
    if (proj == null || file == null
      || !file.getProjectId().equals(proj.getId())) {
      String title =
        String.format(LNG.get("ApplicationCommandAdapter.title.finished"),
          executionName, getFileName(dataFilePath));
      String msg =
        String.format(
          LNG.get("ApplicationCommandAdapter.info.finished.otherProject"),
          executionName, getFileName(dataFilePath), executionName);
      StandardDialogs.showInfoDialog(desktop.getDesktopFrame(), title, msg);
      return false;
    }
    return true;
  }

  /**
   * Aes associadas ao trmino de uma execuo (sem dados sobre como a
   * execuo terminou).
   * 
   * @param data Dados da notificao, enviados pelo SGA.
   */
  protected void notifyCommandFinished(CommandEndNotification data) {
    String title =
      String.format(LNG.get("ApplicationCommandAdapter.title.finished"),
        executionName, getFileName(dataFilePath));

    String msg = data.getCmdDesc() + " - " + data.getCommandId();

    notifyCommandExecuted(data, title, msg);
  }

  /**
   * Aes associadas ao trmino de uma execuo (com sucesso).
   * 
   * @param data Dados da notificao, enviados pelo SGA.
   */
  protected void notifyCommandSuccess(CommandSuccessNotification data) {
    String title =
      String.format(LNG.get("ApplicationCommandAdapter.title.success"),
        executionName, getFileName(dataFilePath));

    String msg =
      String.format(LNG.get("ApplicationCommandAdapter.info.success"),
        getFileName(dataFilePath));

    notifyCommandExecuted(data, title, msg);
  }

  /**
   * Aes associadas ao trmino de uma execuo (com erro).
   * 
   * @param data Dados da notificao, enviados pelo SGA.
   */
  protected void notifyCommandError(CommandErrorNotification data) {
    String title =
      String.format(LNG.get("ApplicationCommandAdapter.title.error"),
        executionName, getFileName(dataFilePath));
    String msg =
      String.format(LNG.get("ApplicationCommandAdapter.info.error"),
        executionName, getFileName(dataFilePath));

    notifyCommandExecuted(data, title, msg);
  }

  /**
   * Aes associadas ao trmino de uma execuo (sem cdigo de retorno).
   * 
   * @param data Dados da notificao, enviados pelo SGA.
   */
  protected void notifyCommandWithNoExitCode(
    CommandWithNoExitCodeNotification data) {
    String title =
      String.format(LNG.get("ApplicationCommandAdapter.title.no_code"),
        executionName, getFileName(dataFilePath));
    String msg =
      String.format(LNG.get("ApplicationCommandAdapter.info.no_code"),
        executionName, getFileName(dataFilePath));

    notifyCommandExecuted(data, title, msg);
  }

  /**
   * Notifica o fim de comandos que conseguiram executar (tendo gerado erros
   * durante a execuo ou no).
   * 
   * @param data Dados da notificao, enviados pelo SGA.
   * @param title Ttulo da notificao.
   * @param message Mensagem da notificao.
   */
  private void notifyCommandExecuted(CommandNotification data, String title,
    String message) {
    if (!isDataFileInCurrentProject()) {
      return;
    }
    if (handleFinishedCommand) {
      notifyCommandFinalizationStatus(title, message);
    }
  }

  /**
   * Apresenta notificao sobre o trmino do comando.
   * 
   * @param title Ttulo da caixa de notificao.
   * @param message Mensagem da caixa de notificao.
   */
  private void notifyCommandFinalizationStatus(String title, String message) {
    final ApplicationFrame applicationFrame = app.getApplicationFrame();
    final StatusBar statusBar = applicationFrame.getStatusBar();
    statusBar.clearStatus();

    final DesktopFrame desktop = DesktopFrame.getInstance();
    final JFrame frame = desktop.getDesktopFrame();

    StandardDialogs.showInfoDialog(frame, title, message);
  }

  /**
   * Aes associadas  interrupo de uma execuo. Remove o arquivo temporrio
   * e muda o estado do arquivo de logging.
   * 
   * @param data dados da notificao, enviados pelo SGA
   */
  protected void notifyCommandKilled(CommandKilledNotification data) {
    final ClientProjectFile dataFile = app.stringArrayToFile(dataFilePath);
    if (dataFile == null) {
      return;
    }

    final RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        final ProjectServiceInterface projectService =
          ClientRemoteLocator.projectService;
        final Object projectId = dataFile.getProjectId();
        final String[] path = dataFile.getPath();
        projectService.setUnderConstruction(projectId, path, false);
      }
    };

    final DesktopFrame dskFrame = DesktopFrame.getInstance();
    final JFrame frame = dskFrame.getDesktopFrame();
    task.execute(frame, app.getName(),
      getString("ApplicationCommandAdapter.info.log.failed"));
    String title =
      String.format(LNG.get("ApplicationCommandAdapter.title.killed"),
        executionName, getFileName(dataFilePath));
    String msg =
      String.format(LNG.get("ApplicationCommandAdapter.info.killed"),
        executionName, getFileName(dataFilePath));
    notifyCommandFinalizationStatus(title, msg);
  }

  /**
   * Consulta uma propriedade da aplicao.
   * 
   * @param key a chave da propriedade
   * 
   * @return o valor da propriedade
   */
  public final String getString(final String key) {
    return app.getString(key);
  }

  /**
   * Aes associadas a falhas em uma execuo.
   * 
   * @param data dados da notificao, enviados pelo SGA
   */
  protected void notifyCommandFailed(CommandFailedNotification data) {
    String title =
      String.format(LNG.get("ApplicationCommandAdapter.title.failed"),
        executionName, getFileName(dataFilePath));
    String msg =
      String.format(LNG.get("ApplicationCommandAdapter.info.failed"),
        executionName, getFileName(dataFilePath));
    notifyCommandFinalizationStatus(title, msg);
  }

  /**
   * Aes associadas ao caso em que o comando terminou mas o SGA no possua
   * referncia para ele. Esta situao acontece p.ex. quando o SGA  reiniciado
   * e percebe que o comando no est mais executando, mas no  capaz de
   * fornecer as informaes de tempo (tempo total, CPU e sistema).
   * 
   * @param data dados da notificao, enviados pelo SGA
   */
  protected void notifyCommandLost(CommandLostNotification data) {
    String title =
      String.format(LNG.get("ApplicationCommandAdapter.title.failed"),
        executionName, getFileName(dataFilePath));
    String msg =
      String.format(LNG.get("ApplicationCommandAdapter.info.failed"),
        executionName, getFileName(dataFilePath));
    notifyCommandFinalizationStatus(title, msg);
  }

}
