/*
 * $Id: ApplicationComponentDialog.java 146656 2013-11-07 17:13:39Z mjulia $
 */

package csbase.client.applications;

import java.awt.Dimension;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.KeyStroke;

/**
 * Modela um dialogo criado por uma aplicao.
 * 
 * @param <A> classe da aplicao.
 */
public class ApplicationComponentDialog<A extends Application> extends JDialog {

  /**
   * A aplicao que chamou o Dialog
   */
  private final A application;

  /**
   * Ajusta a tecla esc para fechar o dilogo.
   */
  private void addEscListener() {
    final AbstractAction cancelAction = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        ApplicationComponentDialog.this.setVisible(false);
      }
    };

    final int mode = JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT;
    final InputMap inputMap = rootPane.getInputMap(mode);
    final ActionMap actionMap = rootPane.getActionMap();
    final int esc = KeyEvent.VK_ESCAPE;
    final KeyStroke cancelStroke = KeyStroke.getKeyStroke(esc, 0);
    final String actionMapKey = cancelStroke.toString();
    inputMap.put(cancelStroke, actionMapKey);
    actionMap.put(actionMapKey, cancelAction);
  }

  /**
   * Retorna a aplicao que chamou o dilogo.
   * 
   * @return a aplicao.
   */
  final public A getApplication() {
    return application;
  }

  /**
   * Consulta a uma string de locale, com base com base na classe de um objeto.
   * 
   * @param key a chave de consulta
   * @return a string baseada no locale.
   */
  public final String getClassString(final String key) {
    return application.getClassString(this.getClass(), key);
  }

  /**
   * Consulta a uma string de locale.
   * 
   * @param key a chave de consulta
   * @return a string baseada no locale.
   */
  public final String getString(final String key) {
    return application.getString(key);
  }

  /**
   * Obtm a verso localizada de uma mensagem que contenha parmetros, com base
   * na classe de um objeto.
   * 
   * @param key chave para a mensagem que contm parmetros.
   * @param args argumentos para a mensagem.
   * 
   * @return mensagem preenchida pelos argumentos e localizada.
   */
  public final String getClassString(final String key, final Object[] args) {
    return application.getClassString(this.getClass(), key, args);
  }

  /**
   * Obtm a verso localizada de uma mensagem que contenha parmetros.
   * 
   * @param key chave para a mensagem que contm parmetros.
   * @param args argumentos para a mensagem.
   * 
   * @return mensagem preenchida pelos argumentos e localizada.
   */
  public final String getString(final String key, final Object[] args) {
    return application.getString(key, args);
  }

  /**
   * Consulta a existncia de uma string de locale, com base na classe de um
   * objeto.
   * 
   * @param key a chave de consulta
   * 
   * @return um indicativo
   */
  public final boolean hasClassString(final String key) {
    return application.hasClassString(this.getClass(), key);
  }

  /**
   * Consulta a existncia de uma string de locale
   * 
   * @param key a chave de consulta
   * 
   * @return um indicativo
   */
  public final boolean hasString(final String key) {
    return application.hasString(key);
  }

  /**
   * Redefine o mtodo setVisible() do frame.  chamada tambm no caso
   * <code>dispose()</code>
   * 
   * @param visible indica se o dilogo deve ou no ser exibido.
   */
  @Override
  public void setVisible(final boolean visible) {
    if (visible) {
      application.addWindow(this);
      setLocationRelativeTo(getOwner());
    }
    else {
      application.removeWindow(this);
    }
    super.setVisible(visible);
  }

  /**
   * Construtor s a aplicao como referncia, usando a dilogo principal como
   * janela-me e o ttulo com o nome da aplicao.
   * 
   * @param application aplicao.
   */
  public ApplicationComponentDialog(final A application) {
    this(application, application.getApplicationFrame());
  }

  /**
   * Construtor que recebe um Dialog como referncia
   * 
   * @param owner janela-me.
   * @param application aplicao.
   */
  public ApplicationComponentDialog(final A application, final Window owner) {
    this(application, owner, ModalityType.DOCUMENT_MODAL);
  }

  /**
   * Construtor que recebe um Dialog como referncia
   * 
   * @param owner janela-me.
   * @param application aplicao.
   * @param modalityType modalidade.
   */
  public ApplicationComponentDialog(final A application, final Window owner,
    final ModalityType modalityType) {
    super(owner, application.getName(), modalityType);
    this.application = application;
    setDefaultCloseOperation(DISPOSE_ON_CLOSE);
    addEscListener();
    setMinimumSize(new Dimension(200, 100));
  }
}
