package csbase.client.applications.algorithmsmanager.actions;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.rmi.RemoteException;
import java.security.AccessControlException;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.ZipEntry;

import javax.swing.ImageIcon;
import javax.swing.JFileChooser;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.algorithmsmanager.dialogs.AlgorithmVersionInfoPanel;
import csbase.client.desktop.LocalTask;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.externalresources.ExternalResources;
import csbase.client.externalresources.LocalFile;
import csbase.client.externalresources.StandaloneLocalFile;
import csbase.client.externalresources.ZipLocalFile;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.SingletonFileChooser;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.CSBaseException;
import csbase.logic.SyncRemoteFileChannel;
import csbase.remote.ClientRemoteLocator;
import csbase.util.FileSystemUtils;
import csbase.util.Unzip;

/**
 * Essa classe representa a ao de importar um pacote de verso de algoritmo,
 * na funcionalidade de gerenciamento de algoritmos.
 * 
 * O mtodo <code> handleEditOperation </code> deve ser redefinido para que seja
 * aberto o dilogo para importao do pacote de verso para o algoritmo
 * selecionado no painel de seleo.
 * 
 */
public class VersionImportAction extends CommonVersionAction {
  /** Modo de seleo para o browser de arquivos. */
  private int selectionMode;

  /** Flag indicando se deve permitir que mais de um arquivo seja importado. */
  private boolean multiSelectionEnabled;

  /**
   * Constri a ao de importar um pacote de verso de algoritmo.
   * 
   * @param versionInfoPanel painel pai que criou a ao
   * @param icon imagem da ao
   */
  public VersionImportAction(AlgorithmVersionInfoPanel versionInfoPanel,
    ImageIcon icon) {
    super(versionInfoPanel, icon);
    setSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
    multiSelectionEnabled = false;
    setEnabled(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleVersionOperation() {
    if (getSelectedAlgorithm() == null) {
      return;
    }

    if (FileSystemUtils.canRead()) {
      fileSystemUpload();
    }
    else {
      appletUpload(); // #TODO A remover quando todos sistemas CSBase forem
      // assinados
    }
  }

  /**
   * Atribui o modo de seleo do browser para escolha de arquivos.
   * 
   * @param selectionMode modo de seleo do browser de arquivos
   */
  public void setSelectionMode(int selectionMode) {
    this.selectionMode = selectionMode;
  }

  /**
   * Obtm o modo de seleo do browser para escolha de arquivos.
   * 
   * @return o modo de seleo do browser de arquivos
   */
  public int getSelectionMode() {
    return selectionMode;
  }

  /**
   * Prepara o servidor para a importao do arquivo.
   * 
   * @param filePath caminho relativo do arquivo a ser importado.
   * 
   * @return informaes para a transferncia dados.
   * 
   * @throws RemoteException se ocorrer falha de RMI
   */
  protected RemoteFileChannelInfo prepareUpload(String filePath)
    throws RemoteException {
    if (getSelectedAlgorithm() == null) {
      return null;
    }
    // O comportamento padro do upload de verso  descompactar qualquer
    // arquivo zip que seja carregado no servidor.
    return ClientRemoteLocator.algorithmService.prepareUploadVersionPack(
      getSelectedAlgorithm().getId(), filePath);
  }

  /**
   * Obtm o arquivo a ser importado e o diretrio de destino na rvore de
   * algoritmos. Cria ento uma {@link csbase.client.desktop.LocalTask} para
   * importar o arquivo.
   */
  private void fileSystemUpload() {
    final File[] sourceFiles = getSourceFiles();
    if (!validateFiles(sourceFiles)) {
      return;
    }

    final LocalFile[] localFiles = createLocalFiles(sourceFiles);

    final List<RequestTransfer> requests = new ArrayList<RequestTransfer>();
    for (LocalFile sourceFile : localFiles) {
      RemoteFileChannelInfo requestInfo = getRequestInfo(sourceFile);
      if (requestInfo != null) {
        requests.add(new RequestTransfer(sourceFile, requestInfo));
      }
    }

    Task<Void> task = new LocalTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        for (RequestTransfer request : requests) {
          request.transfer();
        }
      }

      @Override
      protected void handleError(Exception error) {
        if (error instanceof IOException) {
          error.printStackTrace();
          StandardDialogs.showErrorDialog(getWindow(), getName(),
            LNG.get("algomanager.error.upload_fatal"));
          return;
        }
        super.handleError(error);
      }
    };
    String waitMsg = LNG.get("algomanager.msg.upload_wait");
    task.execute(getWindow(), getName(), waitMsg, false, true);
  }

  /**
   * Exibe uma pgina para upload de um arquivo de documentao, a partir do
   * disco rgido do usurio para o servidor. Essa funcionalidade s est
   * disponvel via applet.
   */
  protected void appletUpload() {
    new ImportVersionDialog().show();
  }

  /**
   * Retorna os arquivos locais a serem importados.
   * 
   * @return arquivo de origem.
   */
  private File[] getSourceFiles() {
    // Acesso ao sistema de arquivos permitido: obtm arquivos com o
    // JFileChooser
    JFileChooser chooser = SingletonFileChooser.getInstance();
    chooser.setFileSelectionMode(getSelectionMode());
    chooser.setMultiSelectionEnabled(multiSelectionEnabled);

    int returnVal = chooser.showOpenDialog(getWindow());
    if (returnVal == JFileChooser.CANCEL_OPTION) {
      return new File[0];
    }

    if (multiSelectionEnabled) {
      return chooser.getSelectedFiles();
    }
    else {
      return new File[] { chooser.getSelectedFile() };
    }
  }

  /**
   * Verifica se os arquivos especificados so vlidos, isto , se possuem nomes
   * aceitos pelas regras de nomenclatura dos arquivos do CSBase e se realmente
   * existem no sistema local de arquivos.
   * 
   * @param files Os arquivos a serem verificados.
   * 
   * @return <tt>true</tt> se todos os arquivos tem seus nomes vlidos e existem
   *         no cliente.
   */
  private boolean validateFiles(File[] files) {
    return 0 < files.length && validateFileExists(files) && validateZip(files)
      && validateFileName(files);
  }

  /**
   * Cria instncias de {@link LocalFile} representando os arquivos no sistema
   * do cliente.
   * 
   * @param files O arquivos do cliente.
   * 
   * @return instncias de {@link LocalFile} representando os arquivos no
   *         sistema do cliente.
   */
  private LocalFile[] createLocalFiles(File[] files) {
    try {
      // Se tiver algum diretrio, envia tudo como um arquivo zipado.
      for (File file : files) {
        if (file.isDirectory()) {
          try {
            String fileName =
              String.format("%s.zip", file.getParentFile().getName());
            return new LocalFile[] { new ZipLocalFile(fileName, files) };
          }
          catch (OutOfMemoryError e) {
            String fatalErrorMsg =
              LNG.get("algomanager.error.upload_fatal.outOfMemory");
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              fatalErrorMsg, e);
            return new LocalFile[0];
          }
          catch (IOException e) {
            String fatalErrorMsg = LNG.get("algomanager.error.upload_fatal.io");
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              fatalErrorMsg, e);
            return new LocalFile[0];
          }
          catch (Exception e) {
            String fatalErrorMsg = LNG.get("algomanager.error.upload_fatal");
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              fatalErrorMsg, e);
            return new LocalFile[0];
          }
        }
      }

      List<LocalFile> sourceFiles = new ArrayList<LocalFile>();
      for (File selFile : files) {
        if (selFile == null) {
          continue;
        }
        StandaloneLocalFile standAloneFile = new StandaloneLocalFile(selFile);
        sourceFiles.add(standAloneFile);
      }

      return sourceFiles.toArray(new LocalFile[sourceFiles.size()]);
    }
    catch (AccessControlException ex1) {
      // Acesso ao sistema de arquivos negado (rodando em "sandbox"): obtm
      // arquivos com a API JNLP. No  possvel a importao de diretrios.
      String fatalErrorMsg = LNG.get("algomanager.error.upload_fatal");
      if (!ExternalResources.getInstance().isEnabled()) {
        StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
          fatalErrorMsg);
        return null;
      }
      try {
        return ExternalResources.getInstance().openMultiFileDialog(".", null);
      }
      catch (CSBaseException ex2) {
        StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
          fatalErrorMsg, ex2);
        return null;
      }
      catch (IOException ex3) {
        StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
          LNG.get("algomanager.error.upload_io"), ex3);
        return null;
      }
    }
  }

  /**
   * <p>
   * Obtm um {@link RemoteFileChannelInfo}, isto , um objeto contendo as
   * informaes necessrias para solicitar a transferncia.
   * </p>
   * <p>
   * Verifica primeiro se o arquivo j existe no servidor. Em caso positivo,
   * confirma se o usurio deseja ou no sobrescrever o arquivo remoto.
   * </p>
   * 
   * @param file arquivo local a ser importado.
   * 
   * @return objeto contendo as informaes necessrias para a transferncia.
   */
  private RemoteFileChannelInfo getRequestInfo(LocalFile file) {
    final String fileName;
    try {
      fileName = file.getName();
    }
    catch (IOException e) {
      StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
        LNG.get("algomanager.error.upload_io"), e);
      return null;
    }

    Task<RemoteFileChannelInfo> prepareUploadTask =
      new RemoteTask<RemoteFileChannelInfo>() {
        @Override
        protected void performTask() throws Exception {
          setResult(prepareUpload(fileName));
        }
      };

    String waitMsg;
    waitMsg = LNG.get("algomanager.msg.upload_wait");
    if (!prepareUploadTask
      .execute(getWindow(), getName(), waitMsg, false, true)) {
      return null;
    }
    return prepareUploadTask.getResult();
  }

  /**
   * Valida a existncia dos arquivos no cliente.
   * 
   * @param files arquivos a serem importados.
   * 
   * @return <tt>true</tt> se todos os arquivos existem no cliente.
   */
  private boolean validateFileExists(File[] files) {
    List<File> invalidFiles = new ArrayList<File>();
    for (File file : files) {
      if (!file.exists()) {
        invalidFiles.add(file);
      }
    }
    if (0 < invalidFiles.size()) {

      // Formatando uma lista dos arquivos invlidos para mostrar no dilogo de
      // erro.
      StringBuilder invalidFilesList = new StringBuilder();
      for (File file : invalidFiles) {
        invalidFilesList.append(String.format(
          LNG.get("algomanager.error.upload_files_not_found.files.list"),
          file.getName()));
      }
      StandardErrorDialogs.showErrorDialog(getWindow(), getName(), String
        .format(LNG.get("algomanager.error.upload_files_not_found"),
          invalidFilesList.toString()));

      return false;
    }
    else {
      return true;
    }
  }

  /**
   * Valida os arquivos que forem zip.<br>
   * Verifica se os arquivos contidos em cada zip, seguem as regras de
   * nomenclatura dos arquivos do CSBase.
   * 
   * @param files
   * 
   * @return <tt>true</tt> se no forem encontrados arquivos invlidos dentro
   *         dos zips.
   */
  private boolean validateZip(File[] files) {

    try {
      for (File file : files) {
        if (!file.getName().toLowerCase().endsWith("zip")) {
          break;
        }

        Unzip unzipped = new Unzip(file);
        List<ZipEntry> entries = unzipped.listZipEntries();
        for (ZipEntry entry : entries) {

          // O nome da entrada representa o caminho de um arquivo compactado.
          // O replace abaixo pega o ltimo elemento deste caminho.
          //
          // Por exemplo, se tivessemos um zip com as seguintes entradas:
          // f0.xxx
          // dir1/
          // dir1/f1.xxx
          // dir1/dir2/
          // dir1/dir2/f2.xxx
          // dir1/dir2/dir3/
          // 
          // Aplicando o replace abaixo a cada uma das entradas, teramos:
          // f0.xxx
          // dir1
          // f1.xxx
          // dir2
          // f2.xxx
          // dir3
          String fileName =
            entry.getName().replaceAll("^(.*/)?([^/]+)/?$", "$2");
          if (!ClientUtilities.isValidFileName(fileName)) {
            StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
              LNG.get("UTIL_NAME_CHARACTER_ERROR"));
            return false;
          }
        }
      }
      return true;
    }
    catch (IOException e) {
      StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
        LNG.get("algomanager.error.upload_io"), e);
      return false;
    }
    catch (Exception e) {
      e.printStackTrace();
      StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
        LNG.get("UTIL_NAME_CHARACTER_ERROR"));
      return false;
    }
  }

  /**
   * Valida se o nome de cada arquivo possui possui apenas caracteres aceitos
   * pelas regras de nomenclatura dos arquivos do CSBase.
   * 
   * @param files
   * 
   * @return <tt>true</tt> se o nome de todos os arquivos forem vlidos.
   */
  private boolean validateFileName(File... files) {
    for (File file : files) {
      // Se o nome for invlido, mostra uma mensagem de erro.
      if (!ClientUtilities.isValidFileName(file.getName())) {
        StandardErrorDialogs.showErrorDialog(getWindow(), getName(),
          LNG.get("UTIL_NAME_CHARACTER_ERROR"));

        return false;
      }
      // Se for um diretrio, valida o nome dos filhos.
      if (file.isDirectory() && !validateFileName(file.listFiles())) {
        return false;
      }
    }

    return true;
  }

  /**
   * Modela uma requisio de transferncia.
   */
  static class RequestTransfer {
    /**
     * Arquivo local.
     */
    LocalFile file;

    /**
     * Informao para construo do canal remoto.
     */
    RemoteFileChannelInfo info;

    /**
     * Cria uma requisio de transferncia.
     * 
     * @param file O arquivo local.
     * @param info A informao para construo do canal remoto.
     */
    RequestTransfer(LocalFile file, RemoteFileChannelInfo info) {
      this.file = file;
      this.info = info;
    }

    /**
     * Efetua a transferncia do arquivo local atravs do canal remoto.
     * 
     * @throws Exception Em caso de falha na operao.
     */
    void transfer() throws Exception {
      SyncRemoteFileChannel channel =
        new SyncRemoteFileChannel(info.getIdentifier(), info.isWritable(),
          info.getHost(), info.getPort(), info.getKey());
      channel.open(!info.isWritable());

      InputStream inputStream = file.getInputStream();
      channel.syncTransferFrom(inputStream, 0, file.getLength());
      inputStream.close();
      channel.close();
    }
  }

  /**
   * Um dilogo para importao de pacotes de verso.
   * 
   */
  private class ImportVersionDialog {
    /**
     * Chama o browser para upload do pacote de verso. O upload s est
     * disponivel quando os recursos externos esto habilitados.
     */
    public void show() {
      if (getSelectedAlgorithm() == null) {
        return;
      }
      if (ExternalResources.getInstance().isEnabled()) {
        String address =
          AlgorithmManagementProxy.getVersionURL(
            getSelectedAlgorithm().getId(), getWindow());
        if (address == null) {
          return;
        }
        try {
          final URL url = new URL(address);
          ExternalResources.getInstance().showDocument(url);
        }
        catch (Exception e) {
          StandardErrorDialogs.showErrorDialog(getWindow(),
            LNG.get("algomanager.title.error"),
            LNG.get("PRJ_PROJECT_FILE_UPLOAD_IO_ERROR"), e);
        }
      }
      else {
        StandardDialogs.showInfoDialog(getWindow(),
          LNG.get("algomanager.title.error"),
          LNG.get("algomanager.error.upload_not_available"));
      }
    }
  }

}
