package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.util.List;
import java.util.SortedSet;
import java.util.Vector;

import javax.swing.DefaultComboBoxModel;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.Application;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.applications.algorithmsmanager.models.DataInterface;
import csbase.logic.algorithms.AlgorithmInfo;

/**
 * Essa classe implementa uma viso de uma lista de algoritmos para seleo.
 * 
 */
public class AlgorithmListView {
  /** Algoritmos que fazem parte da viso */
  private SortedSet<AlgorithmInfo> algorithms;

  /** Lista de algoritmos */
  private JList algoList;

  /** Painel de seleo que criou essa viso de algoritmos */
  private AlgorithmSelectionPanel selectionPanel;

  /**
   * Constri uma lista de algoritmos.
   * 
   * @param selectionPanel painel de seleo que criou essa viso
   * @param algorithms algoritmos a serem exibidos
   */
  public AlgorithmListView(AlgorithmSelectionPanel selectionPanel,
    SortedSet<AlgorithmInfo> algorithms) {
    this.selectionPanel = selectionPanel;
    this.algorithms = algorithms;
    algoList = buildAlgorithmListComponent();
  }

  /**
   * Obtm uma lista com os dados de algoritmo selecionados.
   * 
   * @return a lista de algoritmos selecionados
   */
  public List<DataInterface> getSelectedDataList() {
    List<DataInterface> selectedList = new Vector<DataInterface>();
    Object[] selectedValues = algoList.getSelectedValues();
    for (Object value : selectedValues) {
      selectedList.add((DataInterface) value);
    }
    return selectedList;
  }

  /**
   * Obtm o painel referente a essa viso de algoritmos.
   * 
   * @return o painel referente a essa viso de algoritmos
   */
  public JPanel getPanel() {
    return buildAlgoListPanel();
  }

  /**
   * Obtm o painel contendo a lista de algoritmos.
   * 
   * @return o painel com a lista de algoritmos
   */
  private JPanel buildAlgoListPanel() {
    JPanel algoListPanel = new JPanel(new GridBagLayout());

    JLabel description =
      new JLabel(getApplication().getString(
        "AlgorithmListView.description.algorithms.available"));
    algoList = buildAlgorithmListComponent();
    JScrollPane algoPane = new JScrollPane(algoList);

    algoListPanel.add(description, new GBC(0, 0).none().west().insets(5, 5, 5,
      5));
    algoListPanel.add(algoPane, new GBC(0, 1).both().west().insets(5, 5, 5, 5));
    return algoListPanel;
  }

  /**
   * Obtm a aplicao.
   * 
   * @return a aplicao
   */
  private Application getApplication() {
    return selectionPanel.getApplication();
  }

  /**
   * Constri o componente com scroll de uma lista de algoritmos.
   * 
   * @return o componente criado
   */
  private JList buildAlgorithmListComponent() {
    if (algoList == null) {
      algoList = new JList();
      algoList.addListSelectionListener(new ListSelectionListener() {

        @Override
        public void valueChanged(ListSelectionEvent e) {
          if (e.getSource() instanceof JList) {
            JList algoList = (JList) e.getSource();
            selectionPanel.initAlgorithmSelection((AlgorithmListItem) algoList
              .getSelectedValue());

            if (algoList.getSelectedIndex() != -1) {
              algoList.ensureIndexIsVisible(algoList.getSelectedIndex());
            }
          }
        }
      });
    }
    algoList.setModel(new DefaultComboBoxModel(getAlgorithmListItems()));
    algoList.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    algoList.setLayoutOrientation(JList.VERTICAL);
    algoList.setVisibleRowCount(15);

    return algoList;
  }

  /**
   * Obtm os itens da lista correspondentes s informaes dos algoritmos
   * considerados para criao da viso.
   * 
   * @return os itens da lista correspondentes s informaes dos algoritmos
   *         considerados para criao da viso
   */
  private Vector<AlgorithmListItem> getAlgorithmListItems() {
    Vector<AlgorithmListItem> listItems = new Vector<AlgorithmListItem>();
    for (AlgorithmInfo algoInfo : algorithms) {
      listItems.add(new AlgorithmListItem(algoInfo));
    }
    return listItems;
  }

  /**
   * Obtm o item da lista correspondente s informaes do algoritmo
   * especificado.
   * 
   * @param algoInfo informaes do algoritmo
   * @return o item da lista correspondente s informaes do algoritmo
   *         especificado
   */
  public AlgorithmListItem getAlgorithmItem(AlgorithmInfo algoInfo) {
    int index = getAlgorithmIndex(algoInfo);
    return (AlgorithmListItem) algoList.getModel().getElementAt(index);
  }

  /**
   * Obtm o ndice correspondente s informaes do algoritmo especificado.
   * 
   * @param algoInfo informaes do algoritmo
   * @return o ndice correspondente s informaes do algoritmo especificado
   */
  private int getAlgorithmIndex(AlgorithmInfo algoInfo) {
    int i = 0;
    for (AlgorithmInfo algorithm : algorithms) {
      if (algorithm.equals(algoInfo)) {
        return i;
      }
      i++;
    }
    return -1;
  }

  /**
   * Obtm o ndice do algoritmo selecionado na lista.
   * 
   * @return o ndice do algoritmo selecionado na lista
   */
  public int getSelectedIndex() {
    return algoList.getSelectedIndex();
  }

  /**
   * Obtm o nmero de elementos selecionados na lista.
   * 
   * @return o nmero de elementos selecionados na lista
   */
  public int getSelectionCount() {
    return algoList.getSelectedValues().length;
  }

  /**
   * Obtm o nmero de elementos da lista.
   * 
   * @return o nmero de elementos da lista
   */
  public int getCount() {
    return algorithms.size();
  }

  /**
   * Seleciona na lista o algoritmo correspondente s informaes do algoritmo
   * especificado.
   * 
   * @param algoInfo informaes do algoritmo
   */
  public void selectAlgorithm(AlgorithmInfo algoInfo) {
    int algoIndex = getAlgorithmIndex(algoInfo);
    selectAlgorithmIndex(algoIndex);
  }

  /**
   * Seleciona na lista o algoritmo correspondente ao ndice especificado.
   * 
   * @param index ndice a ser selecionado
   */
  public void selectAlgorithmIndex(int index) {
    algoList.setSelectedIndex(index);
  }
}
