package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.List;
import java.util.SortedSet;
import java.util.Vector;

import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.AbstractSimpleApplicationPanel;
import csbase.client.applications.algorithmsmanager.AlgorithmsManager;
import csbase.client.applications.algorithmsmanager.DataPropertiesListener;
import csbase.client.applications.algorithmsmanager.models.DataInterface;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;

/**
 * Dilogo para associao de algoritmos a uma ou mais categorias.
 * 
 */
public class AlgorithmsAndCategoriesBindPanel extends
  AbstractSimpleApplicationPanel<AlgorithmsManager> implements
  DataPropertiesListener {

  /** Dado corrente para edio das associaes */
  private DataInterface currentData;

  /** Filhos disponveis na rvore de dados */
  private SortedSet<DataInterface> availableItems;

  /** Filhos associados aos ns selecionados */
  private SortedSet<DataInterface> selectedItems;

  /** Painel para associao dos dados de algoritmos e categorias */
  private DataBindingPanel dataBindPanel;

  /** Tipo do item a ser associado ao dado corrente */
  private ItemType itemType;

  /** Listeners da associao entre algoritmos e categorias */
  private List<DataPropertiesListener> bindListeners;

  /**
   * Tipo do item a ser associado ao dado corrente.
   * 
   */
  public enum ItemType {
    /** Item  uma categoria */
    CATEGORY_ITEM,
    /** Item  um algoritmo */
    ALGORITHM_ITEM;
  }

  /**
   * Constri a janela para associao de algoritmos e categorias.
   * 
   * @param application aplicao que contm o painel
   * @param currentData dado corrente para edio das associaes
   * @param availableItems dados a serem disponibilizados para associao no
   *        dado corrente
   * @param selectedItems dados a serem previamente selecionados como associados
   *        ao dado corrente
   * @param itemType tipo do item a ser associado ao dado corrente
   */
  public AlgorithmsAndCategoriesBindPanel(AlgorithmsManager application,
    DataInterface currentData, SortedSet<DataInterface> availableItems,
    SortedSet<DataInterface> selectedItems, ItemType itemType) {
    super(application);
    this.currentData = currentData;
    this.availableItems = availableItems;
    this.selectedItems = selectedItems;
    this.itemType = itemType;
    this.bindListeners = new Vector<DataPropertiesListener>();
    buildPanel();
  }

  /**
   * Obtm o painel que realiza a associao dos dados.
   * 
   * @return o painel
   */
  private JPanel getDataBindingPanel() {
    dataBindPanel =
      new DataBindingPanel(getApplication(), availableItems, selectedItems,
        getDataBindingDescription());
    dataBindPanel.addDataBindListener(this);
    return dataBindPanel;
  }

  /**
   * Verifica se o tipo dos itens pais) selecionados na rvore de dados para
   * realizar a associao so categorias.
   * 
   * @return retorna true se os ns pais selecionados na rvore de dados so
   *         categorias, caso contrrio, retorna false
   */
  private boolean isCategoryItem() {
    return itemType.equals(ItemType.CATEGORY_ITEM);
  }

  /**
   * Obtm os identificadores das categorias a serem associadas ao dado
   * corrente.
   * 
   * @return os identificadores das categorias a serem associadas
   */
  public List<String> getSelectedCategoriesIds() {
    List<String> categoryIds = new Vector<String>();
    if (isCategoryItem()) {
      categoryIds = Arrays.asList(getSelectedItemIds().toArray(new String[0]));
    }
    else {
      categoryIds.add(getCurrentData().getId());
    }
    return categoryIds;
  }

  /**
   * Obtm os identificadores dos algoritmos a serem associados.
   * 
   * @return os identificadores dos algoritmos a serem associados
   */
  public List<Object> getSelectedAlgorithmsIds() {
    List<Object> algorithmIds = new Vector<Object>();
    if (!isCategoryItem()) {
      algorithmIds = getSelectedItemIds();
    }
    else {
      algorithmIds.add(getCurrentData().getId());
    }
    return algorithmIds;
  }

  /**
   * Obtm os identificadores dos dados selecionados.
   * 
   * @return os identificadores dos dados selecionados
   */
  private List<Object> getSelectedItemIds() {
    List<Object> selectedIds = new Vector<Object>();
    List<DataInterface> selectedItems = dataBindPanel.getSelectedItems();
    for (DataInterface item : selectedItems) {
      selectedIds.add(item.getId());
    }
    return selectedIds;
  }

  /**
   * Obtm o dado corrente, onde sero editadas as associaes.
   * 
   * @return o dado corrente
   */
  public DataInterface getCurrentData() {
    return currentData;
  }

  @Override
  protected void buildPanel() {
    setLayout(new GridBagLayout());
    JPanel dataBindingPanel = getDataBindingPanel();
    add(dataBindingPanel, new GBC(0, 1).both().west().insets(5, 5, 5, 5));
  }

  /**
   * Obtm o texto para descrever o tipo de associo a ser feita com os dados.
   * 
   * @return o texto que descreve o tipo de associao dos dados
   */
  private String getDataBindingDescription() {
    String label;
    if (isCategoryItem()) {
      label =
        getString("AlgorithmsAndCategoriesBindPanel.label.binding.category");
    }
    else {
      label =
        getString("AlgorithmsAndCategoriesBindPanel.label.binding.algorithm");
    }
    return label;
  }

  /**
   * Atualiza os itens dos painis de disponveis e selecionados.
   * 
   * @param availableData itens disponveis
   * @param selectedData itens selecionados
   */
  public void updateItems(SortedSet<DataInterface> availableData,
    SortedSet<DataInterface> selectedData) {
    dataBindPanel.updateItems(availableData, selectedData);
  }

  /**
   * Atribui o dado corrente selecionado no painel de seleo de dados.
   * 
   * @param currentData dado corrente selecionado
   */
  public void setCurrentData(DataInterface currentData) {
    this.currentData = currentData;
  }

  /**
   * A partir dos nomes das categorias adicionadas pelo usurio, as categorias
   * sero efetivamente criadas no servidor.
   */
  public void bindData() {
    DataInterface currentData = getCurrentData();
    if (currentData == null) {
      return;
    }

    setCurrentData(currentData);
    List<Object> algorithmIds = getSelectedAlgorithmsIds();
    List<String> categoryIds = getSelectedCategoriesIds();
    if (isCategoryItem()) {
      updateCategoriesAlgorithms(algorithmIds, categoryIds);
    }
    else {
      setAlgorithmsToCategories(algorithmIds, categoryIds);
    }
  }

  /**
   * Atribui efetivamente um conjunto de algoritmos a um conjunto de categorias.
   * 
   * @param algorithmIds identificadores dos algoritmos
   * @param categoryIds identificadores das categorias
   * 
   */
  private void setAlgorithmsToCategories(List<Object> algorithmIds,
    List<String> categoryIds) {
    CategorySet modifiedCategories =
      AlgorithmManagementProxy.setAlgorithmsToCategories(algorithmIds,
        categoryIds, getApplication().getApplicationFrame(),
        AlgorithmManagementProxy.AlgorithmOperation.ADMIN_ALGORITHM);
    if (modifiedCategories.isEmpty()
      || modifiedCategories.getSize() != categoryIds.size()) {
      String msg =
        getString("AlgorithmsAndCategoriesBindPanel.msg.create.failed");
      MessageFormat.format(msg, new Object[] { modifiedCategories.getSize() });
      getApplication().showError(msg);
    }
  }

  /**
   * Atualiza os algoritmos das categorias especificadas.
   * 
   * @param algorithmIds idenficadores dos algoritmos que fazem parte das
   *        categorias especificadas
   * @param categoryIds idenficadores das categorias modificadas
   */
  private void updateCategoriesAlgorithms(List<Object> algorithmIds,
    List<String> categoryIds) {
    // Associa os algoritmos s categorias selecionadas para cont-los
    CategorySet bindModifiedCategories =
      bindAlgorithmsToCategories(algorithmIds, categoryIds);

    if (bindModifiedCategories == null) {
      return;
    }

    // Desassocia os algoritmos das categorias onde os algoritmos foram
    // removidos
    List<String> unbindCategoryIds =
      getRemovedCategoriesIds(selectedItems, bindModifiedCategories);
    CategorySet unbindModifiedCategories =
      unbindAlgorithmsToCategories(algorithmIds, unbindCategoryIds);
    if (unbindModifiedCategories == null) {
      return;
    }

    Integer categoriesCounter = categoryIds.size() + unbindCategoryIds.size();

    Integer modifiedCounter =
      bindModifiedCategories.getSize() + unbindModifiedCategories.getSize();

    if (modifiedCounter == 0 || modifiedCounter != categoriesCounter) {
      String msg =
        getString("AlgorithmsAndCategoriesBindPanel.msg.create.failed");
      MessageFormat.format(msg, new Object[] { modifiedCounter });
      getApplication().showError(msg);
    }
  }

  /**
   * Obtm os identificadores das categorias onde os algoritmos foram removidos.
   * 
   * @param oldCategories categorias antigas
   * @param selectedCategories categorias selecionadas para conter o algoritmo
   * 
   * @return os identificadores das categorias onde os algoritmos foram
   *         removidos.
   */
  private List<String> getRemovedCategoriesIds(
    SortedSet<DataInterface> oldCategories, CategorySet selectedCategories) {
    List<String> removedCategories = new Vector<String>();
    if (selectedCategories != null) {
      for (DataInterface oldCategory : oldCategories) {
        String removedCatId = oldCategory.getId();
        Category category = selectedCategories.getRootCategory(removedCatId);
        if (category == null) {
          removedCategories.add(removedCatId);
        }
      }
    }
    return removedCategories;
  }

  /**
   * Associa efetivamente os algoritmos s categorias.
   * 
   * @param algorithmIds identificadores dos algoritmos
   * @param categoryIds identificadores das categorias
   * 
   * @return retorna true se os algoritmos foram associados s categorias com
   *         sucesso, caso contrrio retorna false
   */
  public CategorySet bindAlgorithmsToCategories(
    final List<Object> algorithmIds, final List<String> categoryIds) {
    return AlgorithmManagementProxy.bindAlgorithmsToCategories(algorithmIds,
      categoryIds, getApplication().getApplicationFrame(),
      AlgorithmManagementProxy.AlgorithmOperation.ADMIN_ALGORITHM);
  }

  /**
   * Desassocia efetivamente os algoritmos das categorias.
   * 
   * @param algorithmIds identificadores dos algoritmos
   * @param categoryIds identificadores das categorias
   * 
   * @return retorna true se os algoritmos foram desassociados das categorias
   *         com sucesso, caso contrrio retorna false
   */
  public CategorySet unbindAlgorithmsToCategories(
    final List<Object> algorithmIds, final List<String> categoryIds) {
    return AlgorithmManagementProxy.unbindAlgorithmsFromCategories(
      algorithmIds, categoryIds, getApplication().getApplicationFrame(),
      AlgorithmManagementProxy.AlgorithmOperation.ADMIN_ALGORITHM);
  }

  /**
   * Inicializa os dados do painel que relaciona categorias e algoritmos.
   * 
   * @param selectedAlgorithm algoritmo selecionado
   * @param availableCategories categorias disponveis
   * @param algorithmCategories categorias dos algoritmos
   */
  public void initializeData(DataInterface selectedAlgorithm,
    SortedSet<DataInterface> availableCategories,
    SortedSet<DataInterface> algorithmCategories) {
    setCurrentData(selectedAlgorithm);
    this.availableItems = availableCategories;
    this.selectedItems = algorithmCategories;
    updateItems(availableCategories, algorithmCategories);
    revalidate();
  }

  /**
   * Verifica se houve alterao nos itens de categorias associados ao dado
   * corrente.
   * 
   * @return retorna true se houver alterao nos itens associados, ou false
   *         caso os itens tenham permanecido como os originais
   */
  public boolean wasModified() {
    if (isCategoryItem()) {
      return wasCategoriesChanged();
    }
    return wasAlgorithmsChanged();
  }

  /**
   * Verifica se houve alterao nos itens de algoritmos associados ao dado
   * corrente.
   * 
   * @return retorna true se houver alterao nos itens associados, ou false
   *         caso os itens tenham permanecido como os originais
   */
  private boolean wasAlgorithmsChanged() {
    List<Object> selectedAlgorithmIds = getSelectedAlgorithmsIds();
    if (selectedItems.size() != selectedAlgorithmIds.size()) {
      return true;
    }
    for (DataInterface oldCategory : selectedItems) {
      String oldCatId = oldCategory.getId();
      if (!selectedAlgorithmIds.contains(oldCatId)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se houve alterao nos itens categorias associados ao dado
   * corrente.
   * 
   * @return retorna true se houve alterao nos itens categorias associados ao
   *         dado corrente, caso contrrio, retorna false
   */
  private boolean wasCategoriesChanged() {
    if (!isCategoryItem()) {
      return false;
    }
    List<String> selectedCategoriesIds = getSelectedCategoriesIds();
    if (selectedItems.size() != selectedCategoriesIds.size()) {
      return true;
    }
    for (DataInterface oldCategory : selectedItems) {
      String oldCatId = oldCategory.getId();
      if (!selectedCategoriesIds.contains(oldCatId)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Adiciona um listener de mudanas na associao entre algoritmos e
   * categorias.
   * 
   * @param bindListener listener de mudanas na associao entre algoritmos e
   *        categorias
   */
  public void addBindListener(DataPropertiesListener bindListener) {
    bindListeners.add(bindListener);
  }

  /**
   * Remove um listener de mudanas na associao entre algoritmos e categorias.
   * 
   * @param bindListener listener de mudanas na associao entre algoritmos e
   *        categorias
   */
  public void removeBindListener(DataPropertiesListener bindListener) {
    bindListeners.remove(bindListener);
  }

  /**
   * Notifica os listeners de que ocorreram mudanas na associao entre
   * algoritmos e categorias
   * 
   */
  public void notifyBindListener() {
    for (DataPropertiesListener bindListener : bindListeners) {
      bindListener.propertiesChanged(wasModified());
    }
  }

  @Override
  public void propertiesChanged(boolean wasModified) {
    notifyBindListener();
  }
}
