/*
 * CommandDetailsFrame.java
 * 
 * $Author:$ $Revision: 155481 $ - $Date:$
 */
package csbase.client.applications.commandsmonitor;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.text.MessageFormat;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.WindowConstants;
import javax.swing.table.DefaultTableModel;

import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.algorithms.commands.cache.events.AbstractCommandUpdatedEventListener;
import csbase.client.algorithms.commands.cache.events.CommandUpdatedEvent;
import csbase.client.applications.Application;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.remote.srvproxies.SGAProxy;
import csbase.client.util.ClientUtilities;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;

/**
 * A classe <code>CommandDetailsFrame</code>  responsvel por exibir as
 * informaes detalhadas de um comando em execuo.
 * 
 * @author ururahy
 * @version $Revision: 155481 $
 */
public class CommandDetailsFrame extends DesktopComponentFrame implements
  MonitorFrame {

  /**
   * Mapa de instncias de CommandDetailsFrame tendo como chave o cmdId
   */
  private static final Lock showCommandDetailsFrameLock = new ReentrantLock();

  /**
   * A aplicao que chamou a CommandDetailsFrame.
   */
  private final Application callerApp;

  /**
   * Identificador do projeto do comando a ser monitorado
   */
  private final Object projectId;

  /**
   * Identificador do comando a ser monitorado
   */
  private final String commandId;

  /**
   * Nome do servidor que executa o comando
   */
  private final String sgaName;

  /**
   * Informaes sobre a monitorao do comando
   */
  private CommandInfo commandInfo;

  /**
   * Painel com informaes dinmicas do comando
   */
  private DynamicInfoPanel dynamicInfoPanel;
  /**
   * Boto de cancelar comando. Guardamos a sua referncia para que possamos
   * desabilita-lo quando o comando tiver terminado.
   */
  private JButton cancelButton;

  /**
   * Constri o monitorador detalhado de um comando.
   * 
   * @param projectId Identificador do projeto.
   * @param cmdId Identificador do comando a ser detalhado.
   * @param sgaName Nome do servidor que executa o comando.
   * @param app A Aplicao que chamou CommandDetailsFrame.
   */
  private CommandDetailsFrame(final Object projectId, final String cmdId,
    final String sgaName, final Application app) {
    super(createDesktopComponentFrameIndex(cmdId), app.getApplicationFrame(),
      app.getString("CommandDetailsFrame.title"));
    this.callerApp = app;

    this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

    this.projectId = projectId;
    this.commandId = cmdId;
    this.sgaName = sgaName;

    initialize();
  }

  /**
   * Cria uma janela contendo detalhes do comando. Caso j exista uma janela
   * detalhando aquele comando, ao invs de criar uma nova, esta  trazida para
   * frente.
   * 
   * @param projectId identificador nico do projeto no qual o comando foi
   *        persistido.
   * @param cmdId Identificador do comando a ser detalhado.
   * @param sgaName Nome do servidor que executa o comando.
   * @param app A Aplicao que chamou CommandDetailsFrame.
   */
  public static final void showCommandDetailsFrame(final Object projectId,
    final String cmdId, final String sgaName, final Application app) {

    final String index = createDesktopComponentFrameIndex(cmdId);
    CommandDetailsFrame commandDetailsFrame =
      (CommandDetailsFrame) getDesktopComponentFrame(index);
    if (null == commandDetailsFrame) {
      showCommandDetailsFrameLock.lock();
      try {
        commandDetailsFrame =
          (CommandDetailsFrame) getDesktopComponentFrame(index);
        if (null == commandDetailsFrame) {
          commandDetailsFrame =
            new CommandDetailsFrame(projectId, cmdId, sgaName, app);
          class WindowCloser extends WindowAdapter {
            CommandDetailsFrame commandDetailsFrame;

            WindowCloser(CommandDetailsFrame commandDetailsFrame) {
              this.commandDetailsFrame = commandDetailsFrame;
            }

            @Override
            public void windowClosed(WindowEvent e) {
              commandDetailsFrame.stop();
              removeDesktopComponentFrame(index);
            }
          }

          commandDetailsFrame.addWindowListener(new WindowCloser(
            commandDetailsFrame));
        }
      }
      finally {
        showCommandDetailsFrameLock.unlock();
      }
    }
    // Se j existe, traz pra frente a janela.
    commandDetailsFrame.display();
  }

  /**
   * Termina a monitorao de um comando.
   */
  public void stop() {
    dynamicInfoPanel.stop();
  }

  /**
   * Cria os listeners de comando, os componentes visuais e os exibe.
   */
  private void initialize() {
    loadCommandInfo();
    createGUIComponents();
    display();
  }

  /**
   * Carrega as informaes da monitorao do comando.
   * 
   * @return true se conseguiu carregar as informaes com sucesso.
   */
  private boolean loadCommandInfo() {
    RemoteTask<CommandInfo> task = new RemoteTask<CommandInfo>() {
      @Override
      protected void performTask() throws Exception {
        CommandInfo newCommandInfo =
          CommandsCache.getInstance().getCommand(projectId, commandId);
        setResult(newCommandInfo);
      }
    };
    if (!task.execute(this, getTitle(),
      getString("CommandDetailsFrame.msg.loadCommand"))) {
      return false;
    }

    if (null == task.getResult()) {
      String msg =
        MessageFormat.format(getString("CommandDetailsFrame.error.comm_sga"),
          new Object[] { sgaName });
      final String title = getTitle();
      StandardDialogs.showErrorDialog(this, title, msg);
    }
    commandInfo = task.getResult();
    return true;
  }

  /**
   * Exibe a janela de monitorao
   */
  private void display() {
    pack();
    center(callerApp.getApplicationFrame());
    setVisible(true);
  }

  /**
   * Cria os componentes da GUI.
   */
  private void createGUIComponents() {
    Container cp = getContentPane();
    cp.setLayout(new BorderLayout(5, 5));

    dynamicInfoPanel = new DynamicInfoPanel();
    cp.add(dynamicInfoPanel, BorderLayout.CENTER);
    dynamicInfoPanel.start();

    cp.add(createButtonsPanel(), BorderLayout.SOUTH);
  }

  /**
   * Cria o painel de botes.
   * 
   * @return painel de botes
   */
  private JPanel createButtonsPanel() {
    JPanel bp = new JPanel(new FlowLayout());

    // Boto de fechamento da janela
    JButton closeButton = new JButton(getString("CommandDetailsFrame.close"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        CommandDetailsFrame.this.close();
      }
    });

    // Boto de cancelamento do comando
    cancelButton = new JButton(getString("CommandDetailsFrame.cancel"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        String msg =
          getString("CommandDetailsFrame.cancel.confirm")
            + "\n"
            + MessageFormat.format(
              getString("CommandDetailsFrame.cancel.details"), commandInfo
                .getTip(), sgaName);
        String descr = commandInfo.getDescription();
        if (descr != null && descr.length() > 0) {
          msg =
            msg
              + MessageFormat.format(
                getString("CommandDetailsFrame.cancel.description"), descr);
        }
        // Obtm confirmao para cancelar o(s) comando(s).
        int res =
          StandardDialogs.showYesNoDialog(CommandDetailsFrame.this, getTitle(),
            msg);
        if (res == JOptionPane.YES_OPTION) {
          SGAProxy.killCommand(sgaName, commandId);
        }
      }
    });
    // Desabilita o boto de cancelar caso o comando j tenha terminado.
    cancelButton.setEnabled(commandInfo.getStatus() != CommandStatus.FINISHED);

    ClientUtilities.adjustEqualSizes(cancelButton, closeButton);
    bp.add(cancelButton);
    bp.add(closeButton);
    return bp;
  }

  /**
   * Consulta a uma string de locale
   * 
   * @param key a chave de consulta
   * 
   * @return a string baseada no locale.
   */
  protected final String getString(final String key) {
    return callerApp.getString(key);
  }

  /**
   * Cria um ndice para ser utilizado como identificador da janela e assim
   * evitar duplicatas.
   * 
   * @param cmdId id
   * @return ndice que identifique a janela.
   */
  private static String createDesktopComponentFrameIndex(final Object cmdId) {
    return "CommandDetailsFrame_" + cmdId;
  }

  /**
   * Painel que contm as informaes do comando.
   */
  private class DynamicInfoPanel extends JPanel {

    /**
     * Flag indicativo de sada da thread de atualizao das informaes do
     * comando
     */
    private final AtomicBoolean running;

    /** Componente que exibe o algoritmo e verso do comando executado */
    JTextField tipText;
    /** Componente que exibe a descrio do comando */
    JTextField descriptionText;
    /** Componente que exibe o nome do servidor rodando o comando */
    JTextField serverText;
    /** Componente que exibe as informaes dinmicas */
    CommandDetailsTable detailsTable;
    /** Listener responsvel pela atualizao dos comandos. */
    AbstractCommandUpdatedEventListener listener;

    /**
     * Constri o painel que contm as informaes dinmicas do comando.
     */
    public DynamicInfoPanel() {
      super(new GridBagLayout());
      this.running = new AtomicBoolean(false);
      this.listener =
        new AbstractCommandUpdatedEventListener(commandInfo.getProjectId(),
          commandInfo.getId()) {

          @Override
          protected void eventFired(final CommandUpdatedEvent.Type type,
            final CommandInfo cmd) {
            getStatusBar().hideStatusBar();
            switch (type) {
              case removed:
              case success:
              case end:
              case error:
              case failed:
              case killed:
              case no_code:
              case lost: {
                SwingThreadDispatcher.invokeLater(new Runnable() {
                  @Override
                  public void run() {
                    stop();
                    // Desabilita o botao cancelar assim que o comando acaba de executar.
                    cancelButton.setEnabled(false);
                    detailsTable.commandFinished(type, cmd);
                  }
                });
                break;
              }
              default: {
                // Atualiza a instncia com o valor recebido pelo evento.
                commandInfo = cmd;
                SwingThreadDispatcher.invokeLater(new Runnable() {
                  @Override
                  public void run() {
                    dynamicInfoPanel.updateInfo();
                  }
                });
              }
            }
          }

          @Override
          protected void eventInterrupted(Exception exception,
            String description) {
            getStatusBar().setError(description);
          }
        };

      createGUIComponents();
    }

    /**
     * Atualiza as informaes da monitorao do comando. Essa atualizao 
     * comandada pela thread de tratamento de eventos do AWT.
     */
    private void updateInfo() {
      // Verifica o estado da comunicao com o servidor
      if (commandInfo == null) {
        detailsTable.noDynamicInfo();
        return;
      }
      tipText.setText(commandInfo.getTip());
      descriptionText.setText(commandInfo.getDescription());
      serverText.setText(commandInfo.getSGAName());
      detailsTable.updateDynamicInfo();
    }

    /**
     * Come a atualizar os dados do comando.
     */
    public synchronized void start() {
      if (running.compareAndSet(false, true)) {
        CommandsCache.getInstance().addEventListener(listener);
      }
    }

    /**
     * Para de atualizar os dados do comando.
     */
    public synchronized void stop() {
      if (running.compareAndSet(true, false)) {
        CommandsCache.getInstance().removeEventListener(listener);
      }
    }

    /**
     * Cria os componentes da GUI.
     */
    private void createGUIComponents() {
      GridBagConstraints constraints = new GridBagConstraints();
      constraints.gridheight = 1;
      constraints.weightx = 0;
      constraints.weighty = 0;
      constraints.anchor = GridBagConstraints.NORTHWEST;

      JLabel tipLabel =
        new JLabel(getString("CommandDetailsFrame.command") + ":");
      constraints.gridx = 0;
      constraints.gridy = 0;
      constraints.fill = GridBagConstraints.NONE;
      constraints.insets = new Insets(12, 12, 0, 0);
      add(tipLabel, constraints);

      tipText = new JTextField();
      tipText.setEditable(false);
      constraints.gridx = 1;
      constraints.gridy = 0;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.insets = new Insets(12, 6, 0, 11);
      add(tipText, constraints);

      JLabel descriptionLabel =
        new JLabel(getString("CommandDetailsFrame.description") + ":");
      constraints.gridx = 0;
      constraints.gridy = 1;
      constraints.fill = GridBagConstraints.NONE;
      constraints.insets = new Insets(6, 12, 0, 0);
      add(descriptionLabel, constraints);

      descriptionText = new JTextField();
      descriptionText.setEditable(false);
      constraints.gridx = 1;
      constraints.gridy = 1;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.insets = new Insets(6, 6, 0, 11);
      add(descriptionText, constraints);

      JLabel serverLabel =
        new JLabel(getString("CommandDetailsFrame.server") + ":");
      constraints.gridx = 0;
      constraints.gridy = 2;
      constraints.fill = GridBagConstraints.NONE;
      constraints.insets = new Insets(6, 12, 0, 0);
      add(serverLabel, constraints);

      serverText = new JTextField();
      serverText.setEditable(false);
      constraints.gridx = 1;
      constraints.gridy = 2;
      constraints.fill = GridBagConstraints.HORIZONTAL;
      constraints.insets = new Insets(6, 6, 0, 11);
      add(serverText, constraints);

      JLabel detailsLabel =
        new JLabel(getString("CommandDetailsFrame.details") + ":");
      constraints.gridx = 0;
      constraints.gridy = 3;
      constraints.gridwidth = 2;
      constraints.fill = GridBagConstraints.NONE;
      constraints.insets = new Insets(6, 12, 0, 11);
      add(detailsLabel, constraints);

      detailsTable = new CommandDetailsTable();
      constraints.gridx = 0;
      constraints.gridy = 4;
      constraints.weightx = 1;
      constraints.weighty = 1;
      constraints.fill = GridBagConstraints.BOTH;
      constraints.insets = new Insets(6, 12, 0, 11);
      add(new JScrollPane(detailsTable), constraints);

      updateInfo();
    }
  }

  /**
   * Tabela que exibe os detalhes do comando.
   */
  private class CommandDetailsTable extends SortableTable {

    /** Pontilhado que aparece quando no h informao para exibir */
    private static final String DOTTED = "------";

    /** Modelo para a tabela */
    protected DefaultTableModel model;
    /** Linha que contm o estado do comando */
    private int statusRow;
    /** Linha que contm o tempo total de execuo */
    private int etimeRow;

    CommandDetailsTable() {
      super();
      statusRow = -1;
      etimeRow = -1;
      model =
        new DefaultTableModel(new Object[] {
            getString("CommandDetailsFrame.details.key"),
            getString("CommandDetailsFrame.details.value") }, 0);
      setModel(model);
      setPreferredScrollableViewportSize(new Dimension(300, 150));
    }

    String getPercString(Double value) {
      if (value == null) {
        return DOTTED;
      }
      return value.intValue() + " %";
    }

    String getWallTimeString(Integer value) {
      if (value == null) {
        return DOTTED;
      }
      return FormatUtils.formatInterval(value);
    }

    void addNewRow(String key, String value) {
      model.addRow(new Object[] { key, value });
    }

    void updateDynamicInfo() {
      // Limpa a tabela e coloca os novos valores.
      model.setRowCount(0);

      String status;
      if (commandInfo.isValid()) {
        if (commandInfo.isQueued()) {
          status = getString("CommandDetailsFrame.queued");
        }
        else {
          status =
            getString("CommandDetailsFrame.status." + commandInfo.getStatus());
        }
      }
      else {
        status = getString("CommandDetailsFrame.disconnected");
      }
      addNewRow(getString("CommandDetailsFrame.status"), status);

      statusRow = 0;
      addNewRow(getString("CommandDetailsFrame.time.elapsed"),
        getWallTimeString(commandInfo.getWallTimeSec()));
      etimeRow = 1;
      addNewRow(getString("CommandDetailsFrame.cpu.usage"),
        getPercString(commandInfo.getCpuPerc()));
      addNewRow(getString("CommandDetailsFrame.ram.usage"),
        getPercString(commandInfo.getRAMMemoryPerc()));
      addNewRow(getString("CommandDetailsFrame.swap.usage"),
        getPercString(commandInfo.getSwapMemoryPerc()));
      Map<String, String> allInfo = commandInfo.getSpecificData();
      if (allInfo != null && allInfo.size() > 0) {
        for (String key : allInfo.keySet()) {
          addNewRow(key, allInfo.get(key));
        }
      }
    }

    void noDynamicInfo() {
      // Coloca pontilhado em todos os valores.
      int nlines = model.getRowCount();
      for (int row = 0; row < nlines; row++) {
        if (row != statusRow) {
          model.setValueAt(DOTTED, row, 1);
        }
      }
    }

    void commandFinished(final CommandUpdatedEvent.Type type,
      final CommandInfo cmd) {
      // Coloca pontilhado em todos os valores.
      noDynamicInfo();
      if (statusRow < 0) {
        return;
      }
      String returnValue = getString("CommandDetailsFrame.status.FINISHED");

      switch (type) {
        case removed:
        case end: {
          setEndTime(cmd);
          break;
        }
        case success: {
          setEndTime(cmd);
          returnValue +=
            " (" + getString("CommandDetailsFrame.command.end.success") + ")";
          break;
        }
        case error: {
          returnValue +=
            " (" + getString("CommandDetailsFrame.command.end.error") + ")";
          break;
        }
        case failed: {
          returnValue +=
            " (" + getString("CommandDetailsFrame.command.end.failure") + ")";
          break;
        }
        case killed: {
          returnValue +=
            " (" + getString("CommandDetailsFrame.command.end.killed") + ")";
          break;
        }
        case lost: {
          returnValue +=
            " (" + getString("CommandDetailsFrame.command.end.lost") + ")";
          break;
        }
        case no_code: {
          returnValue +=
            " (" + getString("CommandDetailsFrame.command.end.no_code") + ")";
          break;
        }
      }
      model.setValueAt(returnValue, statusRow, 1);
    }

    /**
     * Seta o tempo final de execuo do comando.
     * 
     * @param cmd
     */
    private void setEndTime(final CommandInfo cmd) {
      if (etimeRow >= 0) {
        int wallTime = cmd.getWallTimeSec();
        String sWallTime = getWallTimeString(wallTime);
        model.setValueAt(sWallTime, etimeRow, 1);
      }
    }
  }
}
