package csbase.client.applications.commandsmonitor;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.JTable;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import csbase.client.applications.commandsmonitor.events.SelectionChangedEvent;
import csbase.client.facilities.commandtable.AbstractCommandMonitoringTable;
import csbase.client.util.event.EventManager;

/**
 * Gerencia a seleo de tabelas garantindo, que a qualquer momento, apenas uma
 * tenha linhas selecionadas.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class UniqueTableSelectionManager {

  /**
   * Gerenciador de eventos utilizado para informar alteraes nas selees das
   * tabelas.
   */
  private EventManager eventManager;
  /**
   * Marcador utilizado para indicar quando eventos devem ser ignorados e assim
   * evitar notificaes indevidas.
   */
  private AtomicBoolean ignoreEvents = new AtomicBoolean(false);

  /**
   * Indica qual tabela de comando est com o foco.
   */
  private AbstractCommandMonitoringTable onFocusTable;
  /**
   * Lista das tabelas sendo gerenciadas.
   */
  private List<JTable> tables = new ArrayList<JTable>();
  /**
   * Mapa que relaciona as tabelas a seus ouvintes.
   */
  private Map<JTable, ListSelectionListener> selectionListenersByTable =
    new HashMap<JTable, ListSelectionListener>();

  /**
   * Construtor.
   * 
   * @param eventManager gerenciador de eventos que ser utlizado para informar
   *        quando a seleo de uma tabela for alterada.
   */
  public UniqueTableSelectionManager(EventManager eventManager) {
    this.eventManager = eventManager;
  }

  /**
   * Adiciona uma tabela para ter suas selees gerenciadas.
   * 
   * @param table tabela a ser gerenciada.
   */
  public void startManaging(AbstractCommandMonitoringTable table) {
    tables.add(table);
    ListSelectionListener listener = createListener(table);
    selectionListenersByTable.put(table, listener);
    table.getSelectionModel().addListSelectionListener(listener);
  }

  /**
   * Remove uma tabela da gerncia de selees.
   * 
   * @param table tabela a ser removida.
   */
  public void stopManaging(final JTable table) {
    if (tables.remove(table)) {
      ListSelectionListener listener = selectionListenersByTable.get(table);
      table.getSelectionModel().removeListSelectionListener(listener);
      if (table.equals(onFocusTable)) {
        onFocusTable = null;
      }
    }
  }

  /**
   * Remove a seleo de todas as tabelas.
   */
  public void clearSelections() {
    if (ignoreEvents.compareAndSet(false, true)) {
      for (JTable aTable : tables) {
        aTable.getSelectionModel().clearSelection();
      }
      onFocusTable = null;
      ignoreEvents.set(false);
      eventManager.fireEvent(new SelectionChangedEvent());
    }
  }

  /**
   * @return a tabela com o foco.  a ltima que teve alguma clula selecionada.
   */
  public JTable getTableWithFocus() {
    return onFocusTable;
  }

  /**
   * @param table tabela para qual o ouvinte deve ser criado.
   * 
   * @return o ouvinte que ir gerenciar a seleo da tabela.
   */
  private ListSelectionListener createListener(
    final AbstractCommandMonitoringTable table) {

    return new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent e) {
        /*
         * Se essa tabela no tiver foco, remove as selees das demais e mudar
         * o foco para esta.
         */
        if (!table.equals(onFocusTable)) {
          if (ignoreEvents.compareAndSet(false, true)) {
            onFocusTable = table;
            for (JTable aTable : tables) {
              if (table.equals(aTable))
                continue;
              aTable.getSelectionModel().clearSelection();
            }
            ignoreEvents.set(false);
          }
        }

        /*
         * Envia uma notificao indicando que a seleo foi alterada.
         */
        if (!ignoreEvents.get() && null != onFocusTable) {
          eventManager.fireEvent(new SelectionChangedEvent(onFocusTable
            .getSelectedCommands()));
        }
      }
    };
  }
}
