/*
 * $Id$
 */

package csbase.client.applications.commandsmonitor.actions;

import javax.swing.ImageIcon;

import csbase.client.applications.AbstractSimpleApplicationAction;
import csbase.client.applications.commandsmonitor.CommandsMonitor;
import csbase.client.applications.commandsmonitor.events.SelectionChangedEvent;
import csbase.client.util.event.EventListener;
import csbase.logic.CommandInfo;

/**
 * Ao de comandos que se auto-habilita, ou desabilita, de acordo com os
 * comandos selecionados.
 * 
 * @author Tecgraf / PUC-Rio
 */
public abstract class AbstractCommandsAction extends
  AbstractSimpleApplicationAction<CommandsMonitor> {

  /**
   * <tt>false</tt> indica que esta ao deve ser desabilitada caso haja mais
   * de um comando selecionado.
   */
  private boolean acceptMultipleCommands;
  /**
   * Comandos selecionados.
   */
  private CommandInfo[] commands;

  /**
   * Construtor.
   * 
   * @param application Aplicao que detm esta ao.
   * @param acceptMultipleCommands <tt>false</tt> faz com que esta ao seja
   *        desabilitada caso haja mais de um comando selecionado.
   * @param icon cone que representa esta ao
   */
  protected AbstractCommandsAction(CommandsMonitor application,
    boolean acceptMultipleCommands, ImageIcon icon) {
    super(application, icon);
    initialize(acceptMultipleCommands);
  }

  /**
   * Construtor que cria uma ao sem cone.
   * 
   * @param application Aplicao que detm esta ao.
   * @param acceptMultipleCommands <tt>false</tt> faz com que esta ao seja
   *        desabilitada caso haja mais de um comando selecionado.
   */
  protected AbstractCommandsAction(CommandsMonitor application,
    boolean acceptMultipleCommands) {
    super(application);
    initialize(acceptMultipleCommands);
  }

  /**
   * Filtro indicando se esta ao aceita ou no tratar o comando.<br>
   * Este mtodo  utilizado pelo mtodo {@link #setEnable(CommandInfo[])} para
   * definir o estado desta ao de acordo com os comandos selecionados.
   * 
   * @param command Comando.
   * 
   * @return <tt>true</tt> se esta ao trata o comando passado.
   */
  protected abstract boolean accept(CommandInfo command);

  /**
   * Obtm os comandos selecionados.
   * 
   * @return os comandos selecionados.
   */
  protected CommandInfo[] getSelectedCommands() {
    return commands;
  }

  /**
   * Obtm o primeiro comando selecionado.<br>
   * Utilizado por aes que no permitem a seeo de mais de um comando. Neste
   * caso o primeiro comando selecionado seria o nico comando selecionado.
   * 
   * @return o primeiro comando selecionado.
   */
  protected CommandInfo getSelectedCommand() {
    return commands[0];
  }

  /**
   * Inicializa a ao criando um ouvinte de seleo de comandos na aplicao
   * que ir chamar o mtodo {@link #setEnable(CommandInfo[])} quando uma
   * seleo for feita, mantendo assim o estado desta ao.
   * 
   * @param acceptMultipleCommands <tt>false</tt> faz com que esta ao seja
   *        desabilitada caso haja mais de um comando selecionado.
   */
  private void initialize(boolean acceptMultipleCommands) {
    this.acceptMultipleCommands = acceptMultipleCommands;
    setEnabled(false);
    getApplication().addSelectionChangedListener(
      new EventListener<SelectionChangedEvent>() {
        @Override
        public void eventFired(SelectionChangedEvent event) {
          setEnable(event.getSelection());
        }
      });
  }

  /**
   * Habilita ou desabilita essa ao de acordo com os comandos selecionados e a
   * quantidade dos mesmos.
   * 
   * @param commands comandos selecionados.
   */
  private void setEnable(CommandInfo[] commands) {
    if (null == commands || 0 == commands.length) {
      setEnabled(false);
    }
    else if (!acceptMultipleCommands && 1 < commands.length) {
      setEnabled(false);
    }
    else {
      for (CommandInfo aCommand : commands) {
        if (!accept(aCommand)) {
          setEnabled(false);
          return;
        }
      }
      this.commands = commands;
      setEnabled(true);
    }
  }
}
