/*
 * $Id$
 */

package csbase.client.applications.commandsmonitor.actions;

import java.awt.event.ActionEvent;
import java.rmi.RemoteException;
import java.util.Arrays;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.commandsmonitor.CommandsMonitor;
import csbase.client.desktop.RemoteTask;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;

/**
 * Remove a persistncia dos comandos selecionados.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class RemoveCommandsAction extends AbstractCommandsAction {

  /**
   * Construtor.
   * 
   * @param application Aplicao que detm esta ao.
   */
  public RemoveCommandsAction(CommandsMonitor application) {
    super(application, true, ApplicationImages.ICON_DELETE_16);
  }

  /**
   * Filtro indicando que esta ao aceita qualquer comando que esteja no estado
   * {@link CommandStatus#FINISHED terminado} ou
   * {@link CommandStatus#SYSTEM_FAILURE falha no sistema}. {@inheritDoc}
   */
  @Override
  protected boolean accept(CommandInfo command) {
    return command.getStatus() == CommandStatus.FINISHED
      || command.getStatus() == CommandStatus.SYSTEM_FAILURE;
  }

  /**
   * Remove a persistncia dos comandos selecionados. {@inheritDoc}
   */
  @Override
  protected void handleActionPerformed(ActionEvent ae) throws Exception {
    removeCommands(getSelectedCommands());
  }

  /**
   * Remove a persistncia dos comandos.
   * 
   * @param commands Comandos que no sero mais persistidos.
   * 
   * @return <tt>false</tt> se o usurio cancelar a operao ou se houver um
   *         erro.
   */
  private boolean removeCommands(final CommandInfo[] commands) {
    if (!warnRemoveCommand()) {
      return false;
    }
    if (0 == commands.length) {
      return true;
    }

    class RemoveCommandTask extends RemoteTask<boolean[]> {
      @Override
      protected void performTask() throws RemoteException {
        setResult(CommandsCache.getInstance().removeCommands(
          Arrays.asList(commands)));
      }
    }

    String taskTitle =
      getString("RemoveCommandsAction.task.commands.remove.title");
    String taskMessage =
      getString("RemoveCommandsAction.task.commands.remove.message",
        commands.length);
    ApplicationFrame frame = getApplication().getApplicationFrame();
    RemoveCommandTask task = new RemoveCommandTask();
    if (!task.execute(frame, taskTitle, taskMessage)) {
      return false;
    }
    boolean[] cmdsResult = task.getResult();
    // Verifica se houve erro. Se algum comando no pode ser removido.
    boolean error = false;
    for (boolean cmdResult : cmdsResult) {
      error |= !cmdResult;
    }
    /*
     * Em caso de erro, mostrar mensagem com comandos que no puderam ser
     * removidos.
     */
    if (error) {
      String errorMessage =
        getString("RemoveCommandsAction.task.commands.remove.error",
          commands.length);
      getApplication().showError(errorMessage);
    }

    return !error;
  }

  /**
   * Avisa ao usurio que um comando est para ser removido e pede confirmao
   * da operao.
   * 
   * @return {@code true} o usurio informou que quer o comando seja removido ou
   *         {@code false} o usurio informou no quer cancelar a remoo de
   *         comando.
   */
  private boolean warnRemoveCommand() {
    String message =
      getString("RemoveCommandsAction.task.commands.remove.confirmation.query");
    ApplicationFrame frame = getApplication().getApplicationFrame();
    String title = frame.getTitle();
    String[] options = new String[] { LNG.get("UTIL_YES"), LNG.get("UTIL_NO") };

    int option =
      StandardDialogs.showOptionDialog(frame, title, message, options);
    switch (option) {
      case 0:
        return true;
      case 1:
        return false;
      default:
        String errorMessage =
          getString(
            "RemoveCommandsAction.task.commands.remove.confirmation.error",
            option);
        throw new IllegalStateException(errorMessage);
    }
  }
}
