/*
 * $Id: LogViewer.java 116866 2011-04-01 15:13:57Z cassino $
 */
package csbase.client.applications.csvviewer;

import java.awt.BorderLayout;
import java.util.List;

import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JOptionPane;
import javax.swing.JToolBar;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationProject;
import csbase.client.applications.csvviewer.actions.CloseCSVAction;
import csbase.client.applications.csvviewer.actions.HeaderConfigurationAction;
import csbase.client.applications.csvviewer.actions.NewCSVAction;
import csbase.client.applications.csvviewer.actions.OpenCSVAction;
import csbase.client.applications.csvviewer.actions.OpenLocalCSVAction;
import csbase.client.applications.csvviewer.actions.SaveCSVAction;
import csbase.client.applications.csvviewer.actions.SaveCSVAsAction;
import csbase.client.applications.csvviewer.actions.SaveLocalCSVAction;
import csbase.client.applications.flowapplication.multiflow.MultiFlowWizard;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.util.csvpanel.CSVExceptionNotification;
import csbase.client.util.csvpanel.CSVPanel;
import csbase.client.util.csvpanel.CSVTableModelNotification;
import csbase.client.util.event.EventListener;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil.OperationResult;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;

/**
 * Visualizador de arquivos CSV.
 */
public class CSVViewer extends ApplicationProject {

  /**
   * Painel de texto aonde fica o contedo do arquivo.
   */
  private final CSVPanel csvPanel;

  /**
   * Indica se o contedo do painel foi modificado pelo usurio.
   */
  private boolean contentChanged;

  /**
   * Indica se existe um projeto aberto e, portanto, as opes de projeto podem
   * ser habilitadas.
   */
  private boolean projectEnabled;

  /**
   * Ao de criar novo CSV.
   */
  private NewCSVAction newCSVAction;

  /**
   * Ao de abrir arquivo CSV de projeto.
   */
  private OpenCSVAction openCSVAction;

  /**
   * Ao de abrir arquivo CSV local.
   */
  private OpenLocalCSVAction openLocalCSVAction;

  /**
   * Ao de salvar arquivo CSV de projeto.
   */
  private SaveCSVAction saveCSVAction;

  /**
   * Ao de salvar uma cpia do arquivo CSV no projeto.
   */
  private SaveCSVAsAction saveCSVAsAction;

  /**
   * Ao de salvar arquivo CSV local.
   */
  private SaveLocalCSVAction saveLocalCSVAction;

  /**
   * Ao de fechar arquivo.
   */
  private CloseCSVAction closeCSVAction;

  /**
   * Ao de configurar o nmero de linhas que devem ser consideradas como
   * cabealho do CSV.
   */
  private HeaderConfigurationAction headerConfigurationAction;

  /**
   * Ao de sair da aplicao.
   */
  private ApplicationExitAction applicationExitAction;

  /**
   * Ao de mostrar janela com informaes sobre a aplicao.
   */
  private ApplicationAboutAction applicationAboutAction;

  /**
   * Adaptador que acompanha a abertura/fechamento de projetos no desktop.
   */
  private ProjectTreeAdapter projectTreeAdapter;

  /**
   * Tamanho padro da tabela.
   */
  public static final int DEFAULT_SIZE = 5;

  /**
   * Construtor
   * 
   * @param id identificador da aplicao
   */
  public CSVViewer(final String id) {
    super(id);
    csvPanel = new CSVPanel();
    /* Captura os eventos de excees gerados pelo painel de csv. */
    csvPanel
      .addExceptionListener(new EventListener<CSVExceptionNotification>() {
        @Override
        public void eventFired(CSVExceptionNotification event) {
          getApplicationFrame().getStatusBar().setError(event.getMessage(), 10);
        }
      });
    csvPanel
      .addCSVTableModelListener(new EventListener<CSVTableModelNotification>() {
        @Override
        public void eventFired(CSVTableModelNotification event) {
          setContentChanged(true);
        }
      });
    buildFrame(csvPanel);
    handleOpenProject();
    newCSV(DEFAULT_SIZE, DEFAULT_SIZE);
    setContentChanged(false);
  }

  /**
   * Abre um arquivo csv.
   * 
   * @param file o arquivo.
   */
  public final void openFile(final ClientFile file) {
    if (closeFile()) {
      getApplicationFrame().getStatusBar().clearStatus();
      csvPanel.openFile(file, true);
      String type = file.getType();
      if (type != null && type.equals("MFLX")) {
        csvPanel.setNumRowsAsHeader(MultiFlowWizard.CONFIG_FILE_HEADER_SIZE);
      }
      setContentChanged(false);
    }
  }

  /**
   * Salva o arquivo.
   * 
   * @param file o arquivo.
   */
  public final void saveFile(final ClientFile file) {
    getApplicationFrame().getStatusBar().clearStatus();
    ClientFile selectedFile = file;
    if (selectedFile == null) {
      List<String> fileTypes = getFileTypes();
      OperationResult result =
        ClientProjectFileChooserUtil.browseSingleFileInSaveMode(this,
          fileTypes, fileTypes.get(0), true, true);
      if (result == null) {
        return;
      }
      selectedFile = result.getClientProjectFile();
      if (selectedFile == null) {
        return;
      }
    }
    csvPanel.saveFile(selectedFile);
    setContentChanged(false);
  }

  /**
   * Determina se o contedo do painel foi modificado, atualizando o estado da
   * aplicao de acordo.
   * 
   * @param contentChanged verdadeiro se o contedo do painel foi modificado ou
   *        falso, caso contrrio.
   */
  private void setContentChanged(boolean contentChanged) {
    this.contentChanged = contentChanged;
    updateActions();
    updateTitle();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sendMessage(String name, Object value, String senderId) {
    if (value == null) {
      return;
    }
    if (name.equals(PROJECT_FILE_MESSAGE)) {
      ClientProjectFile file = (ClientProjectFile) value;
      openFile(file);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    if (closeFile()) {
      DesktopFrame desktopFrame = DesktopFrame.getInstance();
      desktopFrame.getTree().removeProjectTreeListener(projectTreeAdapter);
    }
  }

  /**
   * Confirma se o arquivo aberto atualmente pode ser fechado, verificando se
   * foram feitas modificaes ainda no salvas no arquivo. Caso positivo,
   * mostra um dilogo de confirmao para o usurio. O usurio pode escolher
   * salvar ou no o arquivo (prosseguindo com a operao de fechamento) ou
   * ainda cancelar a operao como um todo e manter o arquivo atual aberto.
   * 
   * @return verdadeira se a operao de fechamento foi confirmado ou falso,
   *         caso o usurio a tenha cancelado.
   */
  private boolean confirmFileClosing() {
    if (contentChanged) {
      int result =
        StandardDialogs.showYesNoCancelDialog(getApplicationFrame(), getName(),
          getString("confirmFileClosing.message"));
      switch (result) {
        case JOptionPane.YES_OPTION:
          saveFile(getFile());
          return true;
        case JOptionPane.NO_OPTION:
        case JOptionPane.CLOSED_OPTION:
          return true;
        case JOptionPane.CANCEL_OPTION:
          return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    return closeFile();
  }

  /**
   * Montagem do dilogo principal.
   * 
   * @param panel o painel de csv.
   */
  private void buildFrame(final CSVPanel panel) {
    final DesktopComponentFrame mainFrame = getApplicationFrame();
    createActions();
    mainFrame.getContentPane().add(createToolBar(), BorderLayout.NORTH);
    mainFrame.getContentPane().add(panel, BorderLayout.CENTER);
    mainFrame.setJMenuBar(createMenuBar());
    mainFrame.getStatusBar().showStatusBar();
    mainFrame.setSize(800, 600);
  }

  /**
   * Indica se aplicao deve aceitar arquivos locais.
   * 
   * @return indicativo
   */
  private boolean allowLocalFiles() {
    final String propName = "allow.local.files";
    boolean flag = getBooleanSpecificProperty(propName, false);
    return flag;
  }

  /**
   * Cria as aes da aplicao.
   */
  public void createActions() {
    newCSVAction = new NewCSVAction(this);
    openCSVAction = new OpenCSVAction(this);
    openLocalCSVAction = new OpenLocalCSVAction(this);
    saveLocalCSVAction = new SaveLocalCSVAction(this);
    saveCSVAsAction = new SaveCSVAsAction(this);
    saveCSVAction = new SaveCSVAction(this);
    closeCSVAction = new CloseCSVAction(this);
    headerConfigurationAction = new HeaderConfigurationAction(this);
    applicationExitAction = new ApplicationExitAction(this);
    applicationAboutAction = new ApplicationAboutAction(this);
  }

  /**
   * Cria a barra de menu na aplicao.
   * 
   * @return a barra de menu
   */
  public JMenuBar createMenuBar() {
    JMenuBar menu = new JMenuBar();

    JMenu fileMenu = new JMenu(getString("file.menu"));
    fileMenu.add(newCSVAction);
    fileMenu.add(openCSVAction);
    boolean allowLocalFiles = allowLocalFiles();
    if (allowLocalFiles) {
      fileMenu.add(openLocalCSVAction);
    }
    fileMenu.add(saveCSVAction);
    fileMenu.add(saveCSVAsAction);
    if (allowLocalFiles) {
      fileMenu.add(saveLocalCSVAction);
    }
    fileMenu.add(closeCSVAction);
    fileMenu.addSeparator();

    fileMenu.add(applicationExitAction);

    JMenu optionsMenu = new JMenu(getString("options.menu"));
    optionsMenu.add(headerConfigurationAction);

    JMenu helpMenu = new JMenu(getString("help.menu"));
    helpMenu.add(applicationAboutAction);

    menu.add(fileMenu);
    menu.add(optionsMenu);
    menu.add(helpMenu);
    return menu;
  }

  /**
   * Obtm a barra de ferramentas.
   * 
   * @return a barra de ferramentas.
   */
  private final JToolBar createToolBar() {
    JToolBar toolBar = new JToolBar();
    toolBar.setFloatable(false);
    toolBar.add(newCSVAction);
    toolBar.add(openCSVAction);
    if (allowLocalFiles()) {
      toolBar.add(openLocalCSVAction);
    }
    toolBar.add(saveCSVAction);
    toolBar.add(saveCSVAsAction);
    toolBar.addSeparator();
    toolBar.add(applicationAboutAction);
    return toolBar;
  }

  /**
   * Fecha o arquivo csv corrente.
   * 
   * @return verdadeiro se a operao de fechamento foi confirmada ou falso,
   *         caso o usurio a tenha cancelado.
   */
  public boolean closeFile() {
    boolean confirmed = confirmFileClosing();
    if (confirmed) {
      getApplicationFrame().getStatusBar().clearStatus();
      csvPanel.closeFile();
      setContentChanged(false);
    }
    return confirmed;
  }

  /**
   * Atualiza o estado das aes da aplicao. Habilitando/desabilitando as
   * aes de acordo com o estado atual.
   */
  private void updateActions() {
    openCSVAction.setEnabled(projectEnabled);
    saveCSVAsAction.setEnabled(projectEnabled);
    saveCSVAction.setEnabled(projectEnabled && contentChanged);
  }

  /**
   * Retorna o arquivo aberto no visualizador.
   * 
   * @return o arquivo aberto ou nulo, caso no haja nenhum arquivo aberto.
   */
  public ClientFile getFile() {
    return csvPanel.getFile();
  }

  /**
   * Atualiza o ttulo da aplicao.
   */
  private void updateTitle() {
    ClientFile file = getFile();
    String title = getName();
    if (file != null) {
      title += " - " + file.getStringPath();
      if (!csvPanel.isEditable()) {
        title += " [" + getString("readonly.tag") + "]";
      }
    }
    if (contentChanged) {
      title += "*";
    }
    getApplicationFrame().setTitle(title);
  }

  /**
   * Trata a existncia de projetos abertos no sistema.
   */
  private void handleOpenProject() {
    projectTreeAdapter = new ProjectTreeAdapter() {
      @Override
      public void projectChanged(CommonClientProject proj) {
        if (proj == null) {
          setProjectEnabled(false);
        }
        else {
          setProjectEnabled(true);
        }
      }
    };

    final DesktopFrame desktopFrame = DesktopFrame.getInstance();
    final ProjectTree projectTree = desktopFrame.getTree();
    projectTree.addProjectTreeListener(projectTreeAdapter);
    if (projectTree.getProject() == null) {
      setProjectEnabled(false);
    }
    else {
      setProjectEnabled(true);
    }
  }

  /**
   * Determina se as funes dependentes de projeto devem ser habilitadas.
   * 
   * @param projectEnabled verdadeiro se as opes devem ser habilitadas ou
   *        falso, caso contrrio.
   */
  private void setProjectEnabled(boolean projectEnabled) {
    this.projectEnabled = projectEnabled;
    updateActions();
  }

  /**
   * Cria um novo CSV com o nmero de linhas e colunas indicadas.
   * 
   * @param rows nmero de linhas.
   * @param cols nmero de colunas.
   */
  public void newCSV(int rows, int cols) {
    csvPanel.closeFile();
    csvPanel.setData(new String[rows][cols], true);
    setContentChanged(true);
  }

  /**
   * Determina o nmero de linhas no incio do arquivo CSV que devem ser
   * consideradas como cabealho da tabela.
   * 
   * @param lines nmero de linhas que devem ser consideradas como cabealho. Se
   *        igual a zero, no usa nenhuma das linhas no cabealho.
   */
  public void setNumRowsAsHeader(int lines) {
    csvPanel.setNumRowsAsHeader(lines);
  }

  /**
   * Indica o nmero de linhas no incio do arquivo CSV que devem ser
   * consideradas como cabealho da tabela.
   * 
   * @return nmero de linhas que devem ser consideradas como cabealho. Se
   *         igual a zero, no usa nenhuma das linhas no cabealho.
   */
  public int getNumRowsAsHeader() {
    return csvPanel.getNumRowsAsHeader();
  }
}
