/**
 * $Id$
 */
package csbase.client.applications.diskinfo;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;

import javax.swing.table.AbstractTableModel;

import csbase.client.applications.Application;

/**
 * Modelo de dados da tabela de dados de ocupao de disco por usurio. Criado
 * para permitir ordenao de forma mais adequada: se deixamos a tabela com
 * Object[][] ao invs de um modelo, e simplesmente chama-se
 * table.setAutoCreateRowSorter(true);, temos problema com ordenao dos valores
 * Double formatados
 * 
 * @author Tecgraf
 */
public class DiskUserTableModel extends AbstractTableModel {

  /** Nomes das colunas */
  private String[] columnNames = new String[3];

  /** Nmero de linhas */
  private int rowAmount;

  /** Dados formatados */
  private Object[][] formattedData;

  /**
   * Construtor
   * 
   * @param app app referncia para aplicao (para obter rtulos)
   * @param usersData dados dos usurios: hashtable onde a chave  o id do
   *        usurio e o valor  quantidade em megabytes ocupado por ele
   * 
   */
  public DiskUserTableModel(Application app, Hashtable<Object, Double> usersData) {
    columnNames[0] = app.getString("tableFrame.columUser");
    columnNames[1] = app.getString("tableFrame.columAmount");
    columnNames[2] = app.getString("tableFrame.columPercentage");
    formattedData = getFormattedData(usersData);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowCount() {
    return rowAmount;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    return columnNames.length;
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass(int c) {
    // Primeira coluna  String (id do usurio). As demais so double
    return (c == 0) ? String.class : Double.class;
  }

  /**
   * 
   * {@inheritDoc}
   */
  @Override
  public String getColumnName(int col) {
    return columnNames[col];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueAt(int rowIndex, int columnIndex) {
    return formattedData[rowIndex][columnIndex];
  }

  /**
   * Varre os dados de usurios e retorna dados formatados para a tabela: array
   * de arrays. Cada array  uma tupla com o nome do usurio, a quantidade de
   * espao em megabytes e o valor em porcentagem ocupado
   * 
   * @param usersData
   * @return os dados de usurios e retorna dados formatados para a tabela
   */
  private Object[][] getFormattedData(Hashtable<Object, Double> usersData) {
    Object[][] data = null;

    // Formata de acordo
    DecimalFormat formatter = new DecimalFormat("##0.##");
    DecimalFormatSymbols dfs = new DecimalFormatSymbols();
    dfs.setDecimalSeparator('.');
    formatter.setDecimalFormatSymbols(dfs);

    /*
     * Acrescenta os dados de usurio e montante por usurio, acumulando o total
     * para depois fazer o clculo das percentagens
     */
    Enumeration<Object> userIds = usersData.keys();
    Double totalAmount = 0.0;
    ArrayList<ArrayList<Object>> userDataList =
      new ArrayList<ArrayList<Object>>();
    while (userIds.hasMoreElements()) {
      String userId = (String) userIds.nextElement();
      Double amount = usersData.get(userId);
      totalAmount += amount;
      ArrayList<Object> userRecord = new ArrayList<Object>();
      userRecord.add(userId);
      userRecord.add(new Double(formatter.format(amount)));
      userDataList.add(userRecord);
    }

    rowAmount = userDataList.size();
    data = new Object[rowAmount][3];
    int i = 0;
    // Agora varre o userDataList para calcular o percentual
    for (Iterator<ArrayList<Object>> iterator = userDataList.iterator(); iterator
      .hasNext();) {
      ArrayList<Object> userRecord = iterator.next();
      Double userPercentage = ((Double) userRecord.get(1) / totalAmount) * 100;
      userRecord.add(new Double(formatter.format(userPercentage)));
      data[i++] = userRecord.toArray();
    }

    return data;
  }
}
