/*
 * $Id: FileTransferClientAction.java 90572 2009-04-16 19:28:11Z clinio $
 */

package csbase.client.applications.filetransferclient.actions.core;

import java.awt.event.ActionEvent;

import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JComponent;

import csbase.client.applications.ApplicationAction;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.filetransferclient.FileTransferClient;
import csbase.client.applications.filetransferclient.FileTransferClientUI;

/**
 * @author Tecgraf/PUC-Rio
 */
abstract public class FileTransferClientAction extends ApplicationAction<FileTransferClient> {

  /**
   * Mtodo de busca de uma string de internacionalizao dentro do escopo da
   * ao conforme o padro "<code>NomeDaClasse.TagDesejada</code>"
   * 
   * @param tag tag desejada
   * @return o texto de internacionalizao
   */
  final public String getString(final String tag) {
    final FileTransferClient ftpClient = getApplication();
    final Class<? extends FileTransferClientAction> cls = this.getClass();
    final String prefix = cls.getSimpleName();
    return ftpClient.getString(prefix + "." + tag);
  }

  /**
   * Consulta (se existe) de uma string de internacionalizao
   * 
   * @param tag tag desejada
   * @return indicativo
   */
  final public boolean hasString(final String tag) {
    final FileTransferClient ftpClient = getApplication();
    final Class<? extends FileTransferClientAction> cls = this.getClass();
    final String prefix = cls.getSimpleName();
    return ftpClient.hasString(prefix + "." + tag);
  }

  /**
   * Mtodo de consulta ao texto de <i>tooltip</i> da ao montado no padro "<code>NomeDaClasse.NAME_SUFIX</code>"
   * 
   * @return o texto (nome) da ao.
   * @see #getString(String)
   */
  final public String getName() {
    final String TAG_NAME = "name";
    if (!hasString(TAG_NAME)) return null;
    return getString(TAG_NAME);
  }

  /**
   * Mtodo de consulta ao texto de <i>tooltip</i> da ao montado no padro "<code>NomeDaClasse.tooltip</code>"
   * 
   * @return o texto de tooltip
   * @see #getString(String)
   */
  final public String getToolTipText() {
    final String TAG_TOOLTIP = "tooltip";
    if (!hasString(TAG_TOOLTIP)) return null;
    return getString(TAG_TOOLTIP);
  }

  /**
   * Consulta ao mnemnico da ao.
   * 
   * @return o mnemnico
   */
  final public Integer getMnemonic() {
    final String TAG_MNEMONIC = "mnemonic";
    if (!hasString(TAG_MNEMONIC)) return null;

    final String mn = getString(TAG_MNEMONIC);
    if (mn == null || FileTransferClientUI.isEmptyString(mn)) return null;

    return Integer.valueOf(mn.charAt(0));
  }

  /**
   * Mtodo de definio da imagem associada a ao.
   * 
   * @return um objeto do tipo <code>ImageIcon</code>.
   */
  final private ImageIcon getImageIcon() {
    final ImageIcon stdIcon = getStandardImageIcon();
    if (stdIcon != null) return stdIcon;

    final String IMG_ACTION_DIR = "actions";
    final String prefix = getClass().getSimpleName();
    final String name = prefix + ".16";
    final String pth = IMG_ACTION_DIR + "/" + name;
    final ImageIcon icon = getApplication().buildApplicationImage(pth);
    if (icon == ApplicationImages.ICON_NONE) return null;

    return icon;
  }

  /**
   * {@inheritDoc}
   * 
   * Repasse para outro mtodo com encapsulamento de excees.
   * 
   * @see #actionDone(JComponent)
   */
  @Override
  final public void actionPerformed(final ActionEvent ae) {
    try {
      final JComponent cmp = (JComponent) ae.getSource();
      actionDone(cmp);
    }
    catch (Exception e) {
      final FileTransferClient application = getApplication();
      application.showExceptionDialog(e);
    }
  }

  /**
   * Construtor padro.
   * 
   * @param ftpClient a aplicao que originou a ao.
   */
  public FileTransferClientAction(final FileTransferClient ftpClient) {
    super(ftpClient);
    final String name = getName();
    if (name != null) {
      putValue(Action.NAME, name);
    }

    final Integer mnemonic = getMnemonic();
    if (mnemonic != null) {
      putValue(Action.MNEMONIC_KEY, mnemonic);
    }

    final ImageIcon icon = getImageIcon();
    if (icon != null) {
      putValue(Action.SMALL_ICON, icon);
    }

    final String tooltip = getToolTipText();
    if (tooltip != null) {
      putValue(Action.SHORT_DESCRIPTION, tooltip);
    }
  }

  /**
   * Mtodo de ao a ser definido dentro do Planref.
   * 
   * @param component o componente que ativou a ao.
   * 
   * @throws Exception se houver falha na ao.
   */
  abstract public void actionDone(JComponent component) throws Exception;

  /**
   * Consulta se a ao possui uma imagem standard; se houver a imagem usada
   * aqui ser usada, seno valer a definida pelo mtodo
   * <code>getImageIcon()</code>.
   * 
   * @return uma imagem standard (ou null)
   */
  abstract public ImageIcon getStandardImageIcon();
}
