package csbase.client.applications.filetransferclient.panels.localpanel;

import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.Image;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.ListSelectionModel;
import javax.swing.SortOrder;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.applications.filetransferclient.FileTransferClient;
import csbase.client.applications.filetransferclient.FileTransferClientUI;
import csbase.client.applications.filetransferclient.actions.ChangeLocalDirectoryAction;
import csbase.client.applications.filetransferclient.actions.HomeLocalAction;
import csbase.client.applications.filetransferclient.actions.RefreshLocalAction;
import csbase.client.applications.filetransferclient.actions.UpLocalDirectoryAction;
import csbase.client.applications.filetransferclient.panels.core.FileTransferClientPanel;
import csbase.client.applications.filetransferclient.panels.queuepanel.FileTransferClientQueuePanel;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ProjectTree;
import csbase.client.project.tasks.GetChildFromNameTask;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.User;

/**
 * @author Tecgraf/PUC-Rio
 */
public class FileTransferClientLocalPanel extends FileTransferClientPanel {

  /**
   * Diretrio local
   */
  private ClientProjectFile localDirectory;

  /**
   * Tabela de arquivos locais.
   */
  private SortableTable localTable = new SortableTable();

  /**
   * 
   */
  private JTextField localDirText = new JTextField();

  /**
   * 
   */
  final private JTextField projectText = new JTextField(30);

  /**
   * 
   */
  final private JTextField userText = new JTextField(30);

  /**
   * Filtragem de arquivos comeados por ".".
   */
  private boolean dotFiltered = true;

  /**
   * Consulta se painel est filtrando elementos comeados por "."
   * 
   * @return indicativo
   */
  public boolean isDotFiltered() {
    return dotFiltered;
  }

  /**
   * Indica se painel deve filtrar elementos comeados por "."
   * 
   * @param dotFiltered indicativo
   */
  public void setDotFiltered(boolean dotFiltered) {
    this.dotFiltered = dotFiltered;
    refresh();
  }

  /**
   * Consulta o item (arquivo ou diretrio) selecionado.
   * 
   * @return o item selecionado
   */
  final public List<ClientProjectFile> getSelectedFiles() {
    final int[] sels = localTable.getSelectedRows();
    if (sels == null || sels.length == 0) {
      return null;
    }

    final LocalTableModel model = (LocalTableModel) localTable.getModel();
    final ArrayList<ClientProjectFile> modelList = model.getFileList();

    final List<ClientProjectFile> files = new ArrayList<ClientProjectFile>();
    for (int sel : sels) {
      final int modelSel = localTable.convertRowIndexToModel(sel);
      final ClientProjectFile file = modelList.get(modelSel);
      files.add(file);
    }

    Collections.unmodifiableList(files);
    return files;
  }

  /**
   * 
   */
  private void doubleClickedOnTable() {
    final List<ClientProjectFile> sels = getSelectedFiles();
    if (sels.size() != 1) {
      return;
    }
    final ClientProjectFile selected = sels.get(0);
    if (selected == null) {
      return;
    }

    final String fileName = selected.getName();
    if (selected.isDirectory()) {
      final ClientProjectFile localDir = getLocalDirectory();
      final ClientProjectFile newLocalDir =
        GetChildFromNameTask.runTask(localDir, fileName);
      setLocalDirectory(newLocalDir);
    }
    else {
      refresh();
      // TODO
      final FileTransferClient application = getApplication();
      final FileTransferClientQueuePanel queuePanel =
        application.getQueuePanel();
      queuePanel.refresh();
    }
  }

  /**
   * Montagem do painel
   * 
   * @param serverName nome do servidor (ttulo)
   */
  private void mountPanel(final String serverName) {
    setLayout(new BorderLayout());

    final JPanel infoPanel = buildInfoPanel(serverName);
    final JToolBar toolBar = buildToolBar();

    final String[] cNames =
      new String[] { getString("fd.column"), getString("name.column"),
          getString("size.column"), };
    final LocalTableModel model = new LocalTableModel(cNames);
    localTable.setModel(model);

    localTable.setComparator(0, new LocalTypeComparator());
    localTable.setNoSortStateEnabled(true);

    localTable.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    localTable.sort(0, SortOrder.ASCENDING);

    final LocalTableRenderer renderer = new LocalTableRenderer();
    localTable.setDefaultRenderer(Boolean.class, renderer);
    localTable.setDefaultRenderer(String.class, renderer);
    localTable.setDefaultRenderer(Long.class, renderer);
    localTable.setDefaultRenderer(ClientProjectFile.class, renderer);

    localTable.addMouseListener(new MouseAdapter() {
      @Override
      public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() == 2) {
          doubleClickedOnTable();
        }
      }
    });

    final TableColumnModel colModel = localTable.getColumnModel();
    localTable.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
    final TableColumn c0 = colModel.getColumn(0);
    final TableColumn c1 = colModel.getColumn(1);
    final TableColumn c2 = colModel.getColumn(2);
    c0.setPreferredWidth(10);
    c1.setPreferredWidth(150);
    c2.setPreferredWidth(40);

    final JScrollPane scrollTable = new JScrollPane(localTable);

    add(infoPanel, BorderLayout.NORTH);
    add(scrollTable, BorderLayout.CENTER);
    add(toolBar, BorderLayout.SOUTH);

    FileTransferClientUI.setEtchedBorder(this);
  }

  /**
   * @return a barra
   */
  private JToolBar buildToolBar() {
    final FileTransferClient ftpClient = getApplication();

    final JToolBar toolBar = new JToolBar(JToolBar.HORIZONTAL);
    toolBar.setFloatable(false);

    toolBar.add(new HomeLocalAction(ftpClient));
    toolBar.addSeparator();
    toolBar.add(new ChangeLocalDirectoryAction(ftpClient));
    toolBar.add(new UpLocalDirectoryAction(ftpClient));
    toolBar.add(new JPanel());
    toolBar.add(new RefreshLocalAction(ftpClient));

    return toolBar;
  }

  /**
   * Construo do painel de informao.
   * 
   * @param serverName o nome do servidor CSBASE
   * @return o painel
   */
  private JPanel buildInfoPanel(final String serverName) {
    localDirText.setEditable(false);
    userText.setEditable(false);
    projectText.setEditable(false);

    final JLabel localDirLabel = new JLabel();
    localDirLabel.setText(getString("local.directory.label"));

    final JLabel userLabel = new JLabel();
    userLabel.setText(getString("local.user.label"));
    final User user = User.getLoggedUser();
    userText.setText(user.getLogin());

    final JLabel projectLabel = new JLabel();
    projectLabel.setText(getString("local.project.label"));
    final CommonClientProject project =
      getApplication().getApplicationProject();
    projectText.setText(project.getName());

    final JPanel infoPanel = new JPanel();
    final JComponent[][] cmps =
      new JComponent[][] { { userLabel, userText },
          { projectLabel, projectText }, { localDirLabel, localDirText }, };
    GUIUtils.mountBasicGridPanel(infoPanel, cmps);

    final JLabel serverLabel = new JLabel();
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final Image desktopImage = desktop.getFrameImage();
    if (desktopImage != null) {
      final ImageIcon icon = new ImageIcon(desktopImage);
      serverLabel.setIcon(icon);
      serverLabel.setText(serverName);
    }

    final JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());
    panel.add(serverLabel, new GBC(0, 0).none().west().insets(0, 0, 20, 0));
    panel.add(infoPanel, new GBC(0, 1).both());
    return panel;
  }

  /**
   * @return the localDirectory
   */
  public final ClientProjectFile getLocalDirectory() {
    return localDirectory;
  }

  /**
   * Ajuste do diretrio local.
   * 
   * @param localDirectory diretrio local
   */
  public void setLocalDirectory(final ClientProjectFile localDirectory) {
    if (localDirectory == null) {
      final DesktopFrame mainFrame = DesktopFrame.getInstance();
      final CommonClientProject project = mainFrame.getProject();
      this.localDirectory = project.getRoot();
    }
    else {
      this.localDirectory = localDirectory;
    }

    final String text = FileTransferClientUI.pathToString(this.localDirectory);
    if (FileTransferClientUI.isEmptyString(text)) {
      localDirText.setText("/");
    }
    else {
      localDirText.setText(text);
    }

    refresh();
  }

  /**
   * Atualizao do painel
   * 
   */
  final public void refresh() {
    final LocalTableModel model = (LocalTableModel) localTable.getModel();
    final ArrayList<ClientProjectFile> list =
      new ArrayList<ClientProjectFile>();
    try {
      if (localDirectory == null) {
        return;
      }
      final ClientProjectFile[] children = localDirectory.getChildren();
      for (ClientProjectFile file : children) {
        if (filterAllows(file)) {
          list.add(file);
        }
      }
    }
    catch (Exception e) {
      // No faz nada e a lista fica vazia!
    }
    model.setFileList(list);
  }

  /**
   * Indica se arquivo deve aparecer na lista com base nos filtros existentes.
   * 
   * @param file arquivo
   * @return indicativo
   */
  private boolean filterAllows(ClientProjectFile file) {
    final String name = file.getName();
    if (dotFiltered) {
      if (name.startsWith(".")) {
        return false;
      }
    }
    return true;
  }

  /**
   * Construtor
   * 
   * @param application aplicao.
   * @param serverName o nome do servidor que ser usado no ttulo.
   */
  public FileTransferClientLocalPanel(final FileTransferClient application,
    String serverName) {
    super(application);
    mountPanel(serverName);
    setLocalDirectory(null);
    addProjectTreeListener();
  }

  /**
   * Adiciona listener na rvore do projeto para que o painel seja atualizado
   * quando houver mudana de arquivos, diretrios etc.
   */
  private void addProjectTreeListener() {
    final ProjectTree tree = DesktopFrame.getInstance().getTree();
    tree.addTreeModelListener(new TreeModelListener() {
      @Override
      public void treeNodesChanged(TreeModelEvent e) {
        refresh();
      }

      @Override
      public void treeNodesInserted(TreeModelEvent e) {
        refresh();
      }

      @Override
      public void treeNodesRemoved(TreeModelEvent e) {
        refresh();
      }

      @Override
      public void treeStructureChanged(TreeModelEvent e) {
        refresh();
      }
    });
  }

}
