package csbase.client.applications.flowapplication.filters;

import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;

import tecgraf.vix.TypeMessage;
import csbase.client.applications.flowapplication.Workspace;
import csbase.client.applications.flowapplication.messages.CheckClipboardFlavorMessage;
import csbase.client.applications.flowapplication.messages.ClipboardIsEmptyMessage;
import csbase.client.applications.flowapplication.messages.CopyMessage;
import csbase.client.applications.flowapplication.messages.PasteMessage;

/**
 * Filtro de copy/paste
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class CopyAndPasteFilter extends WorkspaceFilter implements
  ClipboardOwner {

  /**
   * Clipboard
   */
  private final Clipboard clipboard;

  /**
   * Construtor
   * 
   * @param clipboard a rea de transferncia.
   * @param workspace workspace
   */
  public CopyAndPasteFilter(final Clipboard clipboard, final Workspace workspace) {
    super(workspace);
    if (clipboard == null) {
      throw new IllegalArgumentException("O parmetro clipboard est nulo.");
    }
    this.clipboard = clipboard;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void lostOwnership(final Clipboard clpbrd, final Transferable contents) {
    // Ignora o evento.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean msgHandlerVS(final TypeMessage message) {
    if (message instanceof CopyMessage) {
      handleCopyMessage((CopyMessage) message);
      return true;
    }
    if (message instanceof PasteMessage) {
      handlePasteMessage((PasteMessage) message);
      return true;
    }
    if (message instanceof ClipboardIsEmptyMessage) {
      handleClipboardIsEmptyMessage((ClipboardIsEmptyMessage) message);
      return true;
    }
    if (message instanceof CheckClipboardFlavorMessage) {
      handleCheckClipboardFlavorMessage((CheckClipboardFlavorMessage) message);
      return true;
    }
    return super.msgHandlerVS(message);
  }

  /**
   * Tratador de mensagem: checagem de tipo.
   * 
   * @param message a mensagem.
   */
  private void handleCheckClipboardFlavorMessage(
    CheckClipboardFlavorMessage message) {
    Transferable contents = this.clipboard.getContents(this);
    if (contents != null) {
      DataFlavor flavor = message.getFlavor();
      boolean isFlavorSupported = contents.isDataFlavorSupported(flavor);
      message.setSupportsDataFlavor(isFlavorSupported);
    }
    else {
      message.setSupportsDataFlavor(false);
    }
  }

  /**
   * Tratador de mensagem: vazio
   * 
   * @param message mensagem
   */
  private void handleClipboardIsEmptyMessage(
    final ClipboardIsEmptyMessage message) {
    message.setClipboradIsEmpty(this.clipboard.getContents(this) != null);
  }

  /**
   * Tratador de mensagem: copy
   * 
   * @param message mensagem
   */
  private void handleCopyMessage(final CopyMessage message) {
    this.clipboard.setContents(message.getTransferable(), this);
  }

  /**
   * Tratador de mensagem: paste
   * 
   * @param message mensagem
   */
  private void handlePasteMessage(final PasteMessage message) {
    message.setTransferable(this.clipboard.getContents(this));
  }
}
