package csbase.client.applications.flowapplication.graph;

import java.awt.Color;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import csbase.client.applicationmanager.ApplicationException;
import csbase.exception.ParseException;

/**
 * Gerente de cores dos tipos de arquivo no editor de fluxo.
 * 
 * @author Tecgraf
 */
public final class FileTypeColorManager {

  /**
   * Cor padro.
   */
  private static final Color UNKNOWN_FILE_TYPE_COLOR = Color.BLACK;

  /**
   * A instncia compartilhada.
   */
  private static FileTypeColorManager instance;

  /**
   * Mapeia os tipos de arquivo para as respectivas cores.
   */
  private final Map<String, Color> colorsByFileType;

  /**
   * Construtor.
   * 
   * @param fileTypes A lista de tipos de arquivo.
   * @param rgbs A lista das cores em rgb.
   * @throws ParseException caso alguma cor esteja mal definida
   * @throws ApplicationException caso as listas estejam inconsistentes
   */
  private FileTypeColorManager(final List<String> fileTypes,
    final List<String> rgbs) throws ParseException, ApplicationException {
    this.colorsByFileType = new HashMap<String, Color>();
    if (fileTypes == null) {
      throw new ApplicationException("Parmetro fileTypes no pode ser nulo!");
    }
    if (rgbs == null) {
      throw new ApplicationException("Parmetro rgbs no pode ser nulo!");
    }

    if (fileTypes.size() != rgbs.size()) {
      throw new ApplicationException("Configurao de cores invlida!");
    }

    for (int i = 0; i < fileTypes.size(); i++) {
      final String fileType = fileTypes.get(i);
      final Color color = stringToColor(rgbs.get(i));
      this.colorsByFileType.put(fileType, color);
    }
  }

  /**
   * Retorna a instncia compartilhada desta classe.
   * 
   * @return a instncia.
   */
  public static FileTypeColorManager getInstance() {
    return instance;
  }

  /**
   * Converte texto de cor (#RRGGBB) em uma instncia de <code>Color</code>.
   * 
   * @param txt string para converso (#RRGGBB)
   * @return a cor
   * @throws ParseException caso no seja possvel converter o texto em uma cor
   *         vlida.
   */
  private Color stringToColor(final String txt) throws ParseException {
    if (txt == null) {
      throw new ParseException("Texto que representa a cor est nulo!");
    }
    try {
      final String str = txt.trim();
      final String c = (str.charAt(0) == '#') ? str.substring(1) : str;
      final int colvalue = Integer.parseInt(c, 16);
      return new Color(colvalue);
    }
    catch (final NumberFormatException e) {
      throw new ParseException("Erro ao converter cor: ", e);
    }
  }

  /**
   * Cria uma instncia.
   * 
   * @param fileTypes A lista de tipos de arquivo.
   * @param rgbs A lista das cores em rgb.
   * @throws ParseException caso alguma cor esteja mal definida
   * @throws ApplicationException caso as listas estejam inconsistentes
   */
  public static void createInstance(final List<String> fileTypes,
    final List<String> rgbs) throws ParseException, ApplicationException {
    if (instance == null) {
      instance = new FileTypeColorManager(fileTypes, rgbs);
    }
  }

  /**
   * Consulta cor de um tipo de arquivo
   * 
   * @param fileType o tipo do arquivo
   * @return a cor
   */
  public Color getColor(final String fileType) {
    Color color = this.colorsByFileType.get(fileType);
    if (color == null) {
      color = UNKNOWN_FILE_TYPE_COLOR;
    }
    return color;
  }

}
