package csbase.client.applications.flowapplication.graph;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.geom.Rectangle2D;

import tecgraf.vix.TypeVS;
import tecgraf.vix.VO;

/**
 * Vo de texto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class GraphString extends VO {
  private static final int DEFAULT_BOTTOM_PADDING = 1;
  private static final Color DEFAULT_COLOR = Color.BLACK;
  private static final Font DEFAULT_FONT = new Font(Font.SANS_SERIF, Font.BOLD,
    12);
  private static final int DEFAULT_LEFT_PADDING = 1;
  private static final int DEFAULT_RIGHT_PADDING = 1;
  private static final int DEFAULT_TOP_PADDING = 1;
  private static final String ETC = "...";
  private static final double NIL = Double.NEGATIVE_INFINITY;
  private double bottomPadding;
  private double centerX;
  private double centerY;
  private Color color;
  private Font font;
  private double height;
  private double leftPadding;
  private double maxWidth;
  private double rightPadding;
  private String text;
  private double topPadding;
  private String visualText;
  private double width;
  private double x0;
  private double y0;

  /**
   * Construtor
   * 
   * @param vs VS
   * @param text texto
   */
  public GraphString(final TypeVS vs, final String text) {
    this(vs, text, DEFAULT_FONT);
  }

  /**
   * Construtor
   * 
   * @param vs VS
   * @param text texto
   * @param font fonte
   */
  public GraphString(final TypeVS vs, final String text, final Font font) {
    changeVS(null, vs);
    this.visualText = this.text = text;
    this.bottomPadding = DEFAULT_BOTTOM_PADDING;
    this.color = DEFAULT_COLOR;
    this.font = (font != null ? font : DEFAULT_FONT);
    this.leftPadding = DEFAULT_LEFT_PADDING;
    this.maxWidth = Double.POSITIVE_INFINITY;
    this.rightPadding = DEFAULT_RIGHT_PADDING;
    this.topPadding = DEFAULT_TOP_PADDING;
    computeMeasures();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackRepaint(final Graphics2D g) {
    if (this.x0 == NIL || this.y0 == NIL || this.width == NIL
      || this.height == NIL) {
      computeMeasures();
    }
    g.setColor(this.color);
    setGraphicsFont(g);
    g.drawString(toString(), (float) this.x0, (float) this.y0);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(final Object object) {
    if (object == null) {
      return false;
    }
    if (!(object instanceof GraphString)) {
      return false;
    }
    final GraphString string = (GraphString) object;
    return this.text.equals(string.text);
  }

  public double getBottomPadding() {
    return this.bottomPadding;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Rectangle2D getBounds2D() {
    final Rectangle2D bounds = new Rectangle();
    if (this.x0 == NIL || this.y0 == NIL || this.width == NIL
      || this.height == NIL) {
      if (!computeMeasures()) {
        bounds.setFrame(0, 0, 0, 0);
        return bounds;
      }
    }

    double w = Math.max(width, 0);
    double h = Math.max(height, 0);
    bounds.setFrameFromCenter(centerX, centerY, centerX + w / 2
      + this.rightPadding + this.leftPadding, centerY - h / 2
      - this.bottomPadding - this.topPadding);
    return bounds;
  }

  public double getLeftPadding() {
    return this.leftPadding;
  }

  public double getRightPadding() {
    return this.rightPadding;
  }

  public String getText() {
    return this.text;
  }

  public double getTopPadding() {
    return this.topPadding;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return this.text.hashCode();
  }

  public void setBottomPadding(final double bottomPadding) {
    this.bottomPadding = bottomPadding;
  }

  public void setColor(final Color color) {
    this.color = (color != null ? color : DEFAULT_COLOR);
  }

  public void setFont(final Font font) {
    this.font = (font != null ? font : DEFAULT_FONT);
    computeMeasures();
  }

  public Font getFont() {
    return this.font;
  }

  public void setLeftPadding(final double leftPadding) {
    this.leftPadding = leftPadding;
  }

  public void setLocation(final double centerX, final double centerY) {
    this.centerX = centerX;
    this.centerY = centerY;
    computeMeasures();
  }

  public void setMaxWidth(final double maxWidth) {
    if (maxWidth > 0) {
      this.maxWidth = maxWidth - this.leftPadding - this.rightPadding;
    }
    else {
      this.maxWidth = Double.POSITIVE_INFINITY;
    }
    computeMeasures();
  }

  public void setRightPadding(final double rightPadding) {
    this.rightPadding = rightPadding;
  }

  public void setText(final String text) {
    this.text = text;
  }

  public void setTopPadding(final double topPadding) {
    this.topPadding = topPadding;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.visualText;
  }

  private boolean computeMeasures() {
    this.visualText = this.text;
    final Graphics2D g2D = this.vs.getGraphics2D();
    if (g2D != null) {
      setGraphicsFont(g2D);
      FontMetrics fontMetrics = g2D.getFontMetrics(this.font);
      Rectangle2D bounds = fontMetrics.getStringBounds(this.visualText, g2D);
      this.width = bounds.getWidth();
      double maxTextWidth = this.maxWidth;
      if (this.width > maxTextWidth) {
        Rectangle2D etcBounds = fontMetrics.getStringBounds(ETC, g2D);
        maxTextWidth = maxTextWidth - etcBounds.getWidth();
        while (this.width > maxTextWidth) {
          this.visualText =
            this.visualText.substring(0, this.visualText.length() - 1);
          bounds = fontMetrics.getStringBounds(this.visualText, g2D);
          this.width = bounds.getWidth();
        }
        this.visualText += ETC;
        this.width += etcBounds.getWidth();
      }
      this.height = bounds.getHeight();
      this.x0 = this.centerX;
      if (this.width > 0) {
        this.x0 -= this.width / 2;
      }
      this.y0 = this.centerY;
      if (this.height > 0) {
        this.y0 += this.height / 4;
      }
      return true;
    }
    this.x0 = this.y0 = this.width = this.height = NIL;
    return false;
  }

  private void setGraphicsFont(final Graphics2D g) {
    g.setFont(this.font);
    g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,
      RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
  }
}
