package csbase.client.applications.imageviewer;

import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.InputStream;
import java.io.OutputStream;

import javax.imageio.ImageIO;

import csbase.client.applications.ApplicationTask;
import csbase.logic.ClientFile;

/**
 * Classe de I/O para aplicao.
 * 
 * @author Tecgraf
 */
public class ImageViewerIO {

  /**
   * Texto de internacionalizao.
   * 
   * @param application aplicao.
   * @param tag tag
   * @return texto
   */
  static private String getString(final ImageViewer application,
    final String tag) {
    final String prefix = ImageViewerIO.class.getSimpleName();
    return application.getString(prefix + "." + tag);
  }

  /**
   * Execuo em task da gravao.
   * 
   * @param application aplicao.
   * @param image imagem
   * @param file arquivo
   * @throws Exception exceo
   */
  static public void writeTask(final ImageViewer application,
    final BufferedImage image, final ClientFile file) throws Exception {
    final OutputStream stream = file.getOutputStream();
    final String type = file.getType();

    final String msg = getString(application, "write.task.msg");
    final ApplicationTask<Void> task =
      new ApplicationTask<Void>(application, msg) {
        @Override
        protected void performTask() throws Exception {
          ImageIO.write(image, type, stream);
          stream.close();
        }
      };

    final boolean executed = task.execute();
    if (!executed) {
      final Exception error = task.getError();
      if (error != null) {
        throw error;
      }
      final String tag = "write.error.msg";
      final String err = getString(application, tag);
      throw new Exception(err);
    }
  }

  /**
   * Cria uma Imagem a partir de um ClientProjectFile.
   * 
   * @param app aplicao.
   * @param file arquivo com a imagem a ser aberta.
   * @return a imagem gerada
   * @throws Exception em caso de erro.
   */
  static public Image readTask(final ImageViewer app, final ClientFile file)
    throws Exception {
    if (file == null) {
      final String tag = "read.null.file.error.msg";
      final String err = getString(app, tag);
      throw new Exception(err);
    }
    if (file.isDirectory()) {
      final String tag = "read.directory.file.error.msg";
      final String err = getString(app, tag);
      throw new Exception(err);
    }

    final InputStream stream = file.getInputStream();
    final String msg = getString(app, "read.task.msg");
    final ApplicationTask<BufferedImage> task =
      new ApplicationTask<BufferedImage>(app, msg) {
        @Override
        public void performTask() throws Exception {
          final BufferedImage read = ImageIO.read(stream);
          setResult(read);
          stream.close();
        }
      };

    final boolean executed = task.execute();
    if (!executed) {
      String err = getString(app, "read.error.msg");
      final Exception error = task.getError();
      if (error != null) {
        err = err + "\n" + error.getMessage();
      }
      throw new Exception(err);
    }

    final Image img = task.getResult();
    if (img == null) {
      String err = getString(app, "read.error.msg");
      throw new Exception(err);
    }
    return img;
  }
}
