package csbase.client.applications.jobmonitor.filters;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.jobmonitor.JobMonitor;
import csbase.client.applications.jobmonitor.rowmodel.JobInfoRow;
import csbase.client.facilities.configurabletable.table.ColumnState;
import csbase.client.facilities.configurabletable.table.ConfigurableTable;

/**
 * Painel que encapsula o filtro utilizado na aplicao que monitora jobs.
 * 
 * @author Tecgraf
 */
public class FilteredTablePanel extends JPanel {

  /** Combo box que exibe as colunas usadas na filtragem. */
  private JComboBox columnsComboBox;

  /** Check box que define se a filtragem ser pela palavra completa. */
  private JCheckBox wholeWordCheckBox;

  /** Campo em que o usurio digita a string que ser usada na filtragem. */
  private JTextField filterField;

  /** Referncia para a aplicao. */
  private JobMonitor app;

  /** Tabela a ser exibida. */
  private ConfigurableTable<JobInfoRow> table;

  /**
   * Construtor padro.
   * 
   * @param app - referncia para a aplicao.
   * @param table - tabela a ser filtrada.
   */
  public FilteredTablePanel(JobMonitor app, ConfigurableTable<JobInfoRow> table) {
    super(new GridBagLayout());
    this.app = app;
    this.table = table;

    columnsComboBox = createComboBox();
    wholeWordCheckBox = createWholeWordCheckBox();
    filterField = createFilterField();

    JLabel filterLabel = new JLabel(app.getString("label.filter"));

    JPanel filterPanel = new JPanel(new GridBagLayout());

    filterPanel.add(filterLabel, new GBC(0, 0).west());
    filterPanel.add(filterField, new GBC(1, 0).left(10).horizontal());
    filterPanel.add(columnsComboBox, new GBC(2, 0).left(10).none());
    filterPanel.add(wholeWordCheckBox, new GBC(3, 0).left(10).right(50).none());

    add(filterPanel, new GBC(0, 0).horizontal());
    add(new JScrollPane(table), new GBC(0, 1).both().top(10));
  }

  /**
   * Cria o combo box que define qual coluna ser considerada na filtragem da
   * tabela.
   * 
   * @return combo box usando na filtragem.
   */
  public JComboBox createComboBox() {
    Vector<String> items = new Vector<String>();
    items.add(app.getString("all.columns"));

    for (ColumnState columnState : table.getColumnsState()) {
      items.add(columnState.getColumnId());
    }

    JComboBox comboBox = new JComboBox(items);
    comboBox.setMaximumRowCount(items.size());
    comboBox.addActionListener(new ActionListener() {

      @Override
      public void actionPerformed(ActionEvent e) {
        changeTableFilter();
      }
    });

    return comboBox;
  }

  /**
   * Cria o campo em que o usurio digita a string usada na filtragem.
   * 
   * @return campo em que o usurio digita a string usada na filtragem.
   */
  private JTextField createFilterField() {
    JTextField field = new JTextField();

    field.getDocument().addDocumentListener(new DocumentListener() {

      @Override
      public void removeUpdate(DocumentEvent e) {
        changedUpdate(e);
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        changedUpdate(e);
      }

      @Override
      public void changedUpdate(DocumentEvent e) {
        changeTableFilter();
      }

    });

    return field;
  }

  /**
   * Cria o check box que define se o padro deve casar com a palavra inteira.
   * 
   * @return check box que define se o padro deve casar com a palavra inteira.
   */
  private JCheckBox createWholeWordCheckBox() {
    JCheckBox checkBox = new JCheckBox(app.getString("label.whole.word"));

    checkBox.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        changeTableFilter();
      }
    });

    return checkBox;
  }

  /**
   * Modifica o filtro da tabela.
   */
  private void changeTableFilter() {
    try {
      String regex = filterField.getText();
      Pattern newPattern = Pattern.compile(regex);

      int selectedIndex = columnsComboBox.getSelectedIndex();

      List<String> columnsToFilter = new LinkedList<String>();

      JobInfoFilter filter = (JobInfoFilter) table.getFilter();

      if (selectedIndex == 0) {
        for (ColumnState columnState : table.getColumnsState()) {
          columnsToFilter.add(columnState.getColumnId());
        }

      }
      else {
        columnsToFilter.add(columnsComboBox.getSelectedItem().toString());
      }

      filter.setPattern(newPattern);
      filter.setColumnsToFilter(columnsToFilter);
      filter.setWholeWord(wholeWordCheckBox.isSelected());

      table.updateRows();

    }
    catch (PatternSyntaxException exception) {
      // faz nada
    }

  }
}
