package csbase.client.applications.projectsmanager;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.util.ArrayList;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.MenuButton;
import tecgraf.javautils.gui.MenuButton.PopupPosition;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationProject;
import csbase.client.applications.projectsmanager.actions.AllocateAreaAction;
import csbase.client.applications.projectsmanager.actions.AppendUsersAction;
import csbase.client.applications.projectsmanager.actions.ChangeServerOwnerProjectDialogAction;
import csbase.client.applications.projectsmanager.actions.ChangeStatusAllocateAction;
import csbase.client.applications.projectsmanager.actions.ChangeStatusLockAction;
import csbase.client.applications.projectsmanager.actions.ChangeStatusNoAllocationAction;
import csbase.client.applications.projectsmanager.actions.DeallocateAreaAction;
import csbase.client.applications.projectsmanager.actions.DisplayFilterAction;
import csbase.client.applications.projectsmanager.actions.FillProjectsTableAction;
import csbase.client.applications.projectsmanager.actions.OpenMailDialogAction;
import csbase.client.applications.projectsmanager.actions.OpenProjectAreaOccupationDialogAction;
import csbase.client.applications.projectsmanager.actions.OpenProjectCommentsDialogAction;
import csbase.client.applications.projectsmanager.actions.OpenProjectCreationDialogAction;
import csbase.client.applications.projectsmanager.actions.OpenProjectOccupationDialogAction;
import csbase.client.applications.projectsmanager.actions.RefreshProjectsAction;
import csbase.client.applications.projectsmanager.actions.RemoveProjectAction;
import csbase.client.applications.projectsmanager.actions.RemoveUsersAction;
import csbase.client.applications.projectsmanager.actions.ReplicateSharingDataAction;
import csbase.client.applications.projectsmanager.actions.SearchFilesDialogAction;
import csbase.client.applications.projectsmanager.actions.SetProjectPrivateAction;
import csbase.client.applications.projectsmanager.actions.SetProjectPublicAction;
import csbase.client.applications.projectsmanager.actions.SetProjectSharedAction;
import csbase.client.applications.projectsmanager.actions.ShowProjectHistoryAction;
import csbase.client.applications.projectsmanager.actions.UserRemovalFromProjectAction;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.models.ProjectsManagerScope;
import csbase.client.applications.projectsmanager.panels.FilterPanel;
import csbase.client.applications.projectsmanager.panels.InfoPanel;
import csbase.client.applications.projectsmanager.panels.TablePanel;
import csbase.client.applications.projectsmanager.proxy.RetrieveUsersTask;
import csbase.client.desktop.RemoteTask;
import csbase.client.project.tasks.CheckAreaReservedTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommonClientProject;
import csbase.logic.CommonProjectInfo;
import csbase.logic.User;
import csbase.logic.UserOutline;

/**
 * Classe que representa a aplicao ProjectsManager.
 *
 * @author jnlopes
 */
public class ProjectsManager extends ApplicationProject implements Observer {

  /**
   * Painel que apresenta a tabela onde so listados os projetos.
   */
  private TablePanel tablePanel;

  /**
   * Painel que apresenta informaes do(s) projeto(s) selecionado(s).
   */
  private InfoPanel infoPanel;

  /**
   * Painel de filtragem dos projetos listados na tabela.
   */
  private FilterPanel filterPanel;

  /**
   * Ao de preencher inicialmente a tabela com todos os projetos do sistema
   * sem carregar o espao ocupado por eles.
   */
  private FillProjectsTableAction updateProjectsAction;

  /**
   * Ao da toolbar, de preencher a tabela com todos os projetos do sistema,
   * carregando o espao ocupado por eles.
   */
  private FillProjectsTableAction updateProjectsWithDiskUsageAction;

  /**
   * Ao de visualizar/esconder o painel de filtros.
   */
  private DisplayFilterAction displayFilterAction;

  /**
   * Toolbar
   */
  final private JToolBar toolbar = new JToolBar(SwingConstants.HORIZONTAL);

  /**
   * Lista dos projetos existentes.
   */
  final private List<ProjectsManagerData> allProjectsList =
    new ArrayList<ProjectsManagerData>();

  /**
   * Lista de projetos selecionados.
   */
  private List<ProjectsManagerData> selectedProjectsList;

  /**
   * Boto que exibe/oculta os filtros.
   */
  private JToggleButton filterButton;

  /**
   * Boto da toolbar que calcula espao ocupado por projetos.
   */
  private JButton projectsSpaceButton;

  /**
   * Boto para definir projeto como tendo sua rea alocada.
   */
  private JButton allocateButton;

  /**
   * Boto para definir projeto como tendo sua rea desalocada.
   */
  private JButton deallocateButton;

  /**
   * MenuButton para definio de projeto pblico RO ou RW.
   */
  private MenuButton setPublicMenuButton;

  /**
   * Boto para definir projeto como privado.
   */
  private JButton setPrivateButton;

  /**
   * Boto para definir projeto como compartilhado.
   */
  private JButton overwriteUsersButton;

  /**
   * Boto para cpia de esquema de compartilhamento entre projetos.
   */
  private JButton replicateSharedButton;

  /**
   * Boto para remoo de projeto.
   */
  private JButton projectRemovalButton;

  /**
   * Boto onde usurio se remove do projeto.
   */
  private JButton userRemovalButton;

  /**
   * Boto para ler o histrico do projeto selecionado.
   */
  private JButton showHistoryButton;

  /**
   * Boto para adicionar comentrios no histrico de projetos.
   */
  private JButton writeCommentsButton;

  /**
   * Boto para enviar mails referentes a projetos para usurios.
   */
  private JButton mailButton;

  /**
   * Ao que calcula espao ocupado por projetos.
   */
  private RefreshProjectsAction multipleProjectSpaceAction;

  /**
   * Ao para definir projeto como tendo sua rea alocada.
   */
  private AllocateAreaAction allocateAreaAction;

  /**
   * Ao para definir projeto como tendo sua rea desalocada.
   */
  private DeallocateAreaAction deallocateAreaAction;

  /**
   * Ao para definir projeto como pblico RO.
   */
  private SetProjectPublicAction setPublicROProjectSharingStatsAction;

  /**
   * Ao para definir projeto como pblico RW.
   */
  private SetProjectPublicAction setPublicRWProjectSharingStatsAction;

  /**
   * Ao para definir projeto como privado.
   */
  private SetProjectPrivateAction setPrivateProjectSharingStatsAction;

  /**
   * Ao para abrir o dilogo de seleo de usurios compartilhados nos
   * projetos selecionados.
   */
  private SetProjectSharedAction setProjectSharedAction;

  /**
   * Ao para cpia de esquema de compartilhamento entre projetos.
   */
  private ReplicateSharingDataAction replicateSharingDataAction;

  /**
   * Ao para abrir um dilogo que mostra o histrico dos projetos.
   */
  private ShowProjectHistoryAction showProjectHistoryDialogAction;

  /**
   * Ao para abrir o dilogo de adio de comentrios a histrico de projetos.
   */
  private OpenProjectCommentsDialogAction openProjectCommentsDialogAction;

  /**
   * Ao que abre dilogo de envio de email referente a projetos.
   */
  private OpenMailDialogAction openMailDialogAction;

  /**
   * Ao que abre dilogo para busca de arquivos referente a projetos.
   */
  private SearchFilesDialogAction searchFilesDialogAction;

  /**
   * Ao para abrir o dilogo de criao de projetos.
   */
  private OpenProjectCreationDialogAction openProjectCreationDialogAction;

  /**
   * Ao para abrir o dilogo de alterao do nome do dono do projeto.
   */
  private ChangeServerOwnerProjectDialogAction changeServerOwnerProjectDialogAction;

  /**
   * Ao para remoo de projeto.
   */
  private RemoveProjectAction removeProjectAction;

  /**
   * Ao para usurio se remover de projetos.
   */
  private UserRemovalFromProjectAction userRemovalFromProjectAction;

  /**
   * Ao para abrir dilogo de informaes gerais sobre ocupao do espao de
   * projetos.
   */
  private OpenProjectOccupationDialogAction openProjectOccupationDialogAction;

  /**
   * Ao para abrir dilogo de informaes gerais sobre ocupao do espao de
   * projetos.
   */
  private OpenProjectAreaOccupationDialogAction openProjectAreaOccupationDialogAction;

  /**
   * Ao para adio de usurios a projetos com permisso de leitura.
   */
  private AppendUsersAction appendUsersROAction;

  /**
   * Ao para adio de usurios a projetos com permisso de leitura/escrita.
   */
  private AppendUsersAction appendUsersRWAction;

  /**
   * Ao para remoo de usurios compartilhados de projetos.
   */
  private RemoveUsersAction removeUsersAction;

  /**
   * Ao que abre dilogo para definio de projetos selecionados como
   * aguardando alocao.
   */
  private ChangeStatusLockAction changeStatusLockAction;

  /**
   * Ao que abre dilogo para definio de projetos selecionados como contendo
   * rea alocada.
   */
  private ChangeStatusAllocateAction changeStatusAllocateAction;

  /**
   * Ao que abre dilogo para definio de projetos selecionados como no
   * contendo rea alocada.
   */
  private ChangeStatusNoAllocationAction changeStatusNoAllocationAction;

  /**
   * Item que calcula espao ocupado por projetos.
   */
  private JMenuItem multipleProjectSpaceItem;

  /**
   * Item para criar projetos.
   */
  private JMenuItem createProjectItem;

  /**
   * Item para definir projeto como tendo sua rea alocada.
   */
  private JMenuItem allocateAreaItem;

  /**
   * Item para definir projeto como tendo sua rea desalocada.
   */
  private JMenuItem deallocateAreaItem;

  /**
   * Item que define projetos selecionados como aguarando alocao de rea.
   */
  private JMenuItem setLockedStatusItem;

  /**
   * Item que define projetos selecionados como contendo rea alocada.
   */
  private JMenuItem setAllocatedStatusItem;

  /**
   * Item que define projetos selecionados como no contendo rea alocada.
   */
  private JMenuItem setNoAllocationStatusItem;

  /**
   * Item para definir projeto como pblico RO.
   */
  private JMenuItem setPublicROProjectSharingStatsItem;

  /**
   * Item para definir projeto como pblico RW.
   */
  private JMenuItem setPublicRWProjectSharingStatsItem;

  /**
   * Item para definir projeto como privado.
   */
  private JMenuItem setPrivateProjectSharingStatsItem;

  /**
   * Item para definir projeto como compartilhado.
   */
  private JMenuItem overwriteUsersItem;

  /**
   * Inclui usurios a projetos com acesso RO.
   */
  private JMenuItem appendUsersROItem;

  /**
   * Inclui usurios a projetos com acesso RW.
   */
  private JMenuItem appendUsersRWItem;

  /**
   * Remove usurios de projetos compartilhados.
   */
  private JMenuItem removeUsersItem;

  /**
   * Item de menu para cpia de esquema de compartilhamento entre projetos.
   */
  private JMenuItem replicateSharedItem;

  /**
   * Item para remoo de projeto.
   */
  private JMenuItem projectRemovalItem;

  /**
   * Item para usurio se remover de projetos.
   */
  private JMenuItem userRemovalFromProjectItem;

  /**
   * Item para ler o histrico de um projeto selecionado.
   */
  private JMenuItem showHistoryItem;

  /**
   * Item para adicionar comentrios no histrico de projetos.
   */
  private JMenuItem writeCommentsItem;

  /**
   * Item para enviar mails referentes a projetos para usurios.
   */
  private JMenuItem mailItem;

  /**
   * Item para pesquisar arquivos referentes a projetos para usurios.
   */
  private JMenuItem searchItem;

  /**
   * Item para alterar o nome do servidor dono do projeto.
   */
  private JMenuItem changeServerOwnerProjectItem;

  /**
   * Objeto que observa o servio de administrao de projetos para criao e
   * remoo de projetos..
   */
  private ProjectCreationRemovalObserver projectsCreationRemovalObserver;

  /**
   * Objeto que observa o servio de administrao de projetos para alteraes
   * no compartilhamento de projetos.
   */
  private AbstractProjectsManagerObserver projectSharingStatusObserver;

  /**
   * Boolean que indica se as informaes de reserva de rea devem ser exibidas
   * ou no.
   */
  private boolean areaReserved;

  /**
   * Construtor padro.
   *
   * @param id o identificador da aplicao.
   * @throws Exception Erro na obteno do boolean de ProjectService que indica
   *         se o sistema utiliza reserva de rea.
   */
  public ProjectsManager(final String id) throws Exception {
    super(id);

    // areaReserved = getBooleanSpecificProperty("reserveArea", false);
    defineReservedAreaUsage();
    initComponents();
    initActions();
    initToolbar();
    initMenu();
    enableItems(false);
    ensembleFrame();

    refreshProjectsTable(true);
    selectCurrentProject();

    final ApplicationFrame frame = getApplicationFrame();
    frame.setPreferredSize(new Dimension(940, 680));
    frame.pack();
    frame.update(null);

    projectsCreationRemovalObserver = new ProjectCreationRemovalObserver(this);
    projectSharingStatusObserver = new ProjectSharingStatusObserver(this);
  }

  /**
   * Obtm o boolean que indica se o sistema usa ou no reserva de rea.
   *
   * TODO - Preencher texto internacionalizado das tags do mtodo abaixo.
   *
   * @throws Exception Erro na obteno do boolean de ProjectService que indica
   *         se o sistema utiliza reserva de rea.
   */
  private void defineReservedAreaUsage() throws Exception {

    final CheckAreaReservedTask task = new CheckAreaReservedTask();
    task.execute(getApplicationFrame(), ProjectsManagerUI
      .getString("ProjectsManager.area.reserved.title"), ProjectsManagerUI
      .getString("ProjectsManager.area.reserved.msg"));

    if (task.wasCancelled()) {
      areaReserved = false;
      final String title =
        ProjectsManagerUI
        .getString("ProjectsManager.area.reserved.cancelled.title");
      final String msg =
        ProjectsManagerUI
        .getString("ProjectsManager.area.reserved.cancelled.msg");
      StandardDialogs.showErrorDialog(getApplicationFrame(), title, msg);
      return;
    }

    if (task.getStatus() != true) {
      areaReserved = false;
      final Exception exception = task.getError();
      throw exception;
    }
    areaReserved = task.getResult();
  }

  /**
   * Seleciona imediatamente o projeto corrente (caso haja algum aberto) quando
   * esta aplicao for disparada. O projeto corrente (quando houver) ser
   * movido para o topo da tabela, de modo a garantir que alm de selecionado,
   * ele estar visvel ao usurio.
   */
  private void selectCurrentProject() {
    final CommonClientProject ccp = getApplicationProject();
    if (ccp == null) {
      return;
    }

    int counter = 0;
    final String projectName = ccp.getName();
    final List<ProjectsManagerData> filteredProjects =
      tablePanel.getVisibleProjects();
    for (ProjectsManagerData pmd : filteredProjects) {
      if (pmd.getProjectName().equals(projectName)) {
        final ProjectsManagerData aux = filteredProjects.get(0);
        filteredProjects.set(0, pmd);
        filteredProjects.set(counter, aux);
        tablePanel.setTableSelectionInterval(0, 0);
        return;
      }
      counter++;
    }
  }

  /**
   * Inicializa as aes da toolbar e do menu que variam em funo da seleo na
   * tabela de projetos.
   */
  private void initActions() {

    multipleProjectSpaceAction = new RefreshProjectsAction(this);
    openProjectCreationDialogAction = new OpenProjectCreationDialogAction(this);
    allocateAreaAction = new AllocateAreaAction(this);
    deallocateAreaAction = new DeallocateAreaAction(this);
    setPublicROProjectSharingStatsAction =
      new SetProjectPublicAction(this, true);
    setPublicRWProjectSharingStatsAction =
      new SetProjectPublicAction(this, false);
    setPrivateProjectSharingStatsAction = new SetProjectPrivateAction(this);
    setProjectSharedAction = new SetProjectSharedAction(this);

    replicateSharingDataAction = new ReplicateSharingDataAction(this);

    appendUsersROAction = new AppendUsersAction(this, true);
    appendUsersRWAction = new AppendUsersAction(this, false);
    removeUsersAction = new RemoveUsersAction(this);
    showProjectHistoryDialogAction = new ShowProjectHistoryAction(this);
    openProjectCommentsDialogAction = new OpenProjectCommentsDialogAction(this);
    openMailDialogAction = new OpenMailDialogAction(this);
    searchFilesDialogAction = new SearchFilesDialogAction(this);
    removeProjectAction = new RemoveProjectAction(this);
    userRemovalFromProjectAction = new UserRemovalFromProjectAction(this);
    openProjectOccupationDialogAction =
      new OpenProjectOccupationDialogAction(this);
    openProjectAreaOccupationDialogAction =
      new OpenProjectAreaOccupationDialogAction(this);

    this.changeStatusLockAction = new ChangeStatusLockAction(this);

    this.changeStatusAllocateAction = new ChangeStatusAllocateAction(this);

    this.changeStatusNoAllocationAction =
      new ChangeStatusNoAllocationAction(this);

    this.changeServerOwnerProjectDialogAction =
      new ChangeServerOwnerProjectDialogAction(this);
  }

  /**
   * Inicia componentes.
   */
  private void initComponents() {

    final ApplicationFrame frame = getApplicationFrame();

    displayFilterAction = new DisplayFilterAction(this);
    updateProjectsAction =
      new FillProjectsTableAction(this, isCurrentUserAdmin(), false);
    updateProjectsWithDiskUsageAction =
      new FillProjectsTableAction(this, isCurrentUserAdmin(), true);

    frame.setLayout(new GridBagLayout());
    filterPanel = new FilterPanel(this);
    tablePanel = new TablePanel(this);
    infoPanel = new InfoPanel(this);

    setPublicMenuButton =
      new MenuButton(ProjectsManagerScope.PUBLIC.getIcon(),
        PopupPosition.BOTTOM);
  }

  /**
   * Monta o frame principal da aplicao.
   */
  private void ensembleFrame() {
    final ApplicationFrame frame = getApplicationFrame();
    frame.add(toolbar, new GBC(0, 0).horizontal());
    frame.add(this.tablePanel, new GBC(0, 2).both());
    frame.add(infoPanel, new GBC(0, 3).horizontal());
  }

  /**
   * Exibe ou esconde os filtros da tabela.
   *
   * @param showFilters Indica se os filtros devem ser exibidos ou escondidos.
   */
  final public void displayFilters(final boolean showFilters) {
    final ApplicationFrame ap = this.getApplicationFrame();

    if (showFilters) {
      ap.add(filterPanel, new GBC(0, 1).horizontal());
      filterButton.setSelected(true);
    }
    else {
      filterPanel.emptyFiltersData();
      filterPanel.runFilters(true);
      ap.remove(filterPanel);
      filterButton.setSelected(false);
    }

    ap.validate();
    ap.repaint();
  }

  /**
   * Inicia componentes da toolbar.
   */
  private void initToolbar() {

    setPublicMenuButton.add(setPublicROProjectSharingStatsAction);
    setPublicMenuButton.add(setPublicRWProjectSharingStatsAction);

    toolbar.setFloatable(false);

    toolbar.add(openProjectCreationDialogAction);

    toolbar.addSeparator();

    toolbar.add(openProjectOccupationDialogAction);
    toolbar.add(openProjectAreaOccupationDialogAction);

    toolbar.addSeparator();

    toolbar.add(updateProjectsWithDiskUsageAction);
    projectsSpaceButton = toolbar.add(multipleProjectSpaceAction);

    filterButton = new JToggleButton(displayFilterAction);
    filterButton.setSelected(false);
    toolbar.add(filterButton);

    toolbar.addSeparator();

    if (isCurrentUserAdmin() && hasAreaReserved()) {
      toolbar.addSeparator();
      allocateButton = toolbar.add(allocateAreaAction);
      deallocateButton = toolbar.add(deallocateAreaAction);
    }

    toolbar.addSeparator();
    setPrivateButton = toolbar.add(setPrivateProjectSharingStatsAction);
    overwriteUsersButton = toolbar.add(setProjectSharedAction);
    toolbar.add(setPublicMenuButton);
    toolbar.addSeparator();

    replicateSharedButton = toolbar.add(replicateSharingDataAction);

    toolbar.addSeparator();

    /*
     * Inicialmente a toolbar contava com os botes de "Adicionar" e "Remover"
     * usurios de projetos compartilhados (como existem no menu). Retiramos
     * estes botes sem remov-los ou alterar sua lgica, pois caso eles voltem
     * a fazer parte da toolbar, bastar eliminar estas duas operaes abaixo.
     */

    projectRemovalButton = toolbar.add(removeProjectAction);
    toolbar.addSeparator();

    if (!isCurrentUserAdmin()) {
      userRemovalButton = toolbar.add(userRemovalFromProjectAction);
      toolbar.addSeparator();
    }

    showHistoryButton = toolbar.add(showProjectHistoryDialogAction);
    writeCommentsButton = toolbar.add(openProjectCommentsDialogAction);
    mailButton = toolbar.add(openMailDialogAction);
    toolbar.addSeparator();
    toolbar.add(searchFilesDialogAction);
  }

  /**
   * Inicializa os menus.
   */
  private void initMenu() {
    final ApplicationFrame frame = getApplicationFrame();

    final JMenu optionsMenu = initOptionsMenu();
    final JMenu projectsMenu = initProjectsMenu();

    final JMenuBar menu = new JMenuBar();
    menu.add(optionsMenu);
    menu.add(projectsMenu);

    frame.setJMenuBar(menu);
  }

  /**
   * Constri o menu de opes
   *
   * @return o menu
   */
  private JMenu initOptionsMenu() {
    final String generalText = getMenuString("general");
    final JMenu generalMenu = new JMenu(generalText);

    createProjectItem = new JMenuItem(openProjectCreationDialogAction);
    createProjectItem.setText(getString("create.project.text"));
    generalMenu.add(createProjectItem);
    generalMenu.addSeparator();

    final JMenu reportMenu = initReportMenu();
    generalMenu.add(reportMenu);

    generalMenu.add(updateProjectsWithDiskUsageAction);

    multipleProjectSpaceItem = new JMenuItem(multipleProjectSpaceAction);
    generalMenu.add(multipleProjectSpaceItem);

    final JMenuItem filterItem = new JMenuItem(displayFilterAction);
    filterItem.setText(getString("filter.text"));

    generalMenu.add(filterItem);
    generalMenu.addSeparator();
    generalMenu.add(new ApplicationExitAction(this));
    return generalMenu;
  }

  /**
   * Inicia menu com opes de ocupao.
   *
   * @return Menu com opes de ocupao.
   */
  private JMenu initReportMenu() {
    final String otherText = getMenuString("report");
    final JMenu occupationMenu = new JMenu(otherText);
    occupationMenu.add(openProjectOccupationDialogAction);
    occupationMenu.add(openProjectAreaOccupationDialogAction);
    return occupationMenu;
  }

  /**
   * Cria o menu referente a seleo de projetos.
   *
   * @return O menu referente a seleo de projetos.
   */
  private JMenu initProjectsMenu() {
    final String projectsText = getMenuString("projects");
    final JMenu projectsMenu = new JMenu(projectsText);

    if (isCurrentUserAdmin()) {
      allocateAreaItem = new JMenuItem(allocateAreaAction);
      deallocateAreaItem = new JMenuItem(deallocateAreaAction);

      setLockedStatusItem = new JMenuItem(changeStatusLockAction);
      setAllocatedStatusItem = new JMenuItem(changeStatusAllocateAction);
      setNoAllocationStatusItem = new JMenuItem(changeStatusNoAllocationAction);

      changeServerOwnerProjectItem =
        new JMenuItem(changeServerOwnerProjectDialogAction);
    }

    setPublicROProjectSharingStatsItem =
      new JMenuItem(setPublicROProjectSharingStatsAction);
    setPublicRWProjectSharingStatsItem =
      new JMenuItem(setPublicRWProjectSharingStatsAction);

    setPrivateProjectSharingStatsItem =
      new JMenuItem(setPrivateProjectSharingStatsAction);

    overwriteUsersItem = new JMenuItem(setProjectSharedAction);

    appendUsersROItem = new JMenuItem(appendUsersROAction);
    appendUsersRWItem = new JMenuItem(appendUsersRWAction);

    removeUsersItem = new JMenuItem(removeUsersAction);

    replicateSharedItem = new JMenuItem(replicateSharingDataAction);

    projectRemovalItem = new JMenuItem(removeProjectAction);

    if (!isCurrentUserAdmin()) {
      userRemovalFromProjectItem = new JMenuItem(userRemovalFromProjectAction);
    }

    showHistoryItem = new JMenuItem(showProjectHistoryDialogAction);
    writeCommentsItem = new JMenuItem(openProjectCommentsDialogAction);
    mailItem = new JMenuItem(openMailDialogAction);
    searchItem = new JMenuItem(searchFilesDialogAction);

    projectsMenu.addSeparator();

    if (isCurrentUserAdmin() && hasAreaReserved()) {
      projectsMenu.add(allocateAreaItem);
      projectsMenu.add(deallocateAreaItem);
      projectsMenu.addSeparator();
      final JMenu advancedMenu = new JMenu();
      advancedMenu.add(setLockedStatusItem);
      advancedMenu.add(setAllocatedStatusItem);
      advancedMenu.add(setNoAllocationStatusItem);
      advancedMenu.setText(getMenuString("advanced"));
      projectsMenu.add(advancedMenu);
      projectsMenu.addSeparator();
    }

    final JMenu publicMenu = new JMenu();
    publicMenu.setText(ProjectsManagerUI
      .getString("ProjectsManager.set.public.item"));
    publicMenu.add(setPublicROProjectSharingStatsItem);
    publicMenu.add(setPublicRWProjectSharingStatsItem);

    projectsMenu.add(setPrivateProjectSharingStatsItem);
    projectsMenu.add(overwriteUsersItem);
    projectsMenu.add(publicMenu);
    projectsMenu.addSeparator();

    final JMenu appendMenu = new JMenu();
    appendMenu.setText(ProjectsManagerUI
      .getString("ProjectsManager.append.users.item"));
    appendMenu.add(appendUsersROItem);
    appendMenu.add(appendUsersRWItem);
    projectsMenu.add(appendMenu);

    projectsMenu.add(removeUsersItem);
    projectsMenu.addSeparator();

    projectsMenu.add(replicateSharedItem);

    projectsMenu.addSeparator();
    projectsMenu.add(projectRemovalItem);

    if (!isCurrentUserAdmin()) {
      projectsMenu.addSeparator();
      projectsMenu.add(userRemovalFromProjectItem);
    }

    projectsMenu.addSeparator();
    projectsMenu.add(showHistoryItem);
    projectsMenu.add(writeCommentsItem);
    projectsMenu.add(mailItem);
    projectsMenu.add(searchItem);

    if (isCurrentUserAdmin()) {
      projectsMenu.add(changeServerOwnerProjectItem);
    }

    return projectsMenu;
  }

  /**
   * Atualiza a lista de projetos selecionados.
   *
   * @param projectList A lista de projetos selecionados.
   */
  public void setSelectedProjectsList(
    final List<ProjectsManagerData> projectList) {
    this.selectedProjectsList = projectList;
    boolean disable = projectList == null || projectList.size() == 0;
    enableItems(!disable);
    infoPanel.setSelectedProjects(projectList);
  }

  /**
   * Retorna o texto de internacionalizao de menu.
   *
   * @param tag tag do menu
   * @return o texto
   */
  final private String getMenuString(final String tag) {
    final String menuTag = "ProjectsManager." + tag + ".menu";
    return ProjectsManagerUI.getString(menuTag);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public void killApplication() {
    final JFrame mainFrame = getApplicationFrame();
    projectsCreationRemovalObserver.removeItselfAsObserver();
    projectSharingStatusObserver.removeItselfAsObserver();
    mainFrame.dispose();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final protected boolean userCanKillApplication() {
    return true;
  }

  /**
   * Retorna um vetor com todos os projetos acessveis.
   *
   * @return vetor com todos os projetos acessveis
   */
  final public List<ProjectsManagerData> getAllProjects() {
    return allProjectsList;
  }

  /**
   * Retorna uma lista os projetos visveis aps aplicao do filtro.
   *
   * @return lista os projetos disponveis aps aplicao do filtro. ATENO:
   *         retorna a prpria lista, e no uma cpia
   */
  final public List<ProjectsManagerData> getVisibleProjects() {
    return tablePanel.getVisibleProjects();
  }

  /**
   * Retorna usurio corrente.
   *
   * @return O usurio corrente.
   */
  public static User getUser() {
    return User.getLoggedUser();
  }

  /**
   * Informa se o usurio corrente  administrador do sistema.
   *
   * @return boolean indicando se o usurio corrente  administrador do sistema.
   */
  public static boolean isCurrentUserAdmin() {
    final User loggedUser = User.getLoggedUser();
    if (loggedUser.isAdmin()) {
      return true;
    }

    return false;
  }

  /**
   * Retorna uma lista de usurios do sistema
   *
   * @return A lista de usurios do sistema.
   * @throws Exception se a tarefa de busca dos usurios o fizer
   */
  public List<UserOutline> getUsers() throws Exception {
    final RetrieveUsersTask raut = new RetrieveUsersTask(this);
    final ApplicationFrame frame = getApplicationFrame();
    raut.execute(frame, "", "");

    if (raut.wasCancelled()) {
      final String err = getString("ProjectsManager.cancelled.message");
      raut.showError(err);
      return null;
    }

    if (raut.getStatus() != true) {
      final Exception exception = raut.getError();
      throw exception;
    }

    final List<UserOutline> users = raut.getResult();
    if (users == null) {
      showMessage(getString("ProjectsManager.error.obtaining.users.message"));
    }
    return users;
  }

  /**
   * Desabilita todos os itens da toolbar e do menu.
   *
   * @param enabled Define se os itens devem ser habilitados ou desabilitados.
   */
  private void enableItems(final boolean enabled) {
    projectsSpaceButton.setEnabled(enabled);
    multipleProjectSpaceItem.setEnabled(enabled);

    if (allocateButton != null) {
      allocateButton.setEnabled(enabled);
    }

    if (allocateAreaItem != null) {
      allocateAreaItem.setEnabled(enabled);
    }

    if (deallocateButton != null) {
      deallocateButton.setEnabled(enabled);
    }

    if (deallocateAreaItem != null) {
      deallocateAreaItem.setEnabled(enabled);
    }

    if (setLockedStatusItem != null) {
      setLockedStatusItem.setEnabled(enabled);
    }

    if (setAllocatedStatusItem != null) {
      setAllocatedStatusItem.setEnabled(enabled);
    }

    if (setNoAllocationStatusItem != null) {
      setNoAllocationStatusItem.setEnabled(enabled);
    }

    setPublicMenuButton.setEnabled(enabled);
    setPublicROProjectSharingStatsItem.setEnabled(enabled);

    setPublicRWProjectSharingStatsItem.setEnabled(enabled);

    setPrivateButton.setEnabled(enabled);
    setPrivateProjectSharingStatsItem.setEnabled(enabled);

    overwriteUsersButton.setEnabled(enabled);
    overwriteUsersItem.setEnabled(enabled);

    appendUsersROItem.setEnabled(enabled);
    appendUsersRWItem.setEnabled(enabled);

    removeUsersItem.setEnabled(enabled);

    replicateSharedButton.setEnabled(enabled);
    replicateSharedItem.setEnabled(enabled);

    projectRemovalButton.setEnabled(enabled);
    projectRemovalItem.setEnabled(enabled);

    if (!isCurrentUserAdmin()) {
      userRemovalButton.setEnabled(enabled);
      userRemovalFromProjectItem.setEnabled(enabled);
    }

    showHistoryButton.setEnabled(enabled && getSelectedProjects().size() == 1);
    showHistoryItem.setEnabled(enabled && getSelectedProjects().size() == 1);

    writeCommentsButton.setEnabled(enabled);
    writeCommentsItem.setEnabled(enabled);

    mailButton.setEnabled(enabled);
    mailItem.setEnabled(enabled);

    if (changeServerOwnerProjectItem != null) {
      changeServerOwnerProjectItem.setEnabled(enabled);
    }

  }

  /**
   * Obtm um projeto com o mesmo nome e dono que outro (especificado via
   * {@link CommonProjectInfo}).
   *
   * @param cpi informaes do projeto. Apenas o nome e o dono so usados na
   *        busca.
   * @return o projeto encontrado, ou null.
   */
  public ProjectsManagerData getProject(final CommonProjectInfo cpi) {
    return getProject(cpi.projectId);
  }

  /**
   * Obtm um projeto com o mesmo nome e dono que outro (especificado via
   * {@link CommonClientProject}).
   *
   * @param ccp informaes do projeto. Apenas o nome e o dono so usados na
   *        busca.
   * @return o projeto encontrado, ou null.
   */
  private ProjectsManagerData getProject(final CommonClientProject ccp) {
    return getProject(ccp.getId());
  }

  /**
   * Obtm um projeto com o mesmo nome e dono que outro (especificado via
   * {@link CommonClientProject}).
   *
   * @param projectId Id do projeto.
   * @return o projeto encontrado, ou null.
   */
  private ProjectsManagerData getProject(final Object projectId) {
    if (allProjectsList == null) {
      return null;
    }
    for (ProjectsManagerData pmd : allProjectsList) {
      if (pmd.getProjectId().equals(projectId)) {
        return pmd;
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void update(Observable o, Object arg) {
    if (o == null || !(o instanceof CommonClientProject)) {
      return;
    }
    CommonClientProject ccp = (CommonClientProject) o;
    ProjectsManagerData pmd = getProject(ccp);
    if (pmd == null) {
      pmd =
        new ProjectsManagerData(ccp.getId(), ccp.getName(), ccp.getUserId());
    }
    updateSpecificProject(pmd, ccp);
  }

  /**
   * Atualiza um projeto da classe ProjectsManagerData a partir de um projeto da
   * classe CommonClientProject.
   *
   * @param newProjectData Projeto da classe ProjectsManagerData.
   * @param newProject Projeto da classe CommonClientProject.
   */
  private void updateSpecificProject(final ProjectsManagerData newProjectData,
    final CommonClientProject newProject) {

    if (newProjectData == null || newProject == null) {
      return;
    }

    final GetUserNameTask gunt = new GetUserNameTask(newProject.getUserId());

    if (gunt.wasCancelled()) {
      final String err = getString("ProjectsManager.cancelled.message");
      StandardErrorDialogs.showErrorDialog(getApplicationFrame(), err);
      return;
    }

    if (gunt.getStatus() != true) {
      final Exception exception = gunt.getError();
      StandardErrorDialogs.showErrorDialog(getApplicationFrame(), exception);
      return;
    }

    final String username = gunt.getResult();
    if (username == null) {
      final String err = getString("ProjectsManager.user.not.found.message");
      StandardErrorDialogs.showErrorDialog(getApplicationFrame(), err);
      return;
    }

    newProjectData.setOwnerName(username);
    newProjectData
    .setProjectSpaceAllocation(ProjectSpaceAllocation.NO_ALLOCATION);

    newProjectData.setProjectDescription(newProject.getDescription());
    newProjectData.setCreationDate(newProject.getCreationDate());
    newProjectData.setModificationDate(newProject.getLastModificationDate());

    newProjectData.setSharingType(newProject.getSharingType());
    newProjectData.setUsers(newProject.getUsersRO(), newProject.getUsersRW());

    newProjectData.setServerOwnerName(newProject.getOwnerServerName());

    addProject(newProjectData);
  }

  /**
   * Informa se o painel de filtragem est sendo exibido ou no.
   *
   * @return boolean indicando se o painel de filtragem est sendo exibido ou
   *         no.
   */
  public boolean isFilterOn() {
    return filterButton.isSelected();
  }

  /**
   * Atualiza a lista de projetos <b>localmente</b>.
   */
  public void refreshProjectsTable() {
    refreshProjectsTable(false);
  }

  /**
   * Atualiza a lista de projetos, opcionalmente obtendo dados do servidor.
   *
   * TODO verificar a relao com o filtro
   *
   * @param updateFromServer - obtm dados atuais do servidor
   * @param updateOccupiedSpace - atualiza a ocupao em disco dos projetos.
   *        Esta opo tem precedncia sobre <code>updateFromServer</code>
   */
  private void refreshProjectsTable(final boolean updateFromServer) {
    try {
      if (updateFromServer) {
        updateProjectsAction.actionDone();
      }
    }
    catch (Exception e) {
      StandardErrorDialogs.showErrorDialog(getApplicationFrame(), e);
    }
    tablePanel.refreshTable();
  }

  /**
   * Desmarca linhas selecionadas da tabela de projetos.
   */
  public void clearProjectSelection() {
    tablePanel.clearTableSelection();
  }

  /**
   * Remove um projeto da estrutura projectsList a partir do seu nome.
   *
   * @param cpi Nome do projeto.
   */
  public void removeProject(CommonProjectInfo cpi) {
    final ProjectsManagerData pmd = getProject(cpi.projectId);
    if (pmd == null) {
      return;
    }
    allProjectsList.remove(pmd);
    tablePanel.removeProjectFromTable(pmd);
  }

  /**
   * Remove projetos da lista de projetos.
   *
   * @param projectList projetos a serem removidos
   */
  public void removeProjects(final List<ProjectsManagerData> projectList) {
    final List<ProjectsManagerData> newList =
      new ArrayList<ProjectsManagerData>();
    for (ProjectsManagerData prj : selectedProjectsList) {
      final ProjectSpaceAllocation spaceAllocation =
        prj.getProjectSpaceAllocation();
      if (!projectList.contains(prj)) {
        newList.add(prj);
      }
      else if (spaceAllocation == ProjectSpaceAllocation.ALLOCATED) {
        updateProjectToWaitingDeallocation(prj);
        newList.add(prj);
        tablePanel.refreshTable();
      }
      else {
        allProjectsList.remove(prj);
        tablePanel.removeProjectFromTable(prj);
      }
    }
    setSelectedProjectsList(newList);
  }

  /**
   * Atualiza as propriedades de um projeto do tipo
   * <code>ProjectsManagerData</code> que estava configurado como "alocado",
   * para "esperando desalocao".
   *
   * @param pmd O projeto a ser definido como esperando desalocao.
   */
  private void updateProjectToWaitingDeallocation(ProjectsManagerData pmd) {
    pmd.setProjectSpaceAllocation(ProjectSpaceAllocation.WAITING_DEALLOCATION);
    pmd.setAllocatedSpace(0);
    pmd.setOccupiedSpace(-1);
    pmd.setScope(ProjectsManagerScope.NOT_APPLICABLE);
    pmd.setUsers(null, null);
  }

  /**
   * Atualiza o painel com as informaes sobre os projetos selecionados.
   */
  public void refreshInfoPanel() {
    infoPanel.updateFields();
  }

  /**
   * Limpa campos do painel de informaes detalhadas dos projetos selecionados
   */
  public void clearInfoFields() {
    infoPanel.clearFields();
  }

  /**
   * Define a lista dos projetos visveis, i.e. que passaram pelo filtro.
   *
   * @param filteredProjects projetos que passaram pelo filtro
   */
  public void setFilteredProjects(
    final List<ProjectsManagerData> filteredProjects) {
    tablePanel.setVisibleProjects(filteredProjects);
  }

  /**
   * Remove todos os projetos.
   */
  public void clearAllProjects() {
    allProjectsList.clear();
    tablePanel.emptyTableData();
  }

  /**
   * Limpa a tabela.
   */
  public void clearTable() {
    tablePanel.emptyTableData();
  }

  /**
   * Limpa todos os filtros.
   */
  public void clearFilters() {
    filterPanel.emptyFiltersData();
  }

  /**
   * Adiciona um novo projeto  lista. Caso haja filtros ativos, o projeto ser
   * adicionado apenas se passar por todos os filtros.
   *
   * @param prj O projeto.
   */
  public void addProject(final ProjectsManagerData prj) {
    if (prj == null) {
      return;
    }
    allProjectsList.add(prj);
    final boolean isVisible =
      !isFilterOn() || filterPanel.projectMatchesFilters(prj);
    if (isVisible) {
      tablePanel.addProjectToTable(prj);
    }
  }

  /**
   * Obtm uma lista com os projetos selecionados pelo usurio.
   *
   * @return lista com os projetos selecionados pelo usurio
   */
  public List<ProjectsManagerData> getSelectedProjects() {
    return tablePanel.getSelectedProjects();
  }

  /**
   * Atualiza o valor de ocupao de um projeto em todas as listas onde este
   * projeto se encontra.
   *
   * @param projectsManagerData O projeto cuja ocupao foi alterada.
   */
  public void updateProjectOccupiedSpace(
    final ProjectsManagerData projectsManagerData) {
    final String projectName = projectsManagerData.getProjectName();
    final Object projectOwner = projectsManagerData.getOwnerId();
    for (ProjectsManagerData pmd : allProjectsList) {
      final String pName = pmd.getProjectName();
      final Object oId = pmd.getOwnerId();
      if (pName.equals(projectName) && oId.equals(projectOwner)) {
        pmd.setOccupiedSpace(projectsManagerData.getOccupiedSpace());
      }
    }
  }

  /**
   * Aplica todos os filtros e atualiza a tabela.
   *
   * @param clearSelection boolean que indica se a seleo de linhas corrente
   *        deve ser desfeita.
   */
  public void runFilters(final boolean clearSelection) {
    filterPanel.runFilters(clearSelection);
  }

  /**
   * Retorna boolean que indica se as informaes de reserva de rea devem ser
   * exibidas ou no.
   *
   * @return Boolean que indica se as informaes de reserva de rea devem ser
   *         exibidas ou no.
   */
  public boolean hasAreaReserved() {
    return areaReserved;
  }

}

/**
 * Remote Task para obteno de nome de usurio.
 *
 * @author Tecgraf
 */
class GetUserNameTask extends RemoteTask<String> {

  /**
   * Id do usurio.
   */
  final Object userId;

  /**
   * Construtor.
   *
   * @param userId Id do usurio.
   */
  GetUserNameTask(final Object userId) {
    this.userId = userId;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {

    final String username = User.getName(userId);
    setResult(username);
  }

}
