package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.swing.ImageIcon;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.UserRemovalFromProjectTask;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.User;

/**
 * Ao de remover um usurio de um projeto (do qual ele no  dono).
 * 
 * @author jnlopes
 */
public class UserRemovalFromProjectAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public UserRemovalFromProjectAction(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() <= 0) {
      return;
    }

    final List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    final boolean ret = runTask(projects);
    if (ret) {
      projectsManager.refreshProjectsTable();
    }
    else {
      StandardDialogs
        .showWarningDialog(
          projectsManager.getApplicationFrame(),
          getString("UserRemovalFromProjectAction.shared.user.removal.error.title"),
          getString("UserRemovalFromProjectAction.shared.user.removal.error.message"));
    }
  }

  /**
   * Roda e trata a execuo da task.
   * 
   * @param projects Lista de projetos vlidos para a operao.
   * @return Boolean indicando se a operao foi bem sucedida.
   * @throws Exception em caso de erro.
   */
  private boolean runTask(List<ProjectsManagerData> projects) throws Exception {
    final ProjectsManager projectsManager = getProjectsManager();
    final ApplicationFrame frame = projectsManager.getApplicationFrame();
    final UserRemovalFromProjectTask urfp =
      new UserRemovalFromProjectTask(getProjectsManager(), projects);
    urfp.execute(frame, getString("UserRemovalFromProjectAction.title"),
      getString("UserRemovalFromProjectAction.message"));
    if (urfp.wasCancelled()) {
      final String err =
        getString("UserRemovalFromProjectAction.cancelled.message");
      urfp.showError(err);
      return false;
    }
    if (urfp.getStatus() != true) {
      final Exception exception = urfp.getError();
      throw exception;
    }
    return urfp.getResult();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ApplicationImages.ICON_CANCEL_16;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    final List<ProjectsManagerData> prjList =
      new ArrayList<ProjectsManagerData>();
    prjList.addAll(projectList);

    final List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    final User loggedUser = User.getLoggedUser();
    final Object loggedUserId = loggedUser.getId();

    int i = 0;
    while (i < prjList.size()) {
      final ProjectsManagerData pmd = prjList.get(i);
      final ProjectSpaceAllocation spaceAllocation =
        pmd.getProjectSpaceAllocation();

      final SharingType sharingType = pmd.getSharingType();
      final Set<Object> usersRW = pmd.getUsersRW();

      final boolean isAdmin = loggedUser.isAdmin();
      final boolean isOwner = pmd.getOwnerId().equals(loggedUserId);
      final boolean hasWriteAccess =
        sharingType == SharingType.PARTIAL && usersRW.contains(loggedUserId);

      /*
       * Para que um usurio possa se remover de um projeto,  preciso que: - O
       * projeto possa ser aberto (no esteja aguardando alocao /
       * desalocao). - O usurio no seja o administrador, ou o dono do
       * projeto. - O projeto esteja no modo compartilhado e ele tenha permisso
       * de escrita.
       */
      if (spaceAllocation.isOpenable() && !isAdmin && !isOwner
        && hasWriteAccess) {
        i++;
      }
      else {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
    }

    final ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      final String deniedMessage =
        getString("UserRemovalFromProjectAction.project.selection.denied.message")
          + getString("UserRemovalFromProjectAction.project.requirements.message");
      StatusDialog
        .showNoneOkDialog(
          projectsManager,
          deniedMessage,
          failedProjects,
          getString("UserRemovalFromProjectAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      final String someDeniedMessage =
        getString("UserRemovalFromProjectAction.project.selection.some.denied.message")
          + getString("UserRemovalFromProjectAction.project.requirements.message");
      final int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("UserRemovalFromProjectAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }

    return prjList;
  }
}
