package csbase.client.applications.projectsmanager.dialogs;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.dialogs.core.ProjectsManagerDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpecificAreaOccupationData;
import csbase.client.applications.projectsmanager.panels.renderers.TableOccupationRenderer;
import csbase.client.applications.projectsmanager.proxy.GetSpecificSpaceUsageTask;
import csbase.logic.User;

/**
 * Model de usurio para exibio na tabela.
 *
 * @author Tecgraf/PUC-Rio
 */
class UserSpaceModel extends AbstractTableModel {

  /**
   * O dado.
   */
  private ProjectSpecificAreaOccupationData data;

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    return 3;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowCount() {
    if (data == null) {
      return 0;
    }

    final User[] users = data.getUsers();
    return users.length;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueAt(int rowIndex, int columnIndex) {
    final User[] users = data.getUsers();
    if (columnIndex == 0) {
      return users[rowIndex].getId();
    }
    else if (columnIndex == 1) {
      return users[rowIndex].getName();
    }
    else if (columnIndex == 2) {
      final Double[] spaceByUser = data.getSpaceByUser();
      return spaceByUser[rowIndex];
    }
    else {
      return null;
    }
  }

  /**
   * Ajuste do dado.
   *
   * @param projectSpecificAreaOccupationData o dado
   */
  public void setData(
    final ProjectSpecificAreaOccupationData projectSpecificAreaOccupationData) {
    this.data = projectSpecificAreaOccupationData;
  }
}

/**
 * Dilogo que exibe informaes de ocupao da rea de projetos.
 *
 * @author jnlopes
 */
public class ProjectOccupationDialog extends ProjectsManagerDialog {

  /**
   * Tabela de ocupao por usurio.
   */
  final private SortableTable userSpaceTable = new SortableTable();

  /**
   * Cria o paibel do botes.
   *
   * @return o painel.
   */
  private JPanel buildButtonsPanel() {
    final JButton closeButton = new JButton();
    final JButton refreshButton = new JButton();

    refreshButton.setAction(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        refreshData();
      }
    });
    refreshButton.setText(getString("ProjectOccupationDialog.refresh.button"));

    final ProjectOccupationDialog pod = this;
    closeButton.setAction(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        pod.dispose();
      }
    });
    closeButton.setText(getString("ProjectOccupationDialog.close.button"));

    JComponent[] buttons = { refreshButton, closeButton };
    GUIUtils.matchPreferredSizes(buttons);

    final JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(refreshButton);
    buttonsPanel.add(closeButton);
    return buttonsPanel;
  }

  /**
   * Inicializa tabela e modelo da tabela.
   */
  private void initTable() {
    userSpaceTable.setModel(new UserSpaceModel());
    userSpaceTable.setNoSortStateEnabled(true);

    // Ajuste do renderer
    final TableColumnModel columnModel = userSpaceTable.getColumnModel();
    final TableColumn column2 = columnModel.getColumn(2);
    column2.setCellRenderer(new TableOccupationRenderer());

    // Ajuste de colunas
    final JTableHeader header = userSpaceTable.getTableHeader();
    final TableColumnModel tcm = header.getColumnModel();

    final TableColumn c0 = tcm.getColumn(0);
    final TableColumn c1 = tcm.getColumn(1);
    final TableColumn c2 = tcm.getColumn(2);

    c0.setPreferredWidth(50);
    c1.setPreferredWidth(250);
    c2.setPreferredWidth(150);

    c0.setHeaderValue(getString("ProjectOccupationDialog.login.header"));
    c1.setHeaderValue(getString("ProjectOccupationDialog.name.header"));
    c2.setHeaderValue(getString("ProjectOccupationDialog.space.header"));
  }

  /**
   * Monta o painel com componentes especficos para visualizao de espao
   * ocupado por usurios.
   */
  private void initPanel() {
    setLayout(new GridBagLayout());

    final JScrollPane userSpaceTablePane = buildListPanel();
    final JPanel buttonsPanel = buildButtonsPanel();

    add(userSpaceTablePane, new GBC(0, 0).both());
    add(buttonsPanel, new GBC(0, 1).horizontal());
  }

  /**
   * Montagem do painel com os usurios
   *
   * @return o painel de usurios
   */
  private JScrollPane buildListPanel() {
    final JScrollPane userSpaceTablePane = new JScrollPane(userSpaceTable);
    final JPanel specificPanel = new JPanel(new GridBagLayout());
    specificPanel.add(userSpaceTablePane, new GBC(0, 0).both());
    BorderUtil.setLowBorder(userSpaceTablePane);
    return userSpaceTablePane;
  }

  /**
   * Atualiza campos com informaes de ocupao da rea de projetos.
   */
  private void refreshData() {
    final ProjectsManager projectsManager = getProjectsManager();
    final GetSpecificSpaceUsageTask gssut =
      new GetSpecificSpaceUsageTask(projectsManager);
    final ApplicationFrame appFrame = projectsManager.getApplicationFrame();

    final String msg = getString("ProjectOccupationDialog.task.message");
    gssut.execute(appFrame, projectsManager.getName(), msg);

    if (gssut.wasCancelled()) {
      final String err =
        getString("ProjectOccupationDialog.task.cancelled.message");
      gssut.showError(err);
      return;
    }

    if (gssut.getStatus() != true) {
      final Exception exception = gssut.getError();
      projectsManager.showExceptionStack(exception);
      return;
    }

    final ProjectSpecificAreaOccupationData paod = gssut.getResult();
    ((UserSpaceModel) userSpaceTable.getModel()).setData(paod);
  }

  /**
   * Construtor
   *
   * @param projectsManager A aplicao.
   */
  public ProjectOccupationDialog(final ProjectsManager projectsManager) {
    super(projectsManager);
    initTable();
    initPanel();
    refreshData();
  }

}
