package csbase.client.applications.projectsmanager.proxy;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.ProjectsManagerUI;
import csbase.client.applications.projectsmanager.models.ProjectHistoryBlockType;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.core.ProjectsManagerTask;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectAdminInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Task que trata alterao na alocao de projetos. Projetos com espao alocado
 * que sero desalocados e projeto esperando alocao que sero alocados.
 * 
 * @author jnlopes
 * 
 */
public class AllocateAreaTask extends ProjectsManagerTask<Boolean> {

  /**
   * Informaes dos projetos.
   */
  final private List<ProjectsManagerData> projects;

  /**
   * Boolean que indica se rea deve ser definida como alocada ou desalocada.
   */
  final private boolean allocate;

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao.
   * @param projects Informaes dos projetos.
   * @param allocate boolean que indice se a rea deve ser definida como alocada
   *        ou desalocada.
   */
  public AllocateAreaTask(ProjectsManager projectsManager,
    List<ProjectsManagerData> projects, boolean allocate) {
    super(projectsManager);
    this.projects = projects;
    this.allocate = allocate;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws RemoteException {
    final ProjectAdminInfo[] prjAdmInfo = ProjectAdminInfo.getAll();
    if (prjAdmInfo == null) {
      setResult(false);
      return;
    }
    List<ProjectAdminInfo> selectedProjects =
      new ArrayList<ProjectAdminInfo>();
    if (allocate) {
      for (ProjectAdminInfo p : prjAdmInfo) {
        for (ProjectsManagerData prj : projects) {
          if (p.isLocked() && p.getProjectName().equals(prj.getProjectName())
            && p.getOwnerId().equals(prj.getOwnerId())) {
            p.setAreaLockedSize(p.getAreaRequestSize());
            selectedProjects.add(p);
          }
        }
      }
      if (selectedProjects.size() == 0) {
        setResult(true);
        return;
      }
      ProjectAdminInfo[] sp =
        selectedProjects.toArray(new ProjectAdminInfo[selectedProjects.size()]);
      ProjectAdminInfo.allocateAreaForProjects(sp);
      saveInfoToProjectHistory();
    }
    else {
      for (ProjectAdminInfo p : prjAdmInfo) {
        for (ProjectsManagerData prj : projects) {
          if (p.isWaitingAreaFree()
            && p.getProjectName().equals(prj.getProjectName())
            && p.getOwnerId().equals(prj.getOwnerId())) {
            selectedProjects.add(p);
          }
        }
      }
      ProjectAdminInfo[] sp =
        selectedProjects.toArray(new ProjectAdminInfo[selectedProjects.size()]);
      ProjectAdminInfo.freeAreaForProjects(sp);
    }
    setResult(true);
  }

  /**
   * Obtm os objetos CommonClientProject (que so criados no momento em que as
   * reas so liberadas) e adiciona a informao de que o projeto teve sua rea
   * liberada.
   * 
   * @throws RemoteException em caso de falha na execuo de mtodo remoto.
   */
  private void saveInfoToProjectHistory() throws RemoteException {
    ProjectServiceInterface psi = ClientRemoteLocator.projectService;
    for (ProjectsManagerData prj : projects) {
      CommonClientProject ccp = psi.openProject(prj.getProjectId(), false);
      ProjectsManagerUI.addAutomaticHistoryBlock(ccp,
        ProjectHistoryBlockType.AREA_ALLOCATED);
    }
  }
}
