/*
 * $Id:$
 */

package csbase.client.applications.xmlviewer.actions.core;

import java.io.IOException;

import javax.swing.ImageIcon;
import javax.swing.JComponent;

import org.xml.sax.SAXParseException;

import tecgraf.javautils.gui.StatusBar;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.xmlviewer.XMLViewer;
import csbase.client.desktop.Task;
import csbase.client.util.xmlpanel.XMLPanelStyleInterface;
import csbase.exception.PermissionException;
import csbase.logic.ClientFile;

/**
 * Ao abrir arquivo CSV para criao de relatrio
 * 
 * @author Tecgraf/PUC-Rio
 */
abstract public class XMLViewerAbstractOpenAction extends XMLViewerAction {

  /**
   * Mtodo inteno de busca de arquivo.
   * 
   * @return um objeto <code>ClientFile</code> ou <code>null</code>.
   * @throws Exception se houvber falha.
   */
  abstract protected ClientFile getFileForOpen() throws Exception;

  /**
   * {@inheritDoc}
   * 
   * @throws Exception em caso de falha na carga de relatrio.
   */
  @Override
  final public void actionDone(final JComponent cmp) throws Exception {
    final ClientFile file = getFileForOpen();
    if (file == null) {
      return;
    }
    openFileInTask(file);

    final XMLViewer application = getApplication();
    final String type = file.getType();
    final XMLPanelStyleInterface style = application.getTipicalStyle(type);
    application.setStyle(style);
  }

  /**
   * Faz abertura do arquivo (em depurao).
   * 
   * @param file arquivo
   */
  @SuppressWarnings("unused")
  private void _openFileInTask(final ClientFile file) {
    final XMLViewer application = getApplication();
    try {
      application.openXML(file);
    }
    catch (Exception e) {
      application.showExceptionStack(e);
      return;
    }
  }

  /**
   * Faz abertura do arquivo.
   * 
   * @param file arquivo
   */
  private void openFileInTask(final ClientFile file) {
    final XMLViewer application = getApplication();
    final Task<Void> task = new Task<Void>() {
      @Override
      public void performTask() throws Exception {
        application.openXML(file);
      }

      /**
       * {@inheritDoc}
       */
      @Override
      protected void handleError(Exception exception) {
        final String msg = exception.getMessage();
        if (exception instanceof IOException) {
          final String err =
            getString("XMLViewerAbstractOpenAction.open.xml.io.failure");
          application.showError(err + "\n\u2022" + msg);
          return;
        }
        if (exception instanceof SAXParseException) {
          final String err =
            getString("XMLViewerAbstractOpenAction.open.xml.syntax.failure");
          application.showError(err + "\n\u2022" + msg);
          return;
        }
        if (exception instanceof PermissionException) {
          final String err =
            getString("XMLViewerAbstractOpenAction.open.xml.permission.failure");
          application.showError(err);
          return;
        }
        super.handleError(exception);
      }
    };

    final String msg =
      getString("XMLViewerAbstractOpenAction.open.xml.message");
    final ImageIcon icon = ApplicationImages.ICON_ANIMATEDENGINES_32;
    final String title = application.getName();
    final ApplicationFrame frame = application.getApplicationFrame();
    final boolean ok = task.execute(frame, title, msg, icon);

    final StatusBar statusBar = frame.getStatusBar();
    if (task.wasCancelled()) {
      final String cancelMsg =
        getString("XMLViewerAbstractOpenAction.open.xml.cancel");
      statusBar.setWarning(cancelMsg, XMLViewer.STATUS_BAR_TIMEOUT);
      return;
    }

    if (!ok) {
      final String err =
        getString("XMLViewerAbstractOpenAction.open.xml.failure");
      statusBar.setError(err, XMLViewer.STATUS_BAR_TIMEOUT);
      application.resetFile();
      return;
    }

    final String okMsg =
      getString("XMLViewerAbstractOpenAction.open.xml.success");
    statusBar.setInfo(okMsg, XMLViewer.STATUS_BAR_TIMEOUT);
  }

  /**
   * Construtor padro.
   * 
   * @param application a aplicao
   * @param icon cone
   */
  public XMLViewerAbstractOpenAction(final XMLViewer application, ImageIcon icon) {
    super(application, icon);
  }
}
