package csbase.client.csdk.v1_0.filesystem;

import java.awt.Window;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import csbase.client.ClientSmartFile;
import csbase.client.ClientSmartFileFactory;
import csbase.client.csdk.v1_0.core.CSDKAbstractContext;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.project.ProjectTreeListener;
import csbase.client.project.tasks.CreateFileTask;
import csbase.client.project.tasks.GetFileTask;
import csbase.client.project.tasks.RemoveFilesTask;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil.OperationResult;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.applicationservice.ApplicationRegistry;
import csdk.v1_0.api.filesystem.FileSelectionType;
import csdk.v1_0.api.filesystem.IFile;
import csdk.v1_0.api.filesystem.project.IProjectContext;
import csdk.v1_0.api.filesystem.project.IProjectObserver;
import csdk.v1_0.api.filesystem.project.ProjectException;
import tecgraf.javautils.core.lng.LNG;

/**
 * Implementao do contexto de projeto do CSDK, dando acesso ao sistema de
 * arquivos do projeto atualmente aberto no cliente CSBASE.
 */
public class CSDKProjectContext extends CSDKAbstractContext
  implements IProjectContext {

  /**
   * Observador nativo da rvore de projetos;
   */
  private final ProjectTreeListener projectTreeListener;

  /**
   * Observadores de projeto cadastrados via CSDK;
   */
  private final List<IProjectObserver> projectObservers;

  /**
   * O registro da aplicao.
   */
  private ApplicationRegistry registry;

  /**
   * Construtor.
   *
   * @param registry o registro da aplicao.
   */
  public CSDKProjectContext(ApplicationRegistry registry) {
    this.registry = registry;
    this.projectObservers = new ArrayList<IProjectObserver>();
    this.projectTreeListener = new ProjectTreeAdapter() {
      @Override
      public void projectChanged(final CommonClientProject project) {
        if (project == null) {
          return;
        }
        String id = project.getId().toString();
        for (IProjectObserver listener : projectObservers) {
          listener.onProjectOpen(id);
        }
      }

      @Override
      public void projectClosed(final CommonClientProject project) {
        if (project == null) {
          return;
        }
        String id = project.getId().toString();
        for (IProjectObserver listener : projectObservers) {
          listener.onProjectClose(id);
        }
      }
    };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void contextActivated() {
    DesktopFrame frame = DesktopFrame.getInstance();
    ProjectTree projectTree = frame.getTree();
    projectTree.addProjectTreeListener(projectTreeListener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile browseProjectFileInOpenMode(String[] fileTypes, boolean allowAll,
    FileSelectionType selectionType, Window frame) throws ProjectException {
    checkProjectIsOpen();

    Collection<String> types;
    if (fileTypes != null) {
      types = Arrays.asList(fileTypes);
    }
    else {
      types = Collections.emptyList();
    }
    OperationResult result;
    if (selectionType == FileSelectionType.DIRECTORIES_ONLY) {
      result = ClientProjectFileChooserUtil.browseSingleDirectoryInOpenMode(
        frame, types, registry.getApplicationName(LNG.getLocale()), allowAll);
    }
    else {
      result = ClientProjectFileChooserUtil.browseSingleFileInOpenMode(frame,
        types, registry.getApplicationName(LNG.getLocale()), allowAll);
    }
    if (result != null) {
      ClientFile file = result.getClientSmartFile();
      if (file != null) {
        CSDKFile csdkFile = new CSDKFile(file);
        return csdkFile;
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile[] browseMultipleProjectFilesInOpenMode(String[] fileTypes,
    boolean allowAll, FileSelectionType selectionType, Window window)
      throws ProjectException {
    checkProjectIsOpen();

    Collection<String> types;
    if (fileTypes != null) {
      types = Arrays.asList(fileTypes);
    }
    else {
      types = Collections.emptyList();
    }
    List<OperationResult> results;
    if (selectionType == FileSelectionType.DIRECTORIES_ONLY) {
      results =
        ClientProjectFileChooserUtil.browseMultipleDirectoriesInOpenMode(window,
          types, registry.getApplicationName(LNG.getLocale()), allowAll);
    }
    else {
      results = ClientProjectFileChooserUtil.browseMultipleFilesInOpenMode(
        window, types, registry.getApplicationName(LNG.getLocale()), allowAll);
    }

    if (results != null) {
      List<IFile> csdkFiles = new ArrayList<IFile>();
      for (OperationResult result : results) {
        ClientFile file = result.getClientSmartFile();
        if (file != null) {
          csdkFiles.add(new CSDKFile(file));
        }
      }
      return csdkFiles.toArray(new IFile[csdkFiles.size()]);
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile browseProjectFileInSaveMode(String[] fileTypes, boolean allowAll,
    String defaultName, Window frame) throws ProjectException {
    checkProjectIsOpen();

    Collection<String> types;
    if (fileTypes != null) {
      types = Arrays.asList(fileTypes);
    }
    else {
      types = Collections.emptyList();
    }
    OperationResult result =
      ClientProjectFileChooserUtil.browseSingleFileInSaveMode(frame, types,
        null, registry.getApplicationName(LNG.getLocale()), allowAll, true);
    if (result != null) {
      ClientFile file = result.getClientSmartFile();
      if (file != null) {
        CSDKFile csdkFile = new CSDKFile(file);
        return csdkFile;
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile getProjectFile(String[] path, Window window)
    throws ProjectException {
    checkProjectIsOpen();

    ClientProjectFile cpf = getFile(path, window);
    ClientSmartFile smartFile =
      ClientSmartFileFactory.getInstance().create(cpf);
    if (smartFile == null) {
      return null;
    }
    return new CSDKFile(smartFile);
  }

  /**
   * Obtm o arquivo de projeto do caminho especificado.
   *
   * @param path o caminho.
   * @param window janela-me da tarefa.
   * @return o arquivo.
   */
  private ClientProjectFile getFile(String[] path, Window window) {
    DesktopFrame desktop = DesktopFrame.getInstance();
    CommonClientProject project = desktop.getProject();
    ClientProjectFile file = GetFileTask.runTask(window, project, path);
    return file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile createProjectFile(String[] path, String name, String fileType,
    Window window) throws ProjectException {
    checkProjectIsOpen();

    DesktopFrame desktop = DesktopFrame.getInstance();
    CommonClientProject project = desktop.getProject();
    ClientProjectFile parent;
    if (path == null || path.length == 0) {
      parent = project.getRoot();
    }
    else {
      parent = getFile(path, window);
    }
    if (parent == null) {
      return null;
    }
    ClientProjectFile newFile =
      CreateFileTask.runTask(window, project, parent, name, fileType);
    if (newFile == null) {
      return null;
    }
    ClientSmartFile smartFile =
      ClientSmartFileFactory.getInstance().create(newFile);
    return new CSDKFile(smartFile);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeProjectFile(final String[] path, Window parent)
    throws ProjectException {
    checkProjectIsOpen();

    DesktopFrame desktop = DesktopFrame.getInstance();
    final CommonClientProject project = desktop.getProject();
    return RemoveFilesTask.runTask(parent, project, path);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCurrentProjectId() {
    DesktopFrame desktop = DesktopFrame.getInstance();
    CommonClientProject project = desktop.getProject();
    if (project != null) {
      return project.getId().toString();
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addProjectObserver(final IProjectObserver observer) {
    this.projectObservers.add(observer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeProjectObserver(IProjectObserver observer) {
    this.projectObservers.remove(observer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void contextDeactivated() {
    DesktopFrame frame = DesktopFrame.getInstance();
    ProjectTree projectTree = frame.getTree();
    projectTree.removeProjectTreeListener(projectTreeListener);
    this.projectObservers.clear();
  }

  /**
   * Verifica se existe um projeto aberto.
   *
   * @throws ProjectException em caso de no existir projeto aberto.
   */
  private void checkProjectIsOpen() throws ProjectException {
    DesktopFrame desktop = DesktopFrame.getInstance();
    CommonClientProject project = desktop.getProject();
    if (project == null) {
      String errorMsg = LNG.get(CSDKProjectContext.class.getSimpleName()
        + ".project.unavailable.error");
      throw new ProjectException(errorMsg);
    }
  }

}
