package csbase.client.desktop;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import csbase.client.algorithms.commands.CommandPersistenceListener;
import csbase.logic.Notification;
import csbase.logic.NotificationHandler;
import csbase.logic.algorithms.commands.CommandPersistenceNotification;
import csbase.remote.CommandPersistenceServiceInterface;

/**
 * O observador remoto do servio {@link CommandPersistenceServiceInterface}.
 */
public final class CommandPersistenceCommandNotificationHandler implements NotificationHandler {
   
  /** Instncia nica do observador */
  private static CommandPersistenceCommandNotificationHandler instance =
    new CommandPersistenceCommandNotificationHandler();

  /** Os observadores locais. */
  private List<CommandPersistenceListener> listeners;
  
  /**
   * Responsvel por entregar os eventos aos ouvintes.
   */
  private Executor executor;

  /**
   * Construtor que exporta este objeto para o RMI e o registra como observador
   * do algoService e do ServerMonitor.
   */
  private CommandPersistenceCommandNotificationHandler() {
    super();
    listeners = new LinkedList<CommandPersistenceListener>();
    executor = Executors.newSingleThreadExecutor();
  }

  /**
   * Obtm o singleton.
   * 
   * @return O singleton.
   */
  public static CommandPersistenceCommandNotificationHandler getInstance() {
    return instance;
  }

  /**
   * Adiciona um observador local a este observador remoto.
   * 
   * @param listener O observador local (No aceita {@code null}).
   */
  public void addListener(CommandPersistenceListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  /**
   * Remove um observador local deste observador remoto.
   * 
   * @param listener O observador local (No aceita {@code null}).
   * 
   * @return {@code false} se o observador a ser removido no estava cadastrado
   *         neste observador remoto.
   */
  public boolean removeListener(CommandPersistenceListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    return listeners.remove(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void gotNotification(Notification data) {
    if (!(data instanceof CommandPersistenceNotification)) {
      return;
    }
    final CommandPersistenceNotification pData =
      (CommandPersistenceNotification) data;
    Runnable doRun = new Runnable() {
      @Override
      public void run() {
        Object projectId = pData.getProjectId();
        String commandId = pData.getCommandId();
        switch (pData.getType()) {
          case REMOVED: {
            for (CommandPersistenceListener listener : listeners) {
              listener.commandWasRemoved(projectId, commandId);
            }
            break;
          }
          case SAVED: {
            for (CommandPersistenceListener listener : listeners) {
              listener.commandWasSaved(projectId, commandId);
            }
            break;
          }
          case UPDATED: {
            for (CommandPersistenceListener listener : listeners) {
              listener.commandWasUpdated(projectId, commandId);
            }
            break;
          }
          default: {
            String errorMessage =
              String
                .format(
                  "Notificao de tipo desconhecido.\nProjeto: %s.\nComando: %s.\nTipo de notificao: %s.\n",
                  projectId, commandId, pData.getType());
            throw new IllegalArgumentException(errorMessage);
          }
        }
      }
    };
    executor.execute(doRun);
  }
}
