package csbase.client.facilities.algorithms.executor;

import java.awt.Window;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.Client;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applications.executor.ExecutorFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.sga.CommandRequestedListener;
import csbase.exception.OperationFailureException;
import csbase.logic.CommandInfo;
import csbase.logic.CommandNotification;
import csbase.logic.CommandSubmission;
import csbase.logic.Priority;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.remote.ClientRemoteLocator;

/**
 * Classe base para executores de comandos.
 *
 * @author Tecgraf
 */
public abstract class AbstractCommandExecutor implements Serializable {

  /**
   * Identificador do comando que foi executado
   */
  private String commandId;

  /**
   * Indica se um e-mail de aviso ser enviado ao trmino do comando
   */
  private boolean mailAtEnd;

  /**
   * Listeners interessados na solicitao de um comando
   */
  private final List<CommandRequestedListener> listeners;

  /**
   * Construtor.
   */
  public AbstractCommandExecutor() {
    this.listeners = new LinkedList<CommandRequestedListener>();
    this.mailAtEnd = false;
  }

  /**
   * Executa um algoritmo a partir do seu configurador devidamente preenchido.
   *
   * @param configurator configurador do algoritmo, que j possui os parmetros
   *        inicializados
   * @param description descrio do comando
   * @param sgaServerName nome do servidor do SGA a ser usado para execuo. Se
   *        for null, indica que a seleo do servidor ser automtica.
   * @param owner janela-me da ao.
   * @param observers Observadores remotos do comando executado
   *
   * @return retorna true, se o comando foi executado com sucesso, caso
   *         contrrio, retorna false
   * @throws RemoteException caso tenha havido falha na chamada remota
   * @throws OperationFailureException caso o escalonador de comandos no esteja
   *         inicializado
   */
  protected String executeCommand(AlgorithmConfigurator configurator,
    String description, String sgaServerName, Window owner,
    CommandObserver... observers) throws RemoteException,
    OperationFailureException {
    return executeCommand(configurator, description, sgaServerName,
      Priority.ROOT, owner, observers);
  }

  /**
   * Executa um algoritmo a partir do seu configurador devidamente preenchido.
   *
   * @param configurator configurador do algoritmo, que j possui os parmetros
   *        inicializados
   * @param description descrio do comando
   * @param sgaServerName nome do servidor do SGA a ser usado para execuo. Se
   *        for null, indica que a seleo do servidor ser automtica.
   * @param priority indica a prioridade do comando a ser executado.
   * @param owner janela-me da ao.
   * @param observers Observadores remotos do comando executado
   *
   * @return retorna true, se o comando foi executado com sucesso, caso
   *         contrrio, retorna false
   * @throws RemoteException caso tenha havido falha na chamada remota
   * @throws OperationFailureException caso o escalonador de comandos no esteja
   *         inicializado
   */
  protected String executeCommand(AlgorithmConfigurator configurator,
    String description, String sgaServerName, Priority priority, Window owner,
    CommandObserver... observers) throws RemoteException,
    OperationFailureException {
    return executeRemoteCommand(makeCommandSubmission(configurator,
      description, sgaServerName, priority), observers);
  }

  /**
   * Monta a configurao da submisso do comando, a partir de um configurador
   * de algoritmos que j deve estar com seus parmetros inicializados.
   *
   * @param configurator Configurador do algoritmo.
   * @param description Descrio do comando.
   * @param sgaServerName SGA em que o algoritmo vai ser executado.
   * @param priority indica a prioridade do comando a ser executado.
   * @return A configurao da submisso do comando.
   */
  private CommandSubmission makeCommandSubmission(
    AlgorithmConfigurator configurator, String description,
    String sgaServerName, Priority priority) {
    Object projectId = DesktopFrame.getInstance().getProject().getId();
    String clientName = Client.getInstance().getClientName();
    CommandSubmission submission =
      new CommandSubmission(configurator, projectId, clientName);
    submission.configureSimpleExecution(sgaServerName);
    submission.setMailAtEnd(getMailAtEnd());
    submission.setDescription(description);
    submission.setPriority(priority);
    return submission;
  }

  /**
   * Dispara a execuo remota do comando. Deve-se ter o cuidado de chamar esse
   * mtodo em uma <code>RemoteTask</code>.
   *
   * @param submission informaes do comando a ser submetido para execuo
   * @param observers Observadores remotos do comando
   * @return true
   *
   * @throws RemoteException caso tenha havido falha na chamada remota
   * @throws OperationFailureException caso o escalonador de comandos no esteja
   *         inicializado
   *
   */
  private String executeRemoteCommand(final CommandSubmission submission,
    CommandObserver... observers) throws RemoteException,
    OperationFailureException {
    if (ClientRemoteLocator.schedulerService != null) {
      Set<CommandInfo> commands =
        ClientRemoteLocator.schedulerService.submitCommand(submission);
      if (commands == null) {
        throw new OperationFailureException(LNG
          .get("AbstractCommandExecutor.error.submit"));
      }
      String id = commands.iterator().next().getId();
      setCommandId(id);
      if (observers.length > 0) {
        MessageProxy.addListener(new CommandRemoteListener(id, observers),
          CommandNotification.class);
      }
      notifyCommandRequestedListeners(commands);
      return id;
    }
    else {
      throw new OperationFailureException(LNG
        .get("AbstractCommandExecutor.exception.no_scheduler"));
    }
  }

  /**
   * Executa o comando de maneira interativa com o usurio, abrindo a aplicao
   * Executor de Algoritmos somente com o algoritmo de interesse carregado.
   * Dessa maneira, o usurio pode interagir tanto na parametrizao do
   * algoritmo, quanto na submisso do comando para execuo.
   *
   * @param configurator configurador do algoritmo, que j possui os parmetros
   *        inicializados
   * @param description descrio do comando
   * @param owner janela-me da ao.
   * @return retorna true, se o comando foi executado com sucesso, caso
   *         contrrio, retorna false
   */
  public boolean executeInterativeCommand(AlgorithmConfigurator configurator,
    String description, Window owner) {
    try {
      ExecutorFrame executorFrame = new ExecutorFrame("executor");
      executorFrame.enableLoadAlgorithms(false);
      executorFrame.showCommandRequestedInfo(false);
      ApplicationManager applicationManager = ApplicationManager.getInstance();
      applicationManager.runApplication(executorFrame);

      executorFrame.addCommandRequestedListener(new CommandRequestedListener() {
        @Override
        public void commandsWereRequested(Set<CommandInfo> submittedCommands) {
          if (!submittedCommands.isEmpty()) {
            Iterator<CommandInfo> iterator = submittedCommands.iterator();
            setCommandId(iterator.next().getId());
            notifyCommandRequestedListeners(submittedCommands);
          }
        }
      });
      executorFrame.setSingleAlgorithm(configurator.getAlgorithmName(),
        new AlgorithmVersionId[] { configurator.getAlgorithmVersionId() });
      executorFrame.setCommandDescription(description);
      executorFrame.showConfigurator(configurator.getAlgorithmName(),
        configurator.getParameterValuesByName());
      return true;
    }
    catch (ApplicationException e) {
      StandardErrorDialogs.showExceptionDialog(owner, owner.getName(), e);
      return false;
    }
  }

  /**
   * Notifica os listeners de um comando solicitado.
   *
   * @param submittedCommands os comandos solicitados
   */
  protected void notifyCommandRequestedListeners(
    Set<CommandInfo> submittedCommands) {
    for (CommandRequestedListener listener : listeners) {
      listener.commandsWereRequested(submittedCommands);
    }
  }

  /**
   * Adiciona um listener para um comando solicitado.
   *
   * @param listener listener de um comando solicitado
   */
  public void addCommandRequestedListener(CommandRequestedListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  /**
   * Obtm o identificador do comando.
   *
   * @return o identicador do comando
   */
  public Object getCommandId() {
    return commandId;
  }

  /**
   * Estabelece o identificador do comando. Um comando pode ser submetido
   * programaticamente para execuo sem a interferncia do usurio, ou tambm
   * pelo dilogo de execuo de comandos.
   *
   * @param commandId identificador do comando
   */
  protected void setCommandId(String commandId) {
    this.commandId = commandId;
  }

  /**
   * Define se o usurio deve recever um email quando o comando terminar.
   *
   * @param mailAtEnd true se o usurio deve receber o email, false caso
   *        contrrio
   */
  public void setMailAtEnd(boolean mailAtEnd) {
    this.mailAtEnd = mailAtEnd;
  }

  /**
   * Indica se deve ser enviado um e-mail ao final da execuo do comando.
   *
   * @return verdadeiro se dever ser enviado um e-mail de notificao de final
   *         de comando, ou falso caso contrrio.
   */
  public boolean getMailAtEnd() {
    return mailAtEnd;
  }

}
