/*
 * $Id$
 */
package csbase.client.facilities.commandtable;

import java.awt.Window;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Vector;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.algorithms.commands.cache.CommandsFilter;
import csbase.client.algorithms.commands.cache.events.AbstractCacheUpdatedEventListener;
import csbase.client.desktop.RemoteTask;
import csbase.client.facilities.monitoringtable.MonitoringTableEvent;
import csbase.client.facilities.monitoringtable.MonitoringTableObserver;
import csbase.client.util.CodeBlockLog;
import csbase.logic.CommandInfo;
import csbase.logic.MonitoringSet;

/**
 * A classe <code>AbstractCommandMonitoringTable</code> implementa
 * funcionalidades bsicas de uma tabela de monitorao de comandos.
 * Caractersticas:<br>
 * <ul>
 * <li> observvel. Notifica os observadores quando  atualizada.</li>
 * <li>Ela se atualiza atravs de um observador que instala na
 * {@link CommandsCache}.</li>
 * <li>Ela mantm a ordenao e as selees originais mesmo aps a atualizao.</li>
 * <li>Cria a tabela com descritoes de coluna recebidos do mtodo abstrato
 * getTableColumnAttributes()</li>
 * </ul>
 * -  observvel. Notifica os observadores quando  atualizada.
 */
public abstract class AbstractCommandMonitoringTable extends SortableTable {

  /**
   * Logger da classe.
   */
  private static Logger LOGGER = Logger
    .getLogger(AbstractCommandMonitoringTable.class.getName());

  /**
   * Flag indicando se o sistema de monitoraco por listener est ativo.
   */
  private final AtomicBoolean running;
  /**
   * Listener de comandos.
   */
  private final AbstractCacheUpdatedEventListener listener;
  /**
   * Modelo para a tabela.
   */
  protected ObjectTableModel<CommandMonitoringSet> model;
  /**
   * Coleo de observadores da tabela
   */
  private final Collection<MonitoringTableObserver> observers;

  /**
   * Indica se devemos mostrar a coluna 'Progresso'
   */
  protected boolean showProgressColumn;

  /**
   * Tempo mnimo que deve ser garantido ao usurio para editar uma clula sem
   * que essa edio seja cancelada pela atualizao da tabela. O valoe padro 
   * 2 minutos. O valor zero que faz com que a edio das clulas funcione como
   * nas demais tabelas de java.
   */
  private long cellsEditingMinTime;
  /**
   * Utilizado pelo mtodo {@link #reloadTable(Collection)} para armazenar o
   * primeiro momento em que se tomou conhecimento que uma clula estava sendo
   * editada. O valor -1 indica que no h clula sendo editada.
   */
  private long cellEditingBeginTime;

  /**
   * Construtor que constri por default uma tabela de seleo nica.
   * 
   * @param filter filtro que indica que comandos devem aparecer nesta tabela.
   */
  public AbstractCommandMonitoringTable(CommandsFilter filter) {
    this(ListSelectionModel.SINGLE_SELECTION, filter, false);
  }

  /**
   * Construtor que permite escolher o modo de seleo da tabela.
   * 
   * @param selectionMode Indica o modo de seleo da tabela.
   * @param filter filtro que indica que comandos devem aparecer nesta tabela.
   * @see javax.swing.JTable#setSelectionMode(int)
   */
  public AbstractCommandMonitoringTable(int selectionMode, CommandsFilter filter) {
    this(selectionMode, filter, false);
  }

  /**
   * Construtor que permite escolher o modo de seleo da tabela.
   * 
   * @param selectionMode Indica o modo de seleo da tabela.
   * @param filter filtro que indica que comandos devem aparecer nesta tabela.
   * @param showProgress - TRUE indica para mostrarmos a coluna 'progresso' na
   *        tabela de algoritmos em execuo, FALSE para omitir tal coluna.
   *        Valor obtido da propriedade de mesmo nome.
   * @see javax.swing.JTable#setSelectionMode(int)
   */
  public AbstractCommandMonitoringTable(int selectionMode,
    CommandsFilter filter, boolean showProgress) {
    this.showProgressColumn = showProgress;
    this.running = new AtomicBoolean(false);
    this.cellsEditingMinTime = 120000; // 120000 = 2 minutos.
    this.cellEditingBeginTime = -1; // -1 = no h celulas sendo editadas.    
    this.listener = new AbstractCacheUpdatedEventListener(filter) {
      @Override
      public void eventFired(final Collection<CommandInfo> commands) {
        reloadTable(commands);
      }

      @Override
      protected void eventInterrupted(Exception exception, String description) {
        notifyObservers(new MonitoringTableEvent(
          MonitoringTableEvent.REMOTE_ERROR));
      }
    };
    this.observers = new HashSet<MonitoringTableObserver>();
    this.setSelectionMode(selectionMode);
    this.model =
      new ObjectTableModel<CommandMonitoringSet>(
        new Vector<CommandMonitoringSet>(), getObjectTableProvider());
    this.setModel(model);
    this.setRowSelectionAllowed(true);

    TableColumnModel colModel = this.getColumnModel();
    for (int inx = 0; inx < colModel.getColumnCount(); inx++) {
      TableColumn column = colModel.getColumn(inx);
      TableCellRenderer renderer = createTableCellRenderer(inx);
      if (null != renderer) {
        column.setCellRenderer(renderer);
      }
      TableCellEditor editor = createTableCellEditor(inx);
      if (null != editor) {
        column.setCellEditor(editor);
      }
    }
  }

  /**
   * Altera o filtro que indica que tipo de comandos devem aparecer na tabela.
   * 
   * @param filter o filtro.
   */
  public void setFilter(CommandsFilter filter) {
    listener.setFilter(filter);
  }

  /**
   * <p>
   * Atribui o tempo mnimo, em milisegundos, para a edio de uma clula.
   * </p>
   * <p>
   * Sempre que s dados da tabela so atualizados, qualquer edio de clula
   * que no tenha sido finalizada  cancelada. O tempo mnimo faz com que as
   * atualizaes sejam ignoradas por aquele perodo, dando uma chance maior do
   * usurio concluir a edio.
   * </p>
   * 
   * @param cellsEditingMinTime o tempo mnimo, em milisegundos, para a edio
   *        de uma clula. O vaor padro  2 minutos. O valor zero que faz com
   *        que a edio das clulas funcione como nas demais tabelas de java.
   */
  public void setCellsEditingMinTime(long cellsEditingMinTime) {
    this.cellsEditingMinTime = cellsEditingMinTime;
  }

  /**
   * Inicia a atualizao.
   */
  public void start() {
    if (running.compareAndSet(false, true)) {
      initializeData(this.listener.getFilter());
      CommandsCache.getInstance().addEventListener(listener);
    }
  }

  /**
   * Interrompe a atualizao.
   */
  public void stop() {
    if (running.compareAndSet(true, false)) {
      CommandsCache.getInstance().removeEventListener(this.listener);
    }
  }

  /**
   * Retorna um objeto representando a linha especificada.
   * 
   * @param modelRow ndice da linha no modelo da tabela.
   * @return objeto representando a linha especificada.
   */
  public CommandMonitoringSet getMonitoringSet(int modelRow) {
    return model.getRow(modelRow);
  }

  /**
   * Adiciona um observador  lista de observadores da tabela.
   * 
   * @param o observador.
   */
  public void addObserver(MonitoringTableObserver o) {
    observers.add(o);
  }

  /**
   * Remove um observador da lista de observadores da tabela.
   * 
   * @param o observador.
   */
  public void removeObserver(MonitoringTableObserver o) {
    observers.remove(o);
  }

  /**
   * Retorna um objeto contendo informaes sobre o comando correspondente 
   * linha especificada.
   * 
   * @param viewRow ndice de linha na viso tabela.
   * 
   * @return objeto contendo informaes sobre um comando.
   */
  public CommandInfo getCommandInfo(int viewRow) {
    int modelRow = convertRowIndexToModel(viewRow);
    return getMonitoringSet(modelRow).getCommandInfo();
  }

  /**
   * Obtm os comandos selecionados.
   * 
   * @return Um array contendo os comandos selecionados.
   */
  public CommandInfo[] getSelectedCommands() {
    int[] rows = getSelectedRows();
    CommandInfo[] commands = new CommandInfo[rows.length];
    for (int inx = 0; inx < rows.length; inx++) {
      commands[inx] = getCommandInfo(rows[inx]);
    }

    return commands;
  }

  /**
   * Cria renderer da coluna
   * 
   * @param columnIndex ndice da coluna.
   * @return {@code null}
   */
  protected TableCellRenderer createTableCellRenderer(int columnIndex) {
    return null;
  }

  /**
   * Cria editor da coluna.
   * 
   * @param columnIndex ndice da coluna.
   * @return {@code null}
   */
  protected TableCellEditor createTableCellEditor(int columnIndex) {
    return null;
  }

  /**
   * Mtodo abstrato para obter o provedor de dados/metadados da tabela.
   * 
   * @return provedor de dados/metadados da tabela.
   */
  protected abstract ObjectTableProvider getObjectTableProvider();

  /**
   * Cria a lista de comandos monitorados.
   * 
   * @param commands comandos
   * @return a lista
   */
  private List<CommandMonitoringSet> createMonitoringSets(
    Collection<CommandInfo> commands) {
    List<CommandMonitoringSet> monitoringSets =
      new ArrayList<CommandMonitoringSet>();
    for (CommandInfo command : commands) {
      monitoringSets.add(new CommandMonitoringSet(command));
    }

    return monitoringSets;
  }

  /**
   * Inicializa dados.
   * 
   * @param filter filtro
   */
  private void initializeData(final CommandsFilter filter) {
    RemoteTask<Collection<CommandInfo>> task =
      new RemoteTask<Collection<CommandInfo>>() {
        @Override
        protected void performTask() throws Exception {
          CodeBlockLog log =
            new CodeBlockLog(Level.FINER, "obtendo dados da tabela");
          LOGGER.log(log.begin());

          Collection<CommandInfo> data =
            CommandsCache.getInstance().getCommands(filter);

          LOGGER.log(log.finished());

          setResult(data);
        }
      };
    Window window = SwingUtilities.windowForComponent(this);
    String taskTitle = LNG.get("AbstractCommandMonitoringTable.title");
    String taskMessage = LNG.get("AbstractCommandMonitoringTable.message");
    //TODO DEBUG
    long d_dur = System.currentTimeMillis();
    if (task.execute(window, taskTitle, taskMessage)) {
      CodeBlockLog log = new CodeBlockLog(Level.FINER, "recarregando a tabela");
      LOGGER.log(log.begin());

      reloadTable(task.getResult());

      LOGGER.log(log.finished());
    }
  }

  /**
   * <p>
   * Faz recarga da tabela com novos dados.
   * </p>
   * <p>
   * Essa recarga poder ser ignorada caso o usurio esteja editando uma clula.
   * </p>
   * 
   * @param commands comandos
   * 
   * @see #setCellsEditingMinTime(long)
   */
  private void reloadTable(Collection<CommandInfo> commands) {
    final List<CommandMonitoringSet> monitoringSets =
      createMonitoringSets(commands);
    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        /*
         * Verifica se alguma clula est sendo editada e o tempo mnimo de
         * edio  maior que zero.
         */
        if (getCellEditor() != null && cellsEditingMinTime > 0) {
          // Se sim...

          // Se  o primeiro update desde que comeou a edio, ...
          if (cellEditingBeginTime == -1) {
            /*
             * ... comea a contar o tempo, guardando agora como tempo de incio
             * da edio e cancela a recarga da tabela para que o usurio possa
             * continuar editando.
             */
            cellEditingBeginTime = System.currentTimeMillis();
            return;
          }
          /*
           * Caso contrrio, se o tempo de edio ainda no alcanou o tempo
           * mnimo requisitado, ...
           */
          if (System.currentTimeMillis() - cellEditingBeginTime <= cellsEditingMinTime) {
            /*
             * ... cancela a recarga da tabela para que o usurio possa
             * continuar editando.
             */
            return;
          }
          /*
           * Neste ponto o tempo de edio  maior que o mnimo que deve ser
           * garantido ao usurio. A recarga ser executada, cancelando a edio
           * do usurio.
           */
        }
        // Se  a primeira recarga aps uma edio...
        if (cellEditingBeginTime != -1) {
          // ... ignora ela e reseta o tempo de edio. 
          /*
           * Isso diminui a chance de aparecer o valor antigo na tabela, pois
           * essa nova carga de dados pode ter sido criada antes da alterao
           * ter ido apra o servidor.
           */
          cellEditingBeginTime = -1;
          return;
        }

        // Recarrega a tabela com os novos dados. 
        setTableData(monitoringSets);
        notifyObservers(new MonitoringTableEvent(MonitoringTableEvent.REFRESH));
      }
    });
  }

  /**
   * Atualiza os dados da tabela de monitorao. Este mtodo deve ser chamado na
   * thread no Swing.
   * 
   * @param data os dados a serem apresentados na tabela.
   */
  private void setTableData(List<CommandMonitoringSet> data) {
    Vector<String> selectedKeys = null;

    // Guarda as chaves das linhas atualmente selecionadas na tabela.
    if (data.size() > 0 && model.getRowCount() > 0) {
      selectedKeys = new Vector<String>();
      int[] selectedRows = getSelectedRows();
      if ((selectedRows != null) && (selectedRows.length > 0)) {
        for (int i = 0; i < selectedRows.length; i++) {
          int tableIndex = selectedRows[i];
          int modelIndex = convertRowIndexToModel(tableIndex);
          if (modelIndex >= 0) {
            MonitoringSet selectedSet = getMonitoringSet(modelIndex);
            selectedKeys.add(selectedSet.getKey());
          }
        }
      }
    }

    /*
     * Atualiza as linhas da tabela. O mtodo setRows j chama o mtodo
     * fireTableChanged(new TableModelEvent(this)) que repassa a notificao
     * para todos os listeners da tabela informando que todo contedo da tabela
     * foi modificada.
     */
    model.setRows(data);

    if (selectedKeys == null) {
      return;
    }

    // Procura os ndices correspondentes  antiga seleo nas linhas
    // da tabela aps a atualizao.
    for (String selectedKey : selectedKeys) {
      for (int inx = 0; inx < data.size(); inx++) {
        MonitoringSet newSet = data.get(inx);
        if (newSet.getKey().equals(selectedKey)) {
          int viewIndex = convertRowIndexToView(inx);
          // Seleciona as linhas de ndices encontrados para
          // restaurar a seleo antes da atualizao.
          addRowSelectionInterval(viewIndex, viewIndex);
          break;
        }
      }
    }
  }

  /**
   * Notifica os observadores da tabela da ocorrncia de um evento.
   * 
   * @param event evento a ser notificado.
   */
  private void notifyObservers(MonitoringTableEvent event) {
    for (MonitoringTableObserver obs : observers) {
      obs.update(event);
    }
  }
}
