package csbase.client.facilities.configurabletable;

import java.awt.Dimension;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.SortOrder;
import javax.swing.SwingConstants;

import tecgraf.javautils.core.filter.IFilter;
import csbase.client.facilities.configurabletable.column.IConfigurableColumn;
import csbase.client.facilities.configurabletable.model.Column;
import csbase.client.facilities.configurabletable.model.Columns;
import csbase.client.facilities.configurabletable.model.Config;
import csbase.client.facilities.configurabletable.model.Filter;
import csbase.client.facilities.configurabletable.model.Filters;
import csbase.client.facilities.configurabletable.model.Table;
import csbase.client.facilities.configurabletable.model.Table.ColumnRef;
import csbase.client.facilities.configurabletable.model.Tables;
import csbase.client.facilities.configurabletable.stringprovider.IStringProvider;
import csbase.client.facilities.configurabletable.table.ConfigurableTable;

/**
 * Fbrica de tabelas configurveis. <br/>
 * 
 * Essa fbrica permite a criao de tabelas configurveis a partir de um objeto
 * que representa uma configurao. Este objeto, do tipo {@link Config},  um
 * modelo de pojos que, basicamente, representa quais tabelas essa fbrica
 * disponibiliza. Alm disso, esse objeto de configurao pode ser obtido via
 * leitura de um XML e tambm define diversas propriedades das tabelas, por
 * exemplo, filtro, colunas, visibilidade das colunas etc.
 * 
 * @see ConfigurableTable ITableFactory IConfigurableColumn IFilter Config
 * 
 * @author Tecgraf
 */
public class ConfigurableTableFactory {

  /**
   * Configurao das tabelas.
   */
  private Config tablesConfig;

  /**
   * Provedor de strings.
   */
  private IStringProvider stringProvider;

  /**
   * Mapa auxiliar de tabelas por ses respectivos identificadores.
   */
  private Map<String, Table> tablesById;

  /**
   * Mapa auxiliar de colunas por seus respectivos identificadores.
   */
  private Map<String, Column> columnsById;

  /**
   * Mapa auxiliar de filtros por seus respectivos identificadores.
   */
  private Map<String, Filter> filtersById;

  /**
   * Construtor.
   * 
   * @param stringProvider - usado para internacionalizar o nome das colunas.
   */
  private ConfigurableTableFactory(IStringProvider stringProvider) {
    this.stringProvider = stringProvider;

    if (this.stringProvider == null) {
      this.stringProvider = new IStringProvider() {
        @Override
        public String getString(String key) {
          return key;
        }
      };
    }
  }

  /**
   * Construtor.
   * 
   * @param config - configurao de tabelas.
   * @param stringProvider - usado para internacionalizar o nome das colunas.
   */
  public ConfigurableTableFactory(Config config, IStringProvider stringProvider) {
    this(stringProvider);
    this.tablesConfig = config;

    buildMaps();
  }

  /**
   * Mtodo privado que instncia e popula os mapas auxiliares de tabelas,
   * colunas e filtros.
   */
  private void buildMaps() {

    this.tablesById = new HashMap<String, Table>();
    this.columnsById = new HashMap<String, Column>();
    this.filtersById = new HashMap<String, Filter>();

    if (tablesConfig != null) {
      Columns columns = this.tablesConfig.getColumns();
      for (Column column : columns.getColumn()) {
        columnsById.put(column.getId(), column);
      }

      Tables tables = this.tablesConfig.getTables();
      for (Table table : tables.getTable()) {
        tablesById.put(table.getId(), table);
      }

      Filters filters = this.tablesConfig.getFilters();
      for (Filter filter : filters.getFilter()) {
        filtersById.put(filter.getId(), filter);
      }
    }
  }

  /**
   * Obtm a tabela cujo identificador  <code>tableId</code>.
   * 
   * @param tableId - identificador da tabela.
   * @return tabela vazia.
   */
  public ConfigurableTable<?> getTable(String tableId) {

    Table tableConf = tablesById.get(tableId);

    if (tableConf != null) {
      String className = tableConf.getDatatype();

      Class<?> dataType;
      try {
        dataType = Class.forName(className);
      }
      catch (ClassNotFoundException e) {
        throw new RuntimeException(
          "Classe que representa os objetos da tabela no foi encontrada: "
            + className);
      }

      return getTable(tableId, dataType);
    }

    return null;
  }

  /**
   * Obtm a tabela cujo identificador  <code>tableId</code>. <br/>
   * 
   * @param <T> - tipo dos dados exibidos na tabela.
   * 
   * @param tableId - identificador da tabela.
   * @param dataType - tipo dos dados que so exibidos na tabela.
   * @return tabela cujo id  <code>tableId</code>.
   */
  public <T> ConfigurableTable<T> getTable(String tableId, Class<T> dataType) {

    Table tableConf = tablesById.get(tableId);

    ConfigurableTable<T> table = null;

    if (tableConf != null) {

      // criao das colunas
      List<IConfigurableColumn<T>> columns =
        new ArrayList<IConfigurableColumn<T>>();

      for (Table.ColumnRef columnRef : tableConf.getColumnRef()) {
        Column columnConf = columnsById.get(columnRef.getId());
        columns.add(createColumn(dataType, columnConf, columnRef.isVisible()));
      }

      table = new ConfigurableTable<T>(tableId, columns, null);

      // atribui um filtro a tabela, caso seja definido
      if (tableConf.getFilter() != null) {
        String filterId = tableConf.getFilter();
        Filter filterConf = filtersById.get(filterId);
        IFilter<T> filter = createFilter(dataType, filterConf);
        table.setFilter(filter);
      }

      // verifica se h ordenao default
      if (tableConf.getOrderby() != null) {
        // obtm o ndice da coluna a ser ordenada
        String[] sortElements = tableConf.getOrderby().split("\\|");
        if (sortElements.length == 2) {
          int columnIndex = 0;
          for (int i = 0; i < tableConf.getColumnRef().size(); i++) {
            ColumnRef columnRef = tableConf.getColumnRef().get(i);
            if (columnRef.getId().equals(sortElements[0])) {
              columnIndex = i;
              break;
            }
          }
          //obtm a ordem da ordenao
          if ("desc".equals(sortElements[1])) {
            table.sort(columnIndex, SortOrder.DESCENDING);
          }
          else {
            table.sort(columnIndex, SortOrder.ASCENDING);
          }
        }
      }

      // atribui o tamanho desejvel caso seja definido
      int height =
        (tableConf.getHeight() != null) ? tableConf.getHeight() : 300;
      int width = (tableConf.getWidth() != null) ? tableConf.getWidth() : 500;
      table.setPreferredSize(new Dimension(width, height));
    }

    return table;
  }

  /**
   * Verifica se existe uma tabela cujo identificador  <code>tableId</code>.
   * 
   * @param tableId - identificador da tabela.
   * @return <code>true</code> se a tabela existe, <code>false</code> caso
   *         contrrio.
   */
  public boolean hasTable(String tableId) {
    return tablesById.get(tableId) != null;
  }

  /**
   * Mtodo auxiliar que cria um filtro.
   * 
   * @param <T> - tipo dos dados que o filtro atua.
   * @param dataType - classe dos objetos que representam as linhas da tabela.
   * @param filterConf - modelo do filtro.
   * @return instncia do filtro.
   */
  @SuppressWarnings("unchecked")
  private <T> IFilter<T> createFilter(Class<T> dataType, Filter filterConf) {
    IFilter<T> filter = null;

    Class<?> clazz;
    try {
      clazz = Class.forName(filterConf.getClazz());

      if (filterConf.getParam().size() > 0) {
        Constructor<?> construtor = clazz.getConstructor(Map.class);

        Map<String, String> params = new HashMap<String, String>();
        for (Filter.Param param : filterConf.getParam()) {
          params.put(param.getName(), param.getValue());
        }

        filter = (IFilter<T>) construtor.newInstance(params);

      }
      else {
        Constructor<?> construtor = clazz.getConstructor();
        filter = (IFilter<T>) construtor.newInstance();
      }
    }
    catch (Exception e) {
      throw new RuntimeException("Erro ao construir filtro com id:"
        + filterConf.getId(), e);
    }

    return filter;
  }

  /**
   * Mtodo auxiliar que cria uma coluna.
   * 
   * @param <T> - tipo dos objetos que a coluna exibe.
   * @param dataType - tipo dos objetos que a tabela exibe.
   * @param columnConf - modelo da coluna.
   * @param visibility - visibilidade da coluna.
   * @return instncia da coluna.
   */
  @SuppressWarnings("unchecked")
  private <T> IConfigurableColumn<T> createColumn(Class<T> dataType,
    Column columnConf, boolean visibility) {
    IConfigurableColumn<T> column = null;

    Class<?> clazz;
    try {
      clazz = Class.forName(columnConf.getClazz());
      Constructor<?> construtor =
        clazz.getConstructor(String.class, Boolean.class,
          IStringProvider.class, Integer.class);

      int align = convertColumnAlign(columnConf.getAlign());

      column =
        (IConfigurableColumn<T>) construtor.newInstance(columnConf.getId(),
          visibility, stringProvider, align);

    }
    catch (Exception e) {
      throw new RuntimeException("Erro ao construir coluna com id:"
        + columnConf.getId(), e);
    }

    return column;
  }

  /**
   * Mtodo auxiliar que converte uma string que define o alinhamento de uma
   * coluna especfica em um valor inteiro com a mesma semntica.
   * 
   * @param align - string que define o alinhamento.
   * @return inteiro que define o alinhamento.
   */
  private int convertColumnAlign(String align) {
    if ("left".equals(align)) {
      return SwingConstants.LEFT;
    }
    else if ("right".equals(align)) {
      return SwingConstants.RIGHT;
    }

    return SwingConstants.CENTER;
  }
}
