package csbase.client.facilities.monitoringtable;

import java.rmi.RemoteException;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Vector;

import javax.swing.ListSelectionModel;

import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.remote.srvproxies.SGAProxy;
import csbase.logic.MonitoringSet;

/**
 * A classe <code>MonitoringTable</code> implementa funcionalidades bsicas de
 * uma tabela de monitorao de informaes remotas.  baseada em
 * <code>JTable</code> Exemplos de sub-classes: SGATable, AlgorithmTable
 * Caractersticas: -  observvel. Notifica os observadores quando  atualizada
 * - Executa uma Thread (UpdateMonTableThread), que atualiza periodicamente
 * (apenas quando necessrio) a tabela com os dados recebidos do mtodo abstrato
 * getNewMonitoringSetVector(), mantendo a ordenao e as selees originais. -
 * Cria a tabela com descritoes de coluna recebidos do mtodo abstrato
 * getTableColumnAttributes()
 */
public abstract class MonitoringTable extends SortableTable {
  /** Thread responsvel pela atualizao das informaes. */
  protected UpdateMonTableThread updateThread = null;

  /** Modelo para a tabela */
  protected ObjectTableModel<MonitoringSet> model;

  /** Coleo de observadores da tabela */
  private final Collection<MonitoringTableObserver> observers;

  /**
   * Adiciona um observador  lista de observadores da tabela.
   * 
   * @param o observador.
   */
  public void addObserver(MonitoringTableObserver o) {
    observers.add(o);
  }

  /**
   * Remove um observador da lista de observadores da tabela.
   * 
   * @param o observador.
   */
  public void removeObserver(MonitoringTableObserver o) {
    observers.remove(o);
  }

  /**
   * Notifica os observadores da tabela da ocorrncia de um evento.
   * 
   * @param event evento a ser notificado.
   */
  private void notifyObservers(MonitoringTableEvent event) {
    for (MonitoringTableObserver obs : observers) {
      obs.update(event);
    }
  }

  /**
   * Acorda a thread responsvel por manter a tabela atualizada
   */
  public void wakeupThread() {
    updateThread.wakeupThread();
  }

  /**
   * Termina a thread responsvel por manter a tabela atualizada
   */
  public void stopThread() {
    updateThread.stopThread();
  }

  /**
   * Dispara thread encarregada de atualizar periodicamente os dados da tabela.
   */
  public void startThread() {
    updateThread = new UpdateMonTableThread();
    updateThread.start();
  }

  /**
   * Retorna um objeto representando a linha especificada.
   * 
   * @param row ndice de linha da tabela.
   * 
   * @return objeto representando a linha especificada.
   */
  public MonitoringSet getMonitoringSet(int row) {
    return model.getRow(row);
  }

  /**
   * Retorna a referncia para a thread de monitoramento de comandos.
   * 
   * @return updateThread a thread de monitoramento de comandos.
   */
  protected Thread getUpdateThread() {
    return updateThread;
  }

  /**
   * Mtodo abstrato para obter o provedor de dados/metadados da tabela.
   * 
   * @return provedor de dados/metadados da tabela.
   */
  protected abstract ObjectTableProvider getObjectTableProvider();

  /**
   * Mtodo abstrato para obter novos dados da tabela.
   * 
   * @return coleo de novas linhas para a tabela.
   * 
   * @throws RemoteException em caso de falha de acesso RMI ao servio.
   */
  public abstract Vector<MonitoringSet> getNewRows() throws RemoteException;

  /**
   * Mtodo abstrato para invalidar os dados dinmicos da tabela. Usado quando o
   * cliente perde a conexo com o servidor.
   */
  protected abstract void invalidateRows();

  /**
   * Thread responsvel por atualizar os dados da tabela, de acordo com as
   * informaes obtidas do servio remoto.
   */
  private class UpdateMonTableThread extends Thread {
    /** Informaes recm-recuperadas e ainda no exibidas (Vector) */
    private Vector<MonitoringSet> newRows = new Vector<MonitoringSet>();

    /** Flag responsvel por manter o loop da thread */
    private boolean exitUpdateThread = false;

    /**
     * Encerra o loop da thread
     */
    public void stopThread() {
      exitUpdateThread = true;
      try {
        interrupt();
      }
      catch (Throwable t) {
      }
    }

    /**
     * Acorda a thread para que ela faa um novo refresh da tabela.
     */
    public void wakeupThread() {
      try {
        interrupt();
      }
      catch (Throwable t) {
      }
    }

    /**
     * Atualiza os dados da tabela de monitorao. Este mtodo altera o modelo
     * da tabela. Deve ser chamado na thread no swing (EDT).
     */
    private void updateMonTable() {
      final Vector<String> selectedKeys = new Vector<String>();
      synchronized (newRows) {
        int newRowCount = newRows.size();
        int oldRowCount = model.getRowCount();
        if ((newRowCount > 0) && (oldRowCount > 0)) {
          // Guarda as chaves das linhas atualmente selecionadas na tabela.
          int[] selectedRows = getSelectedRows();
          if ((selectedRows != null) && (selectedRows.length > 0)) {
            for (int i = 0; i < selectedRows.length; i++) {
              int tableIndex = selectedRows[i];
              int modelIndex = convertRowIndexToModel(tableIndex);
              if (modelIndex >= 0) {
                MonitoringSet selectedSet = getMonitoringSet(modelIndex);
                selectedKeys.add(selectedSet.getKey());
              }
            }
          }
        }
        /*
         * Atualiza as linhas da tabela. O mtodo setRows j chama o mtodo
         * fireTableChanged(new TableModelEvent(this)) que repassa a notificao
         * para todos os listeners da tabela informando que todo contedo da
         * tabela foi modificada.
         */
        model.setRows(newRows);
      }
      if (selectedKeys.isEmpty()) {
        return;
      }
      // Seleciona as linhas de ndices encontrados no passo anterior, para
      // restaurar a seleo antes da atualizao.
      SwingThreadDispatcher.invokeLater(new Runnable() {
        @Override
        public void run() {
          // Procura os ndices correspondentes  antiga seleo nas linhas
          // da tabela aps a atualizao.
          List<MonitoringSet> rows = model.getRows();
          int rowCount = rows.size();
          for (String selectedKey : selectedKeys) {
            for (int modelIndex = 0; modelIndex < rowCount; modelIndex++) {
              MonitoringSet set = rows.get(modelIndex);
              String key = set.getKey();
              if (selectedKey.equals(key)) {
                int viewIndex = convertRowIndexToView(modelIndex);
                addRowSelectionInterval(viewIndex, viewIndex);
                break;
              }
            }
          }
        }
      });
    }

    /**
     * Recupera dados atualizados de 5 em 5 segundos. Caso estes sejam
     * diferentes dos atualmente exibidos na tela, atualiza a tabela.
     */
    @Override
    public void run() {
      /* Obtem o intervalo de atualizao e as informaes iniciais do comando */
      int updateIntervalAsInt = SGAProxy.getCommandsUpdateInterval();
      if (updateIntervalAsInt == -1) {
        return;
      }
      long updateInterval = (long) updateIntervalAsInt * (long) 1000;
      while (!exitUpdateThread) {
        try {
          /* _newRows contem os novos valores da tabela */
          Vector<MonitoringSet> _newRows = getNewRows();
          synchronized (newRows) {
            newRows = _newRows;
          }
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              updateMonTable();
              notifyObservers(new MonitoringTableEvent(
                MonitoringTableEvent.REFRESH));
            }
          });
        }
        catch (RemoteException e) {
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              invalidateRows();
              model.fireTableDataChanged();
              notifyObservers(new MonitoringTableEvent(
                MonitoringTableEvent.REMOTE_ERROR));
            }
          });
          ClientRemoteMonitor.getInstance().invalidate();
        }
        try {
          Thread.sleep(updateInterval);
        }
        catch (InterruptedException ex) {
        }
      }
    }
  }

  /**
   * Construtor que constri por default uma tabela de seleo nica.
   */
  public MonitoringTable() {
    this(ListSelectionModel.SINGLE_SELECTION);
  }

  /**
   * Construtor que permite escolher o modo de seleo da tabela.
   * 
   * @param selectionMode Indica o modo de seleo da tabela.
   * 
   * @see javax.swing.JTable#setSelectionMode(int)
   */
  public MonitoringTable(int selectionMode) {
    super();
    observers = new HashSet<MonitoringTableObserver>();
    setSelectionMode(selectionMode);
    model =
      new ObjectTableModel<MonitoringSet>(new Vector<MonitoringSet>(),
        getObjectTableProvider());
    setModel(model);
    setRowSelectionAllowed(true);
    startThread();
  }
}
