/*
 * RoleDefinitionDialog.java
 * 
 * $Author: oikawa $ $Revision: 150777 $ - $Date: 2007-11-22 19:04:50 -0200
 * (Thu, 22 Nov 2007) $
 */
package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.util.Arrays;
import java.util.List;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.selector.ContainerSelection;
import tecgraf.javautils.gui.selector.ContainerSelectionListener;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.remote.srvproxies.PermissionProxy;
import csbase.client.remote.srvproxies.RoleProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.Permission;
import csbase.logic.Role;
import csbase.logic.RoleInfo;

/**
 * A classe <code>RoleDefinitionDialog</code>  responsvel pela criao e
 * alterao de perfis de usurios.
 * 
 * @author Glaucio e Ana
 * @version $Revision: 150777 $
 */
public class RoleDefinitionDialog {
  /** Indica se o perfil est sendo criado ou alterado */
  private boolean isNew;
  /** Janela "pai" */
  private JFrame owner;
  /** Janela de cadastro de perfis */
  private DesktopComponentDialog mainDialog;
  /** Ttulo da janela */
  private String dialogTitle;
  /** Campo nome do perfil */
  private JTextField roleName;
  /** Campo descrio do perfil */
  private JTextField description;
  /** Componente de seleo de permisses */
  private ContainerSelection<Permission> permissionsSelection;
  /** Objeto para persistncia dos dados de perfil fornecidos pelo usurio */
  private Role role;
  /**
   * Boto de ao (ser de incluso ou alterao, dependendo da propriedade
   * isNew)
   */
  private JButton actionButton;

  /**
   * Exibe um dilogo de criao de perfil de usurios.
   */
  public void showCreateDialog() {
    this.isNew = true;
    this.dialogTitle = LNG.get("IAS_ROLE_INCLUSION_TITLE");
    this.mainDialog = new DesktopComponentDialog(owner, dialogTitle);
    Vector<Permission> permissions =
      PermissionProxy.getAllPermissions(mainDialog, dialogTitle,
        LNG.get("IAS_WAITING_ALL_PERMISSIONS"));
    showDialog(null, null, permissions, new Vector<Permission>());
  }

  /**
   * Exibe um dilogo de alterao de perfil de usurios.
   * 
   * @param role perfil a ser atualizado
   */
  public void showUpdateDialog(Role role) {
    this.isNew = false;
    this.dialogTitle = LNG.get("IAS_ROLE_UPDATE_TITLE");
    this.mainDialog = new DesktopComponentDialog(owner, dialogTitle);
    this.role = role;
    Object[] selectedIds = role.getPermissionIds();
    Permission[] selectedPermissions = new Permission[selectedIds.length];
    try {
      for (int i = 0; i < selectedIds.length; i++) {
        Permission permission = Permission.getPermission(selectedIds[i]);
        if (permission != null) {
          selectedPermissions[i] = permission;
        }
        else {
          throw new Exception("No foi possvel carregar a permisso de id "
            + selectedIds[i]);
        }
      }
    }
    catch (Exception e) {
      StandardErrorDialogs.showErrorDialog(mainDialog, LNG.get("ERRO") + " - "
        + dialogTitle, e);
    }
    Vector<Permission> selectedPermissionsVector = new Vector<Permission>();
    selectedPermissionsVector.addAll(Arrays.asList(selectedPermissions));
    Vector<Permission> permissions =
      PermissionProxy.getAllPermissions(mainDialog, dialogTitle,
        LNG.get("IAS_WAITING_ALL_PERMISSIONS"));
    showDialog(role.getName(), role.getDescription(), permissions,
      selectedPermissionsVector);
  }

  /**
   * Cria o dilogo conforme a iterao (criao, alterao).
   * 
   * @param name nome do perfil (alterar) ou null (criar)
   * @param desc descrio do perfil.
   * @param selectablePermissions permisses disponveis para adio
   * @param selectedPermissions permisses selecionadas no perfil
   */
  private void showDialog(String name, String desc,
    Vector<Permission> selectablePermissions,
    Vector<Permission> selectedPermissions) {
    Container cp = mainDialog.getContentPane();
    if (isNew) {
      roleName = new JTextField();
      description = new JTextField();
    }
    else {
      roleName = new JTextField(name);
      roleName.addKeyListener(new KeyAdapter() {
        @Override
        public void keyReleased(KeyEvent e) {
          checkDataChange();
        }
      });
      description = new JTextField(desc);
      description.addKeyListener(new KeyAdapter() {
        @Override
        public void keyReleased(KeyEvent e) {
          checkDataChange();
        }
      });
    }
    JComponent[][] nameComp =
      new JComponent[][] { { new JLabel(LNG.get("IAS_ROLE_NAME")), roleName },
          { new JLabel(LNG.get("IAS_ROLE_DESCRIPTION")), description } };
    cp.add(GUIUtils.createBasicGridPanel(nameComp), BorderLayout.NORTH);
    // Painel para seleo de permisses
    JPanel selectionPanel =
      createSelectionPanel(selectablePermissions, selectedPermissions);
    cp.add(selectionPanel, BorderLayout.CENTER);
    // Boto de "ao" (incluso ou modificao)
    if (isNew) {
      actionButton = new JButton(LNG.get("IAS_ROLE_INCLUDE"));
    }
    else {
      actionButton = new JButton(LNG.get("IAS_ROLE_UPDATE"));
      actionButton.setEnabled(false);
    }
    actionButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (validateFields()) {
          if (isNew) {
            createRole();
          }
          else {
            modifyRole();
            mainDialog.close();
          }
        }
        else {
          StandardErrorDialogs.showErrorDialog(mainDialog, dialogTitle,
            LNG.get("IAS_FILL_ALL_FIELDS"));
        }
      }
    });

    JButton closeButton = new JButton(LNG.get("IAS_CLOSE"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (isNew || !hasChanged()) {
          mainDialog.close();
          return;
        }
        Object[] options =
          { LNG.get("IAS_ROLE_UPDATE"), LNG.get("IAS_CANCEL") };
        int selected =
          JOptionPane.showOptionDialog(mainDialog,
            LNG.get("IAS_ROLE_UPDATE_QUESTION"), dialogTitle,
            JOptionPane.DEFAULT_OPTION, JOptionPane.INFORMATION_MESSAGE, null,
            options, options[0]);
        if (options[selected] == LNG.get("IAS_CANCEL")) {
          mainDialog.close();
        }
        else if (validateFields()) {
          modifyRole();
          mainDialog.close();
        }
      }
    });
    ClientUtilities.adjustEqualSizes(actionButton, closeButton);
    JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(actionButton);
    buttonsPanel.add(closeButton);
    cp.add(buttonsPanel, BorderLayout.SOUTH);
    mainDialog.pack();
    mainDialog.center(owner);
    mainDialog.setVisible(true);
  }

  /**
   * Cria o painel para seleo de permisses.
   * 
   * @param selectablePermissions permisses disponveis para adio
   * @param selectedPermissions permisses selecionadas no perfil
   * 
   * @return o painel criado
   */
  private JPanel createSelectionPanel(Vector<Permission> selectablePermissions,
    Vector<Permission> selectedPermissions) {
    ObjectTableProvider selectableFormat = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        Permission permission = (Permission) item;
        return (permission == null) ? null : new String[] { permission
          .getName() };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_AVAILABLE_PERMISSIONS") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { String.class };
      }
    };
    ObjectTableProvider selectedFormat = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        Permission permission = (Permission) item;
        return (permission == null) ? null : new String[] { permission
          .getName() };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { LNG.get("IAS_SELECTED_PERMISSIONS") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { String.class };
      }
    };
    permissionsSelection =
      new ContainerSelection<Permission>(selectableFormat, selectedFormat,
        true, false);
    permissionsSelection.loadItems(selectablePermissions, selectedPermissions);
    permissionsSelection.adjustTableColumns();
    if (!isNew) {
      permissionsSelection
        .addContainerSelectionListener(new ContainerSelectionListener() {
          @Override
          public void containerChanged() {
            checkDataChange();
          }
        });
    }
    return permissionsSelection.getPanel();
  }

  /**
   * Verifica se o campo nome foi preenchido e se ao menos uma permisso foi
   * escolhida.
   * 
   * @return true se o perfil possuir ao menos um nome e uma permisso, false
   *         caso contrrio
   */
  private boolean validateFields() {
    String nameText = roleName.getText().trim();
    if (nameText.equals("")) {
      return false;
    }
    return true;
  }

  /**
   * Obtm as informaes de perfil fornecidas pelo usurio
   * 
   * @return Objeto contendo as informaes de perfil fornecidas pelo usurio
   */
  private RoleInfo getScreenRoleInfo() {
    String nameText = roleName.getText().trim();
    String descriptionText = description.getText();
    List<Permission> selectedPermissions =
      permissionsSelection.getSelectedItems();
    Object[] permissionIds = new Object[selectedPermissions.size()];
    for (int i = 0; i < selectedPermissions.size(); i++) {
      permissionIds[i] = selectedPermissions.get(i).getId();
    }
    return new RoleInfo(nameText, descriptionText, permissionIds);
  }

  /**
   * Persiste um novo perfil em arquivo
   */
  private void createRole() {
    RoleInfo info = getScreenRoleInfo();
    role =
      RoleProxy.createRole(mainDialog, dialogTitle,
        LNG.get("IAS_WAITING_ROLE_CREATION"), info);
    if (role != null) {
      StandardDialogs.showInfoDialog(mainDialog, dialogTitle,
        LNG.get("IAS_ROLE_INCLUSION_WITH_SUCCESS"));
      clearFields();
    }
  }

  /**
   * Modificado os dados de um perfil j existente em arquivo
   */
  private void modifyRole() {
    RoleInfo info = getScreenRoleInfo();
    Role modifiedRole =
      RoleProxy.modifyRole(mainDialog, dialogTitle,
        LNG.get("IAS_WAITING_ROLE_MODIFICATION"), role.getId(), info);
    if (modifiedRole != null) {
      this.role = modifiedRole;
      StandardDialogs.showInfoDialog(mainDialog, dialogTitle,
        LNG.get("IAS_ROLE_UPDATED_WITH_SUCCESS"));
    }
  }

  /**
   * Limpa os campos.
   */
  private void clearFields() {
    roleName.setText("");
    description.setText("");
    permissionsSelection.clear();
    Vector<Permission> permissions =
      PermissionProxy.getAllPermissions(mainDialog, dialogTitle,
        LNG.get("IAS_WAITING_ALL_PERMISSIONS"));
    permissionsSelection.loadItems(permissions);
    permissionsSelection.adjustTableColumns();
    roleName.requestFocus();
  }

  /**
   * Verifica se algum dos campos de preenchimento de dados da permisso foi
   * alterado.
   * 
   * @return true se algum campo foi editado ou false caso contrario
   */
  private boolean hasChanged() {
    if (!roleName.getText().trim().equals(role.getName())) {
      return true;
    }
    if (!description.getText().trim().equals(role.getDescription())) {
      return true;
    }
    List<Permission> selectedPermissions =
      permissionsSelection.getSelectedItems();
    Object[] selectedPermIds = new Object[selectedPermissions.size()];
    for (int i = 0; i < selectedPermissions.size(); i++) {
      selectedPermIds[i] = selectedPermissions.get(i).getId();
    }
    Arrays.sort(selectedPermIds);
    Object[] rolePermIds = role.getPermissionIds();
    Arrays.sort(rolePermIds);
    if (!Arrays.equals(rolePermIds, selectedPermIds)) {
      return true;
    }
    return false;
  }

  /**
   * Verifica se algum campo foi editado. Em caso positivo, habilita o boto de
   * alterao de perfil. Caso contrrio, este  desabilitado.
   */
  private void checkDataChange() {
    if (hasChanged()) {
      actionButton.setEnabled(true);
    }
    else {
      actionButton.setEnabled(false);
    }
  }

  /**
   * Construtor.
   * 
   * @param owner Janela "pai"
   */
  public RoleDefinitionDialog(JFrame owner) {
    this.owner = owner;
  }
}
