/*
 * Detalhes da ltima alterao:
 * 
 * $Author: moura $ $Date: 2006-08-16 17:31:23 -0300 (Qua, 16 Ago 2006) $
 * $Revision: 54958 $
 */
package csbase.client.ias.server;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileInputStream;
import java.rmi.RemoteException;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateParsingException;
import java.text.MessageFormat;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.SortedSet;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextArea;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.desktop.RemoteTask;
import csbase.client.ias.ManagerPanel;
import csbase.client.remote.manager.server.ServerInfoManager;
import csbase.client.remote.manager.server.ServerInfoManagerListener;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.server.ServerInfo;

/**
 * Painel responsvel por exibir uma tabela com os servidores e opes de
 * gerenciamento dos mesmos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ServerInfoManagerPanel extends ManagerPanel {
  /** A janela onde este painel est inserido. */
  private JFrame parent;

  /** O modelo contendo os servidores atualmente no repositrio. */
  private ObjectTableModel<ServerInfo> tableModel;

  /** Um ouvinte que captura mudanas no repositrio de servidores. */
  private ServerInfoManagerListener managerListener;

  /**
   * Cria um painel para gerenciamento de servidores.
   * 
   * @param parent A janela onde este painel est inserido.
   */
  public ServerInfoManagerPanel(JFrame parent) {
    this.parent = parent;

    final String[] columnNames =
      new String[] { null, LNG.get("IAS_SERVER_NAME"),
          LNG.get("IAS_SERVER_URI"), LNG.get("IAS_SERVER_SUSPENDED"),
          LNG.get("IAS_SERVER_LOCAL"), LNG.get("IAS_SERVER_COMPLETED") };

    final Class<?>[] columnClasses =
      new Class[] { ServerInfo.class, String.class, String.class, String.class,
          String.class, String.class };

    if (ServerInfoManager.getInstance().isEnabled()) {
      ObjectTableProvider provider = new DefaultObjectTableProvider() {
        @Override
        public Object[] getCellValues(Object item) {
          ServerInfo serverInfo = (ServerInfo) item;
          String isSuspended;
          if (serverInfo.isSuspended()) {
            isSuspended = LNG.get("IAS_YES");
          }
          else {
            isSuspended = LNG.get("IAS_NO");
          }
          String isCompleted;
          if (serverInfo.isCompleted()) {
            isCompleted = LNG.get("IAS_YES");
          }
          else {
            isCompleted = LNG.get("IAS_NO");
          }
          String isLocal =
            serverInfo.isLocal() ? LNG.get("IAS_YES") : LNG.get("IAS_NO");

          return new Object[] { serverInfo, serverInfo.getName(),
              serverInfo.getURI(), isSuspended, isLocal, isCompleted };
        }

        @Override
        public String[] getColumnNames() {
          return columnNames;
        }

        @Override
        public Class<?>[] getColumnClasses() {
          return columnClasses;
        }
      };
      this.tableModel =
        new ObjectTableModel<ServerInfo>(this.getServersInfos(), provider);
      Comparator<?>[] comparators =
        { null, ClientUtilities.getStringComparatorIgnoreCase(),
            ClientUtilities.getStringComparatorIgnoreCase(),
            ClientUtilities.getStringComparatorIgnoreCase(),
            ClientUtilities.getStringComparatorIgnoreCase(),
            ClientUtilities.getStringComparatorIgnoreCase() };
      make(this.tableModel, comparators, false);
      this.managerListener = new ServerInfoManagerListener() {
        @Override
        public void wasAddedServerInfo(final ServerInfo addedLocalServer) {
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              tableModel.add(addedLocalServer);
              ServerInfoManagerPanel.this.getTable().adjustColumnWidth();
            }
          });
        }

        @Override
        public void wasRemovedServerInfo(final ServerInfo removedLocalServer) {
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              tableModel.remove(removedLocalServer);
              ServerInfoManagerPanel.this.getTable().adjustColumnWidth();
            }
          });
        }

        @Override
        public void wasModifiedServerInfo(final ServerInfo oldLocalServer,
          final ServerInfo newLocalServer) {
          SwingThreadDispatcher.invokeLater(new Runnable() {
            @Override
            public void run() {
              tableModel.remove(oldLocalServer);
              tableModel.add(newLocalServer);
              ServerInfoManagerPanel.this.getTable().adjustColumnWidth();
            }
          });
        }
      };
      ServerInfoManager.getInstance().addServerInfoManagerListener(
        this.managerListener);

      this.getTable().adjustColumnWidth();
    }
    else {
      this.setLayout(new BorderLayout());
      JTextArea instructionsTextArea =
        new JTextArea(LNG.get("IAS_SERVER_ADMIN_INSTRUCTIONS"));
      instructionsTextArea.setEditable(false);
      instructionsTextArea.setLineWrap(true);
      instructionsTextArea.setWrapStyleWord(true);
      this.add(instructionsTextArea, BorderLayout.CENTER);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JPanel makeButtons() {
    JPanel panel = super.makeButtons();
    panel.add(new JPanel());
    this.createUpdateButton(panel);
    panel.add(new JPanel());
    final JButton addButton = this.createAddCertButton(panel);
    final JButton delButton = this.createRemCertButton(panel);
    GUIUtils.matchPreferredSizes(new JComponent[] { addButton, delButton });
    return panel;
  }

  /**
   * Criao de boto de update
   * 
   * @param panel painel
   * @return o boto.
   */
  private JButton createUpdateButton(JPanel panel) {
    JButton updateButton = new JButton(LNG.get("IAS_SERVER_UPDATE"));
    updateButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        tableModel.setRows(getServersInfos());
      }
    });
    panel.add(updateButton);
    return updateButton;
  }

  /**
   * Criao de boto de adio de certificado.
   * 
   * @param panel o painel.
   * @return o boto.
   */
  private JButton createAddCertButton(JPanel panel) {
    JButton addCertButton = new JButton(LNG.get("IAS_SERVER_ADDCERT"));
    addCertButton.setEnabled(false);
    addCertButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        try {
          addCert();
          tableModel.setRows(getServersInfos());
        }
        catch (CertificateParsingException cpe) {
          StandardErrorDialogs.showErrorDialog(parent,
            LNG.get("IAS_SERVER_ADDCERT_TITLE"),
            LNG.get("IAS_SERVER_ADDCERT_PARSE_EXCEPTION_ERROR"));
        }
        catch (Exception ex) {
          StandardErrorDialogs.showErrorDialog(parent,
            LNG.get("IAS_SERVER_ADDCERT_TITLE"),
            LNG.get("IAS_SERVER_ADDCERT_ERROR"), ex);
        }
      }
    });
    panel.add(addCertButton);
    this.addButtonToSelectionListener(addCertButton);
    return addCertButton;
  }

  /**
   * Criao de boto para remoo de certificado.
   * 
   * @param panel o painel.
   * @return o boto.
   */
  private JButton createRemCertButton(JPanel panel) {
    JButton remCertButton = new JButton(LNG.get("IAS_SERVER_REMCERT"));
    remCertButton.setEnabled(false);
    remCertButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        try {
          remCert();
          tableModel.setRows(getServersInfos());
        }
        catch (Exception ex) {
          StandardErrorDialogs.showErrorDialog(parent,
            LNG.get("IAS_SERVER_REMCERT_TITLE"),
            LNG.get("IAS_SERVER_REMCERT_ERROR"), ex);
        }
      }
    });
    panel.add(remCertButton);
    this.addButtonToSelectionListener(remCertButton);
    return remCertButton;
  }

  /**
   * Remoo de certificado.
   * 
   * @throws Exception em caso de falha
   */
  private void remCert() throws Exception {
    int row = this.getTable().getSelectedRow();
    final ServerInfo serverInfo =
      (ServerInfo) this.getTable().getModel().getValueAt(row, IDENTIFIER_INDEX);

    if (!serverInfo.isCompleted()) {
      return;
    }

    String msg =
      MessageFormat.format(LNG.get("IAS_SERVER_REMCERT_CONFIRMATION_MESSAGE"),
        serverInfo.getName());
    int retVal =
      JOptionPane.showConfirmDialog(this.parent, msg,
        LNG.get("IAS_SERVER_REMCERT_TITLE"), JOptionPane.YES_NO_OPTION);
    if (retVal == JOptionPane.NO_OPTION) {
      return;
    }

    RemoteTask<Boolean> remCertTask = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        ServerInfoManager.getInstance().removeCertificate(serverInfo.getName());
      }
    };

    if (!remCertTask.execute(this.parent, LNG.get("IAS_SERVER_REMCERT_TITLE"),
      LNG.get("IAS_SERVER_REMCERT_MESSAGE"))) {
      throw remCertTask.getError();
    }
  }

  /**
   * Adio de certificado.
   * 
   * @throws Exception em caso de falha.
   */
  private void addCert() throws Exception {
    int row = this.getTable().getSelectedRow();
    final ServerInfo serverInfo =
      (ServerInfo) this.getTable().getModel().getValueAt(row, IDENTIFIER_INDEX);

    JFileChooser fileChooser = new JFileChooser();
    int retVal = fileChooser.showOpenDialog(this.parent);

    if (retVal == JFileChooser.APPROVE_OPTION) {
      File certFile = fileChooser.getSelectedFile();
      FileInputStream certfis = new FileInputStream(certFile);
      CertificateFactory cf = CertificateFactory.getInstance("X.509");
      final Certificate cert = cf.generateCertificate(certfis);
      certfis.close();

      RemoteTask<Boolean> addCertTask = new RemoteTask<Boolean>() {
        @Override
        protected void performTask() throws Exception {
          ServerInfoManager.getInstance().addCertificate(serverInfo.getName(),
            cert);
        }
      };

      if (!addCertTask.execute(this.parent,
        LNG.get("IAS_SERVER_ADDCERT_TITLE"),
        LNG.get("IAS_SERVER_ADDCERT_MESSAGE"))) {
        throw addCertTask.getError();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void add() {
    new ServerInfoDialog(this.parent);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void modify(Object id) {
    ServerInfo localServer = (ServerInfo) id;
    if (localServer == null) {
      throw new IllegalStateException(
        "No foi possvel achar um servidor com o id " + id + ".");
    }
    new ServerInfoDialog(this.parent, localServer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void delete(Object id) {
    final ServerInfo localServer = (ServerInfo) id;
    int option =
      StandardDialogs.showYesNoDialog(this, LNG.get("IAS_SERVER_DELETE_TITLE"),
        MessageFormat.format(LNG.get("IAS_SERVER_DELETE_CONFIRMATION_MESSAGE"),
          new Object[] { localServer.getName() }));
    if (option == JOptionPane.YES_OPTION) {
      RemoteTask<Void> deleteLocalServerTask = new RemoteTask<Void>() {
        @Override
        protected void performTask() throws Exception {
          ServerInfoManager.getInstance().removeServerInfo(localServer);
        }
      };
      deleteLocalServerTask.execute(this.parent, LNG
        .get("IAS_SERVER_DELETE_TITLE"), MessageFormat.format(
        LNG.get("IAS_SERVER_DELETING_MESSAGE"),
        new Object[] { localServer.getName() }));
    }
  }

  /**
   * Obtm as informaes de todos os servidores do repositrio.
   * 
   * @return Um conjunto ordenado (por nome) com as informaes dos servidores.
   *         Caso ocorra algum erro, ser retornado null.
   */
  private List<ServerInfo> getServersInfos() {
    RemoteTask<SortedSet<ServerInfo>> getAllLocalServersTask =
      new RemoteTask<SortedSet<ServerInfo>>() {
        @Override
        protected void performTask() throws RemoteException {
          this.setResult(ServerInfoManager.getInstance().getServersInfos());
        }
      };
    SortedSet<ServerInfo> localServerSortedSet = null;
    if (getAllLocalServersTask.execute(this.parent,
      LNG.get("IAS_SERVER_GET_ALL_TITLE"),
      LNG.get("IAS_SERVER_GETTING_ALL_MESSAGE"))) {
      localServerSortedSet = getAllLocalServersTask.getResult();
    }
    if (localServerSortedSet == null) {
      return new LinkedList<ServerInfo>();
    }
    return new LinkedList<ServerInfo>(localServerSortedSet);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void beforeClose() {
    if (ServerInfoManager.getInstance().isEnabled()) {
      ServerInfoManager.getInstance().removeServerInfoManagerListener(
        this.managerListener);
    }
  }
}
