package csbase.client.preferences;

import java.awt.Window;

import csbase.client.preferences.definition.PreferenceDefinition;
import csbase.client.preferences.util.PreferencesUtil;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Est classe representa o gerenciador de preferncias do sistema. Este
 * gerenciador  uma instncia nica (singleton) e  a partir dele que fazemos a
 * carga e persistncia das preferncias do usurio.
 * 
 * A inicializao do sistema de preferncias  dividido em 2 partes: <br/>
 * - A carga das definies de preferncias (via enumeraes); <br/>
 * - A carga das preferncias do usurio;
 * 
 * Ao fazermos a carga das definies das preferncias do sistema estamos, na
 * verdade, construindo uma rvore de preferncias default. Est carga  feita
 * no momento em que o {@link PreferenceManager}  instanciado e est rvore
 * armazena apenas as informaes das preferncias de aplicaes junto aos seus
 * valores default.
 * 
 * Ao fazemos a primeira chamada ao mtodo
 * {@link PreferenceManager#loadPreferences()} estamos, de fato, fazendo a carga
 * das preferncias do usurio e, assim, preenchendo a rvore de preferncias
 * com os valores do usurio.
 * 
 * Para que o {@link PreferenceManager} possa fazer a carga de uma nova
 * definio de preferncia que no seja de aplicao,  necessrio invocar
 * {@link PreferenceManager#loadDefinition(Class)}. NOTA: Esta carga explcita
 * s pode ser feita antes da carga das preferncias do usurio. Uma vez que a
 * leitura das preferncias do usurio  feita, no  possvel carregar nenhuma
 * outra definio.
 * 
 * Ao fazermos a carga das preferncias do usurio, temos em mos um objeto do
 * tipo {@link PreferenceCategory} que  uma estrutura recursiva que armazena
 * todas as preferncias do usurio. Ao manipularmos os valores de preferncias,
 * podemos fazer a persistncia chamando o mtodo
 * {@link PreferenceManager#savePreferences()}.
 * 
 * @see PreferenceCategory PreferenceValue
 * 
 * @author Tecgraf
 */
public class PreferenceManager {

  /** Referncia para a instncia nica dessa classe. */
  private static PreferenceManager instance;

  /** Raiz da rvore que representa as preferncias do usurio. */
  private PreferenceCategory root;

  /** Flag que indica se as preferncias do usurio j foram carregadas. */
  private boolean alreadyLoaded = false;

  /** Instncia nica do carregador de definies. */
  private DefinitionLoader loader;

  /** Instncia nica do mecanismo que persiste as preferncias. */
  private PreferencePersistence persistence;

  /**
   * Retorna a instncia nica do gerenciador de preferncias.
   * 
   * @return instncia nica do gerenciador de preferncias.
   */
  public static PreferenceManager getInstance() {
    if (instance == null) {
      instance = new PreferenceManager();
    }
    return instance;
  }

  /**
   * Carrega a rvore de preferncias do usurio.
   * 
   * @return raz da rvore de preferncias do usurio.
   */
  public PreferenceCategory loadPreferences() {
    return loadPreferences(null);
  }

  /**
   * Carrega a rvore de preferncias do usurio.
   * 
   * @param parent janela principal que est manipulando as preferncias.
   * 
   * @return raz da rvore de preferncias do usurio.
   */
  public PreferenceCategory loadPreferences(Window parent) {
    if (!alreadyLoaded) {

      try {
        persistence.load(parent, root);
      }
      catch (Exception e) {
        // Se no for possvel carregar as preferncias especficas de um usurio
        // ns adotamos a poltica de ignora-las e retornamos a rvore com os 
        // valores default do sistema.
      }
      alreadyLoaded = true;
    }
    return root;
  }

  /** Salva as preferncias do usurio. */
  public void savePreferences() {
    savePreferences(null);
  }

  /**
   * Salva as preferncias do usurio.
   * 
   * @param parent janela principal que est manipulando as preferncias.
   */
  public void savePreferences(Window parent) {
    if (!alreadyLoaded) {
      throw new PreferenceException(
        "No  possvel salvar antes da carga das preferncias do usurio.");
    }
    persistence.save(parent, root);
  }

  /**
   * Retorna o objeto que encapsula as preferncias de uma determinada
   * aplicao.
   * 
   * @param registry registro de uma aplicao.
   * @param parent janela principal que est manipulando as preferncias.
   * @return objeto que encapsula as preferncias do usurio de uma determinada
   *         aplicao.
   */
  public PreferenceCategory loadAppPreferences(ApplicationRegistry registry,
    Window parent) {

    PreferenceCategory root = loadPreferences(parent);

    Class<? extends PreferenceDefinition> enumClass =
      loader.loadAppEnum(registry);

    return root.getCategory(enumClass);
  }

  /**
   * True se a aplicao possuir preferncias, false caso contrrio.
   * 
   * @param registry registro da aplicao.
   * @return true se a aplicao possuir preferncias, false caso contrrio.
   */
  public boolean hasAppPreferences(ApplicationRegistry registry) {
    return loader.loadAppEnum(registry) != null;
  }

  /**
   * Permite a adio de preferncias de sistema que sero carregadas na
   * construo das preferncias.
   * 
   * NOTA: A carga de uma nova definio s pode ser feita antes da primeira
   * chamada de {@link PreferenceManager#loadPreferences()} quando, de fato, 
   * feita a leitura das preferncias do usurio. Uma vez que a leitura das
   * preferncias do usurio  feita, no  possvel carregar nenhuma outra
   * definio.
   * 
   * @param enumClass enumerao que define preferncias.
   */
  public void loadDefinition(Class<? extends PreferenceDefinition> enumClass) {
    if (alreadyLoaded) {
      throw new PreferenceException(
        "No  possvel carregar uma definio aps as preferncias do usurio terem sido carregadas.");
    }

    loader.loadDefinition(root, enumClass);
  }

  /**
   * Muda o valor dafult de uma preferncia.
   * 
   * NOTA: A alterao do valor default s pode ser feita antes da primeira
   * chamada de {@link PreferenceManager#loadPreferences()} quando, de fato, 
   * feita a leitura das preferncias do usurio. Uma vez que a leitura das
   * preferncias do usurio  feita, no  possvel modificar nenhum valor
   * default.
   * 
   * @param name constante que define a preferncia.
   * @param defaultValue novo valor default.
   */
  public void setDefaultValue(PreferenceDefinition name, String defaultValue) {
    if (alreadyLoaded) {
      throw new PreferenceException(
        "No  possvel modificar o valor default aps as preferncias do usurio terem sido carregadas.");
    }

    PreferenceCategory pc = PreferencesUtil.deepSearch(name, root);
    PreferenceValue<?> preference = pc.getPreference(name);

    preference.setDefaultValueAsStr(defaultValue);
    preference.setValueAsStr(defaultValue);
  }

  /** Construtor privado. */
  private PreferenceManager() {
    this.loader = DefinitionLoader.getInstance();
    this.persistence = PreferencePersistence.getInstance();

    this.root =
      new PreferenceCategory(PreferenceCategory.ROOT_ID,
        loader.getGeneralBundle());

    loader.loadAppDefinitions(root);
  }
}
