package csbase.client.preferences.editors;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.resourcehelpers.PreferredAppDialog;
import csbase.client.preferences.PreferenceEditor;
import csbase.client.preferences.PreferenceValue;
import csbase.client.preferences.types.PVMap;
import csbase.logic.ProjectFileType;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Classe que permite a edio de uma aplicao preferencial de um arquivo
 * especfico a partir de um {@link PVMap}.
 * 
 * @author Tecgraf
 */
public class PreferredAppEditor extends PreferenceEditor<Map<String, String>> {

  /**
   * Construtor padro.
   * 
   * @param pv objeto que encapsula o valor da preferncia.
   */
  public PreferredAppEditor(PreferenceValue<Map<String, String>> pv) {
    super(pv);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createComponent() {
    Map<String, String> value = getValue();

    int y = 0;

    JPanel panel = new JPanel(new GridBagLayout());
    if (!value.isEmpty()) {
      for (Entry<String, String> entry : value.entrySet()) {
        JLabel type = createTypeLabel(entry.getKey());
        JLabel app = createAppLabel(entry.getValue());
        JButton chooser = createChooserButton(entry.getKey(), app);

        panel.add(type, new GBC(0, y).insets(5).none());
        panel.add(app, new GBC(1, y).horizontal());
        panel.add(chooser, new GBC(2, y++).insets(5).none());
      }
    }
    else {
      JLabel noPreferredApp =
        new JLabel(getPrefixedString("no.preferred.applications"));
      noPreferredApp.setFont(font);
      panel.add(noPreferredApp);
    }

    setTitledBorder(panel);

    return panel;
  }

  /**
   * Cria o rtulo que exibe a aplicao escolhida.
   * 
   * @param appId - identificador da aplicao.
   * @return rtulo com o cone da aplicao escolhida.
   */
  private JLabel createAppLabel(String appId) {
    ApplicationManager am = ApplicationManager.getInstance();
    ImageIcon icon = am.getApplicationIcon(appId);

    JLabel label = new JLabel(icon);
    label.setToolTipText(getAppName(appId));

    return label;
  }

  /**
   * Cria o boto que permite o usurio escolher a aplicao preferencial de um
   * dado tipo de arquivo.
   * 
   * @param fileType - tipo do arquivo.
   * @param app - rtulo que exibe a aplicao escolhida.
   * 
   * @return boto.
   */
  private JButton createChooserButton(final String fileType, final JLabel app) {
    final JButton button = new JButton(getPrefixedString("choose"));

    button.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        ApplicationManager am = ApplicationManager.getInstance();
        List<String> apps = am.getApplicationsFromType(fileType);

        PreferredAppDialog dialog =
          new PreferredAppDialog(null, apps, fileType, false);
        dialog.setVisible(true);

        String appId = dialog.getSelectedApp();

        if (appId != null) {
          app.setIcon(am.getApplicationIcon(appId));
          app.setToolTipText(getAppName(appId));
          getValue().put(fileType, appId);
          notifyListeners();
        }
      }
    });

    button.setEnabled(isEditable());
    return button;
  }

  /**
   * Retorna o nome da aplicao.
   * 
   * @param appId - identificador da aplicao.
   * @return nome da aplicao.
   */
  private String getAppName(String appId) {
    ApplicationManager am = ApplicationManager.getInstance();
    ApplicationRegistry registry = am.getApplicationRegistry(appId);
    return am.getApplicationName(registry);
  }

  /**
   * Cria o rtulo que exibe o tipo do arquivo.
   * 
   * @param key - String que representa o tipo do arquivo.
   * @return rtulo.
   */
  private JLabel createTypeLabel(String key) {
    ProjectFileType type = ProjectFileType.getFileType(key);

    JLabel label = new JLabel(type.getDescription());
    label.setFont(font);
    return label;
  }
}
