/**
 * $Id: ExportTask.java 150777 2014-03-19 14:16:56Z oikawa $
 */

package csbase.client.project;

import java.awt.Window;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.LocalTask;
import csbase.client.externalresources.StandaloneLocalFile;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;

/**
 * <code>LocalTask</code> responsvel por exportar um arquivo ou diretrio (com
 * todo o seu contedo) para o sistema de arquivos do cliente.
 */
public class ExportTask extends LocalTask<Void> {

  /** Ttulo do dilogo de exportao */
  private static final String EXPORT_TITLE = "PRJ_PROJECT_FILE_EXPORT_TITLE";

  /** Prefixo para ttulos de dilogos de erros */
  private static final String ERROR_PREFIX = "cabase.title.error";

  /** Mensagem em caso de erro de I/O */
  private static final String EXPORT_IO_ERROR =
    "PRJ_PROJECT_FILE_EXPORT_IO_ERROR";

  /** Tamanho do buffer de I/O. */
  private static final int BUFFER_SIZE = 48 * 1024;

  /** Mensagem para passo da transferncia em execuo */
  private static final String EXPORT_STEP = "PRJ_PROJECT_FILE_EXPORT_STEP";

  /** janela onde sero lanadas as dialogs de erro */
  private Window window;

  /**
   * Arquivo/diretrio a ser exportado. O prefixo "root" aqui indica que ele  o
   * arquivo/diretrio a partir do qual todo o contedo (eventualmente outros
   * arquivos ou diretrios) deve ser transferido. No confundir com o
   * diretrio-raiz do projeto.
   */
  private ClientProjectFile rootFile;
  /**
   * Diretrio de destino para os arquivos/diretrios a serem exportados. O
   * prefixo "root" tem a mesma semntica do campo <code>rootFile</code>:
   * trata-se do diretrio que servir de destino para a transferncia como um
   * todo.
   */
  private StandaloneLocalFile targetFile;

  /** Volume total ocupado pelos arquivos a serem exportados. */
  private long totalSize;

  /**
   * Cria a tarefa.
   * 
   * @param targetFile arquivo/diretrio de destino no cliente.
   * @param sourceFile arquivo/diretrio a ser exportado no projeto.
   * @param window janela onde sero lanadas as dialogs de erro.
   */
  public ExportTask(File targetFile, ClientProjectFile sourceFile, Window window) {
    this(new StandaloneLocalFile(targetFile), sourceFile, window);
  }

  /**
   * Construtor.
   * 
   * @param targetFile arquivo/diretrio de destino no cliente.
   * @param sourceFile arquivo/diretrio a ser exportado no projeto.
   * @param window janela onde sero lanadas as dialogs de erro.
   */
  public ExportTask(StandaloneLocalFile targetFile,
    ClientProjectFile sourceFile, Window window) {
    this.window = window;
    this.rootFile = sourceFile;
    this.targetFile = targetFile;
  }

  /**
   * Especifica como  feito o tratamento de erros. Trata somente
   * <code>IOException</code>, outros erros so repassados para o mtodo pai na
   * hierarquia.
   */
  @Override
  protected void handleError(Exception error) {
    if (error instanceof IOException) {
      String msg =
        MessageFormat.format(LNG.get(EXPORT_IO_ERROR),
          new Object[] { rootFile.getName() });
      StandardErrorDialogs.showErrorDialog(window, LNG.get(ERROR_PREFIX)
        + " - " + LNG.get(EXPORT_TITLE), msg);
      return;
    }
    super.handleError(error);
  }

  /**
   * Executa a tarefa. Transfere o contedo do arquivo ou diretrio para o
   * sistema de arquivos do cliente.
   */
  @Override
  protected void performTask() throws Exception {
    transfer(targetFile, rootFile);
  }

  /**
   * Transfere o arquivo ou diretrio selecionado para o sistema de arquivos do
   * cliente.
   * 
   * @param targetFile arquivo/diretrio de destino no cliente.
   * @param sourceFile arquivo/diretrio a ser transferido no projeto.
   * 
   * @throws IOException
   */
  private void transfer(StandaloneLocalFile targetFile,
    ClientProjectFile sourceFile) throws IOException {
    totalSize += sourceFile.size();
    if (sourceFile.isDirectory()) {
      transferDirectory(targetFile, sourceFile);
    }
    else {
      transferFile(targetFile, sourceFile);
    }
  }

  /**
   * Transfere o contedo de um arquivo para o sistema de arquivos do cliente.
   * 
   * @param targetFile arquivo de destino no cliente.
   * @param sourceFile arquivo a ser transferido no projeto.
   * 
   * @throws IOException
   */
  private void transferFile(StandaloneLocalFile targetFile,
    ClientProjectFile sourceFile) throws IOException {
    String msg =
      MessageFormat.format(LNG.get(EXPORT_STEP),
        new Object[] { sourceFile.getName() });
    setStepText(msg);
    targetFile.createNewFile();
    BufferedOutputStream out =
      new BufferedOutputStream(targetFile.getOutputStream(false));
    InputStream in = null;
    try {
      in = sourceFile.getInputStream();
      byte[] buffer = new byte[BUFFER_SIZE];
      int readBytes = 0;
      while ((readBytes = in.read(buffer)) > 0) {
        out.write(buffer, 0, readBytes);
      }
      out.flush();
    }
    finally {
      if (out != null) {
        out.close();
      }
      if (in != null) {
        in.close();
      }
    }
  }

  /**
   * Transfere o contedo de um diretrio para o sistema de arquivos do cliente.
   * 
   * @param targetDir diretrio de destino no cliente.
   * @param sourceDir diretrio a ser transferido no projeto.
   * 
   * @throws IOException
   */
  private void transferDirectory(StandaloneLocalFile targetDir,
    ClientProjectFile sourceDir) throws IOException {
    targetDir.mkdir();
    ClientProjectFile[] sourceChildren = sourceDir.getChildren();
    for (int i = 0; i < sourceChildren.length; i++) {
      ClientProjectFile sourceChild = sourceChildren[i];
      String targetChildName =
        targetDir.getAbsolutePath() + File.separator + sourceChild.getName();
      StandaloneLocalFile targetChild =
        new StandaloneLocalFile(new File(targetChildName));
      transfer(targetChild, sourceChildren[i]);
    }
  }

  /**
   * Retorna o volume total transferido.
   * 
   * @return o volume total transferido.
   */
  public long getTotalSize() {
    return totalSize;
  }
}
