package csbase.client.project;

import java.util.ArrayList;
import java.util.List;

import javax.swing.table.DefaultTableModel;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.logic.User;
import csbase.logic.UserOutline;
import csbase.logic.UserProjectInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Modelo para a tabela de abertura de projetos.
 * 
 * @author Tecgraf
 */
class ProjectTableModel extends DefaultTableModel {

  /**
   * Colunas visveis nas tabelas.
   */
  public enum VisibleColumns {
    /**
     * Apenas o nome do projeto.
     */
    NAME,
    /**
     * Dono da tabela, nome do projeto.
     */
    OWNER_NAME,
    /**
     * Dono da tabela, nome do projeto, permisso de escrita.
     */
    OWNER_NAME_PERMISSION,
  }

  /**
   * Nomes das colunas.
   */
  private static final String[] COL_NAMES = {
      LNG.get("ProjectTableModel.col.owner"),
      LNG.get("ProjectTableModel.col.project"),
      LNG.get("ProjectTableModel.col.permission") };

  /**
   * Classes das colunas.
   */
  private static Class<?>[] COL_CLASSES = { UserOutline.class,
      UserProjectInfo.class, Boolean.class };

  /**
   * Classe para representar o par (usurio, nome do projeto) associado a cada
   * linha da tabela.
   */
  static class RowData {

    /**
     * Informaes do usurio.
     */
    UserOutline user;

    /**
     * Nome do projeto.
     */
    UserProjectInfo project;

    /**
     * Construtor.
     * 
     * @param userInfo Informaes do usurio
     * @param projectInfo Informaes do projeto.
     */
    RowData(UserOutline userInfo, UserProjectInfo projectInfo) {
      this.user = userInfo;
      this.project = projectInfo;
    }
  }

  /**
   * Lista de pares (arq, fluido). Corresponde s linhas da tabela.
   */
  List<RowData> rows;
  /**
   * Lista com os projetos para os quais o usurio tem permisso de escrita.
   */
  List<Boolean> writableProjects;
  /**
   * Colunas visveis na tabela.
   */
  private VisibleColumns visibleColumns;

  /**
   * Construtor.
   * 
   * @param projects Lista com os pares (usurio, nome do projeto).
   * @param visibleColumns
   */
  public ProjectTableModel(List<RowData> projects, VisibleColumns visibleColumns) {
    this.visibleColumns = visibleColumns;
    rows = new ArrayList<RowData>();
    fillRows(projects);
  }

  /**
   * Troca as linhas atuais da tabela por novas linhas.
   * 
   * @param projects Lista com os pares (usurio, nome do projeto).
   */
  public void replaceRows(List<RowData> projects) {
    getDataVector().removeAllElements();
    fillRows(projects);
    fireTableDataChanged();
  }

  /**
   * Obtm o par (usurio, nome do projeto) associado a uma linha especfica.
   * 
   * @param rowIndex ndice da linha.
   * @return Par (usurio, nome do projeto) associado  linha.
   */
  public RowData getRow(int rowIndex) {
    if (rowIndex < 0) {
      return null;
    }
    return rows.get(rowIndex);
  }

  /**
   * Redefine as colunas visveis.
   * 
   * @param visibleColumns colunas visveis
   */
  public void setVisibleColumns(VisibleColumns visibleColumns) {
    if (visibleColumns != this.visibleColumns) {
      this.visibleColumns = visibleColumns;
      fireTableStructureChanged();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getColumnName(int column) {
    return COL_NAMES[column];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowCount() {
    return rows == null ? 0 : rows.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueAt(int row, int column) {
    RowData rowData = rows.get(row);
    switch (visibleColumns) {
      case NAME:
        return rowData.project;

      case OWNER_NAME:
      case OWNER_NAME_PERMISSION:
        if (column == 0) {
          return rowData.user.getName();
        }
        if (column == 1) {
          return rowData.project;
        }
        else {
          return (writableProjects == null) ? null : writableProjects.get(row);
        }

      default:
        throw new AssertionError("tipo desconhecido " + visibleColumns.name());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isCellEditable(int row, int column) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    switch (visibleColumns) {
      case NAME:
        return 1;

      case OWNER_NAME:
        return 2;

      case OWNER_NAME_PERMISSION:
        return 3;

      default:
        throw new AssertionError("tipo desconhecido " + visibleColumns.name());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass(int columnIndex) {
    return COL_CLASSES[columnIndex];
  }

  /**
   * Preenche as linhas a serem exibidas na tabela.
   * 
   * @param projects lista de projetos.
   */
  private void fillRows(final List<RowData> projects) {
    rows.clear();
    rows.addAll(projects);

    writableProjects = null;

    if (visibleColumns == VisibleColumns.OWNER_NAME_PERMISSION) {
      final Object userId = User.getLoggedUser().getId();
      Task<List<Boolean>> task = new RemoteTask<List<Boolean>>() {

        @Override
        protected void performTask() throws Exception {
          List<Object> prjIds = new ArrayList<Object>();
          for (RowData rowData : projects) {
            prjIds.add(rowData.project.getProjectId());
          }

          ProjectServiceInterface service = ClientRemoteLocator.projectService;
          List<Boolean> result = service.userCanWrite(prjIds, userId);
          setResult(result);
        }
      };

      String title = LNG.get("GENERIC.title.wait");
      String msg = LNG.get("ProjectTableModel.info.checkinPermissions");

      if (task.execute(null, title, msg)) {
        writableProjects = task.getResult();
      }
    }
  }
}
