/**
 * $Id: AlgorithmManagementObserver.java 167962 2015-09-02 23:16:26Z analodi $
 */

package csbase.client.remote;

import java.rmi.RemoteException;
import java.util.Observable;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.AlgoEvent;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.Category;
import csbase.logic.algorithms.CategorySet;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;

/**
 * Classe responsvel por receber os eventos disparados pelo servio de
 * algoritmos (algoService). Sempre que ocorrer criao, modificao ou remoo
 * de algoritmos, o singleton desta classe ser avisado e por sua vez ir
 * notificar quaisquer janelas que se registrarem como observadores deste.
 *
 * @author Tecgraf/PUC-Rio
 */
public class AlgorithmManagementObserver extends Observable {

  /** Instncia nica do observador */
  private static AlgorithmManagementObserver instance = null;

  /**
   * O ouvinte de mensagens que filtra aquelas com o corpo contendo objetos da
   * classe {@link AlgoEvent}.
   */
  private IMessageListener listener = new IMessageListener() {
    @Override
    public void onMessagesReceived(Message... messages) throws Exception {
      for (Message message : messages) {
        AlgoEvent event = (AlgoEvent) message.getBody();
        try {
          if (event.item instanceof AlgorithmInfo
            || event.type == AlgoEvent.RELOAD) {
            AlgorithmInfo.update(event);
          }
          else if (event.item instanceof Category) {
            Category.update(event);
          }
          else if (event.item instanceof CategorySet) {
            CategorySet.update(event);
          }
          else {
            setChanged();
            notifyObservers(event);
          }
        }
        catch (Exception e) {
          StandardErrorDialogs.showErrorDialog(null, LNG.get("ERRO"), e);
        }
      }
    }
  };

  /**
   * Cria um singleton desta classe.
   *
   * @throws RemoteException erro de rmi
   */
  public static void start() throws RemoteException {
    if (instance != null) {
      return;
    }
    instance = new AlgorithmManagementObserver();
  }

  /**
   * @throws RemoteException erro de rmi
   */
  public static void stop() throws RemoteException {
    if (instance == null) {
      return;
    }
    instance.uninstall();
  }

  /**
   * Registra no servidor o observador de eventos relacionados a algoritmo.
   */
  public void install() {
    MessageProxy.addListener(listener, AlgoEvent.class);
  }

  /**
   * Remove do servidor o observador de eventos relacionados a algoritmo.
   */
  public void uninstall() {
    MessageProxy.removeListener(listener);
  }

  /**
   * Obtm a instncia nica (singleton) do observador remoto do servio de
   * algoritmos.
   *
   * @return a instncia nica do observador
   */
  public static AlgorithmManagementObserver getInstance() {
    return instance;
  }

  /**
   * Construtor que exporta este objeto para o RMI e o registra como observador
   * do algoService e do ServerMonitor.
   *
   * @throws RemoteException erro de rmi
   */
  private AlgorithmManagementObserver() throws RemoteException {
    super();
    install();
  }
}
