package csbase.client.remote.srvproxies.messageservice.consumers;

import java.awt.Window;
import java.rmi.RemoteException;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import tecgraf.javautils.core.filter.IFilter;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.Client;
import csbase.client.ClientServerManager;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.util.CodeBlockLog;
import csbase.logic.MonitoredServerListener;
import csbase.logic.ServerURI;
import csbase.remote.ClientRemoteLocator;
import csbase.util.messages.Message;
import csbase.util.rmi.IPingable;
import csbase.util.rmi.PortReference;

/**
 * Consumidor que adapta seu estilo de consumo, entre polling e listener, de
 * acordo caractersticas da rede.
 * 
 * @author Tecgraf
 * 
 * @see MessagePollingConsumer
 * @See MessageListenerStrategy
 */
public class AdaptableMessageConsumer implements IMessageConsumer {

  /**
   * Logger da classe.
   */
  private static Logger LOGGER = Logger
    .getLogger(AdaptableMessageConsumer.class.getName());

  /** Estratgia corrente de consumo de mensagens. */
  private IMessageConsumer consumer;
  /** Indica se o usurio est logado no servidor. */
  private boolean loggedIn;

  /**
   * Ouvinte que ir tratar as mensagens obtidas pelo consumidor real.
   */
  private IMessageConsumer.IListener listener;
  /**
   * Filtro de mensagens.
   */
  private IFilter<Message> filter;

  /**
   * Utilizado para verificar a conectividade do servidor com o cliente em uma
   * determinada porta.<br>
   * Tambm  utilizado como reserva de porta.
   */
  private PortReference portRef;

  /**
   * Construtor.
   */
  public AdaptableMessageConsumer() {
    this.loggedIn = true;
    this.portRef = new PortReference();
    if (listener != null && getConsumer() != null) {
      getConsumer().setListener(listener, filter);
    }

    ClientServerManager.getInstance().addListener(
      new MonitoredServerListener() {
        @Override
        public void notifyLoggedOut(ServerURI serverURI) {
          setLoggedOut();
        }

        @Override
        public void notifyLoggedIn(ServerURI serverURI) {
          setLoggedIn();
        }

        @Override
        public void notifyConnectionReestablished(ServerURI serverURI) {
          setLoggedIn();
        }

        @Override
        public void notifyConnectionLost(ServerURI serverURI) {
          setLoggedOut();
        }

        private void setLoggedOut() {
          loggedIn = false;
          destroyConsumer();
        }

        private void setLoggedIn() {
          loggedIn = true;
          if (listener != null && getConsumer() != null) {
            getConsumer().setListener(listener, filter);
          }
        }
      });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setListener(IListener listener, IFilter<Message> filter) {
    this.listener = listener;
    this.filter = filter;
    if (getConsumer() != null) {
      CodeBlockLog log =
        new CodeBlockLog(Level.FINER, "Alterando o ouvinte de mensagens");
      LOGGER.log(log.begin());

      getConsumer().setListener(listener, filter);

      LOGGER.log(log.finished());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void clearListener() {
    this.listener = null;
    this.filter = null;
    if (getConsumer() != null) {
      CodeBlockLog log =
        new CodeBlockLog(Level.FINER, "Desligando o ouvinte de mensagens");
      LOGGER.log(log.begin());

      getConsumer().clearListener();

      LOGGER.log(log.finished());
    }
  }

  /**
   * Obtm o consumidor de mensagens. Se ele no existe, tenta cri-lo.
   * 
   * @return o consumido de mensagens ou {@code null} caso no seja possvel
   *         consumir mensagens no momento.
   */
  private IMessageConsumer getConsumer() {
    if (!loggedIn) {
      return null;
    }

    if (consumer == null) {
      CodeBlockLog log =
        new CodeBlockLog(Level.FINER, "Criando o consumidor de mensagens");
      LOGGER.log(log.begin());

      RemoteTask<IMessageConsumer> task = new RemoteTask<IMessageConsumer>() {
        @Override
        protected void performTask() throws Exception {
          setResult(createMessageConsumerStrategy());
        }
      };

      Window window = null;
      DesktopFrame mainFrame = DesktopFrame.getInstance();
      if (mainFrame != null) {
        window = mainFrame.getDesktopFrame();
      }
      if (task.execute(window, LNG.get("MESSAGE_PROXY_TITLE"),
        LNG.get("MESSAGE_PROXY_CREATING_CONSUMER"))) {
        consumer = task.getResult();
      }

      LOGGER.log(log.finished());
    }
    return consumer;
  }

  /**
   * Destri o consumidor de mensagens.
   */
  private void destroyConsumer() {
    consumer = null;
  }

  /**
   * <p>
   * Cria a estrategia de recepo de mensagens do servidor.
   * </p>
   * <p>
   * Caso o servidor tenha acesso ao cliente na porta especificada no construtor
   * desta instncia, ser cadastrado um ouvinte no servio de mensagem do
   * servidor. Caso contrrio, ser criado um processo que a cada 5 segundos
   * pede novas mensagens ao servio de mensagens. Essas estratgias so
   * implementadas respectivamente pelas classes {@link MessageListenerConsumer}
   * e {@link MessagePollingConsumer}.
   * </p>
   * 
   * @return A estrategia de recepo de mensagens do servidor.
   * 
   * @see MessageListenerConsumer
   * @see MessagePollingConsumer
   * 
   * @throws RemoteException Em caso de falha na comunicao com o servidor.
   */
  private IMessageConsumer createMessageConsumerStrategy()
    throws RemoteException {

    int rmiExportPort = Client.getInstance().getRMIExportPort();
    if (portRef.bind(rmiExportPort)) {
      IPingable server = ClientServerManager.getInstance().getServer();
      if (portRef.isReachableBy(server)) {
        return new MessageListenerConsumer(ClientRemoteLocator.messageService,
          portRef);
      }
      else {
        portRef.unbind();
      }
    }

    return new MessagePollingConsumer(ClientRemoteLocator.messageService, 5,
      TimeUnit.SECONDS);
  }
}
