package csbase.client.util.charset;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.Client;

/**
 * Singleton do conjunto bsico de charsets que podem (eventualmente e em casos
 * especficos) serem usados pelos aplicativos. Normalmente deve-se usar somente
 * o charset default do sistema definido por
 * {@link Client#getSystemDefaultCharset()}.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class StandardCharsets {

  /**
   * Instncia do singleton
   */
  private static StandardCharsets instance;

  /**
   * Lista de nomes a serem carregados como padronizados.
   */
  final private String[] charsetNames = new String[] { "US-ASCII",
      "ISO-8859-1", "UTF-8", "windows-1252", "MacRoman", "IBM500" };

  /**
   * Charsets padronizados.
   */
  final private List<Charset> charsets = new ArrayList<Charset>();

  /**
   * Monta um texto (internacionalizado) mais legvel do charset conforme
   * descrito nos arquivos de bundle. Caso o charset no seja standard (e/ou no
   * esteja no bundle), o texto ser o defindo em
   * {@code Charset#displayName(Locale)}.
   * 
   * @param charset charset
   * @return o texto
   */
  final public String getLegibleName(Charset charset) {
    final Locale locale = LNG.getLocale();
    final int index = findStandard(charset);
    if (index >= 0) {
      final String stdName = charsetNames[index];
      final String prefix = this.getClass().getSimpleName();
      final String key = prefix + "." + stdName + ".name";
      if (LNG.hasKey(key)) {
        return LNG.get(key);
      }
    }
    final String displayName = charset.displayName(locale);
    return displayName;
  }

  /**
   * Retorna a lista com todos os charsets-padro.
   * 
   * @return a lista
   */
  final public List<Charset> getAllStandardCharsets() {
    return charsets;
  }

  /**
   * Verifica se um charset  um dos padronizados (tpicos para o CSBase).
   * 
   * @param charset charset a ser consultado
   * @return o seu ndice na lista de nomes (ou negativo).
   */
  final private int findStandard(final Charset charset) {
    if (charset == null) {
      return -1;
    }
    final List<Charset> chs = getAllStandardCharsets();
    final int index = chs.indexOf(charset);
    return index;
  }

  /**
   * Verifica se um charset  um dos padronizados (tpicos para o CSBase).
   * 
   * @param charset charset cosultado
   * @return indicativo
   */
  final public boolean isStandard(final Charset charset) {
    final int index = findStandard(charset);
    final boolean flag = (index >= 0);
    return flag;
  }

  /**
   * Construtor
   */
  private StandardCharsets() {
    for (String charsetName : charsetNames) {
      if (Charset.isSupported(charsetName)) {
        final Charset charset = Charset.forName(charsetName);
        if (charset.isRegistered()) {
          charsets.add(charset);
        }
      }
    }
    Collections.unmodifiableList(charsets);
  }

  /**
   * Busca do singleton.
   * 
   * @return objeto
   */
  final public static StandardCharsets getInstance() {
    if (instance == null) {
      instance = new StandardCharsets();
    }
    return instance;
  }
}
