package csbase.client.util.csvpanel.menu;

import java.awt.Component;
import java.awt.Point;

import javax.swing.JPopupMenu;
import javax.swing.table.TableColumnModel;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.util.csvpanel.table.CSVTable;

/**
 * Classe base para os menus popoup da tabela CSV. J mapeia a coordenada do
 * evento de mouse para a linha/coluna correta da tabela.
 * 
 * @author Tecgraf
 */
public abstract class CSVTablePopupMenu extends JPopupMenu {

  /**
   * Armazena a ltima linha da tabela selecionada via menu popup.
   */
  protected int popupRow;
  /**
   * Armazena a ltima coluna da tabela selecionada via menu popup.
   */
  protected int popupCol;

  /**
   * A tabela CSV.
   */
  protected CSVTable table;

  /**
   * Construtor.
   * 
   * @param table a tabela CSV.
   */
  public CSVTablePopupMenu(CSVTable table) {
    super();
    this.table = table;
  }

  /**
   * Armazena os ltimos ndices da tabela (linha e coluna) selecionados via
   * menu popup.
   * 
   * @param p o ponto selecionado.
   */
  protected void setPopupLocation(Point p) {
    if (table.getRowCount() == 0) {
      popupRow = 0;
    }
    else {
      popupRow = table.rowAtPoint(p);
    }
    TableColumnModel columnModel = table.getColumnModel();
    popupCol = columnModel.getColumnIndexAtX((int) p.getX());
  }

  /**
   * Obtm a string localizada da chave de internacionalizao especificada.
   * 
   * @param key a chave de internacionalizao.
   * @return a string localizada.
   */
  protected String getString(String key) {
    return LNG.get(getClass().getName() + "." + key);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void show(Component invoker, int x, int y) {
    setPopupLocation(new Point(x, y));
    if (shouldBeShown()) {
      updateItemStatus();
      super.show(invoker, x, y);
    }
  }

  /**
   * Indica se o menu deve ser mostrado. Esse mtodo deve ser sobrescrito pelas
   * subclasses para permitir o controle de quando o menu deve ser mostrado de
   * acordo com o estado da tabela ou a posio na tabela.
   * 
   * @return verdadeiro se o menu deve ser realmente mostrado ou falso, caso
   *         contrrio.
   */
  protected boolean shouldBeShown() {
    return popupCol >= 0 && popupRow >= 0;
  }

  /**
   * Atualiza o estado dos itens do menu de acordo com o estado atual da tabela.
   * Permite que as subclasses indiquem quais itens devem ser
   * habilitados/desabilitados antes de mostrar o menu.
   */
  protected abstract void updateItemStatus();

}