/*
 * $Id: ClientProjectFileChooserFacilities.java 110158 2010-09-14 23:00:41Z
 * clinio $
 */
package csbase.client.util.filechooser;

import java.awt.Window;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import csbase.client.ClientSmartFile;
import csbase.client.ClientSmartFileFactory;
import csbase.client.applications.Application;
import csbase.client.applications.ApplicationFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.tasks.ChangeFileTypeTask;
import csbase.client.project.tasks.CreateFileTask;
import csbase.client.project.tasks.GetFileTask;
import csbase.client.util.filechooser.filters.ClientFileAllFilter;
import csbase.client.util.filechooser.filters.ClientFileFilterInterface;
import csbase.client.util.filechooser.filters.ClientFileMultipleTypesFilter;
import csbase.client.util.filechooser.filters.ClientFileSingleTypeFilter;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * Utilitrio para escolha de arquivos na rea de projetos.
 * 
 * @author Tecgraf/PUC-Rio.
 */
public class ClientProjectFileChooserUtil {

  /**
   * Estrutura que indica o resultado de uma navegao do tipo SAVE em arquivo.
   * 
   * @author Tecgraf
   */
  static public class OperationResult {
    /**
     * Indicativo se arquivo escolhido j existe
     */
    final private boolean existsFile;

    /**
     * Arquivo de projeto associado.
     */
    final private ClientProjectFile file;

    /**
     * Path retornado
     */
    final private String[] path;

    /**
     * Chooser.
     */
    private final ClientFileChooser chooser;

    /**
     * Retorna indicativo de existncia de arquivo selecionado.
     * 
     * @return existsFile indicativo
     */
    public final boolean existsFile() {
      return existsFile;
    }

    /**
     * Retorna o arquivo associado (se existir).
     * 
     * @return file
     */
    public final ClientProjectFile getClientProjectFile() {
      return file;
    }

    /**
     * Retorna o ClientSmartFile associado ao arquivo de projeto. O uso deste
     * arquivo possibilita a otimizao de acesso  ea de projetos
     * 
     * @return o ClientSmartFile
     */
    public ClientSmartFile getClientSmartFile() {
      if (file == null) {
        return null;
      }
      return ClientSmartFileFactory.getInstance().create(file);
    }

    /**
     * Mtodo utilitrio para montagem de path como string usando o array
     * interno definido (ver {@link #getPath()}).
     * 
     * @return o texto
     */
    final public String getStringPath() {
      final String[] pth = getPath();
      if ((pth == null) || (pth.length <= 0)) {
        return null;
      }
      StringBuilder builder = new StringBuilder();
      final int endIdx = pth.length;
      for (int i = 0; i < endIdx - 2; i++) {
        builder.append(pth[i]);
        builder.append('/');
      }
      builder.append(pth[endIdx - 1]);
      final String text = builder.toString();
      return text;
    }

    /**
     * Retorna
     * 
     * @return path
     */
    public final String[] getPath() {
      return path;
    }

    /**
     * Construtor.
     * 
     * @param chooser chooser
     * @param project projeto.
     * @param path path retornado
     * @param createChosen indicativo de criao automtica do arquivo
     */
    public OperationResult(final ClientFileChooser chooser,
      final CommonClientProject project, final String[] path,
      boolean createChosen) {
      this.chooser = chooser;
      this.path = path;
      ClientProjectFile tmpFile =
        GetFileTask.runTask(chooser.getOwner(), project, path);

      if (createChosen) {
        if (tmpFile == null) {
          this.file = createFromPath();
          this.existsFile = ((this.file == null) ? false : true);
        }
        else {
          this.file = tmpFile;
          this.existsFile = true;
          final String fileName = path[path.length - 1];
          final ProjectFileType fileType = getChosenFileType(fileName);
          ChangeFileTypeTask.runTask(chooser.getOwner(), file, fileType);
        }
      }
      else {
        this.file = tmpFile;
        this.existsFile = ((this.file == null) ? false : true);
      }
    }

    /**
     * Construtor
     * 
     * @param chooser chooser
     * @param file arquivo.
     */
    public OperationResult(final ClientFileChooser chooser,
      final ClientProjectFile file) {
      this.chooser = chooser;
      if (file == null) {
        this.path = null;
        this.existsFile = false;
      }
      else {
        this.path = file.getPath();
        this.existsFile = true;
      }
      this.file = file;
    }

    /**
     * Cria o arquivo.
     * 
     * @return o arquivo (novo) ou {@code null}.
     */
    final public ClientProjectFile createFromPath() {
      if (file != null) {
        return file;
      }

      final DesktopFrame desktopFrame = DesktopFrame.getInstance();
      final CommonClientProject project = desktopFrame.getProject();
      if (project == null) {
        return null;
      }

      if (path == null) {
        return null;
      }

      final int pathLength = path.length;
      final ClientProjectFile parent;
      if (pathLength == 1) {
        parent = project.getRoot();
      }
      else {
        final String[] parentPath = new String[pathLength - 1];
        for (int i = 0; i < parentPath.length; i++) {
          parentPath[i] = path[i];
        }
        parent = GetFileTask.runTask(chooser.getOwner(), project, parentPath);
      }
      final String fileName = path[pathLength - 1];

      final ProjectFileType tp = getChosenFileType(fileName);
      if (tp == null) {
        final String msg = getString("unsupported.file.type");
        final String extension = FileUtils.getFileExtension(fileName);
        final String err = msg + " - (." + extension + ")";
        StandardDialogs.showErrorDialog(chooser.getOwner(), "", err);
        return null;
      }

      final String fileType = tp.getCode();
      final ClientProjectFile newFile = CreateFileTask
        .runTask(chooser.getOwner(), project, parent, fileName, fileType);
      if (newFile == null) {
        return null;
      }
      return newFile;
    }

    /**
     * Busca texto de internacionalizao.
     * 
     * @param tag tag de busca.
     * @param args argumentos.
     * @return texto
     */
    private String getString(final String tag, Object... args) {
      final String prefix1 = ClientFileChooser.class.getSimpleName();
      final String prefix2 = getClass().getSimpleName();
      final String prefix = prefix1 + "." + prefix2;
      final String realTag = prefix + "." + tag;
      final String fmt = LNG.get(realTag);
      final String msg = String.format(fmt, args);
      return msg;
    }

    /**
     * Busca um tipo de arquivo com base em um nome.
     * 
     * @param fileName nome de arquivo.
     * 
     * @return o tipo.
     */
    final private ProjectFileType getChosenFileType(final String fileName) {
      final ProjectFileType filterType =
        chooser.getFileTypeFromSelectedFilter();
      if (filterType == null) {
        final ProjectFileType extensionType =
          chooser.getFileTypeFromFileName(fileName);
        return extensionType;
      }
      return filterType;
    }
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param application a aplicao.
   * @param fileTypes a lista de tipos a serem aceitos
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInOpenMode(
    final Application application, final Collection<String> fileTypes,
    final boolean allowAll) {
    final ApplicationFrame frame = application.getApplicationFrame();
    return browseSingleFileInOpenMode(frame, fileTypes, application.getName(),
      allowAll);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param fileType tipos a ser aceito
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInOpenMode(
    final Window window, final String fileType, final String title,
    final boolean allowAll) {
    final List<String> fileTypes = new ArrayList<String>();
    fileTypes.add(fileType);
    return browseSingleFileInOpenMode(window, fileTypes, title, allowAll);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInOpenMode(
    final Window window, final Collection<String> fileTypes, final String title,
    final boolean allowAll) {
    return browseSingleInOpenMode(window, title,
      ClientFileChooserSelectionMode.FILES_ONLY, allowAll, null,
      fileTypes.toArray(new String[] {}));
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param fileType tipo aceito.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleDirectoryInOpenMode(
    final Window window, final String fileType, final String title,
    final boolean allowAll) {
    return browseSingleInOpenMode(window, title,
      ClientFileChooserSelectionMode.DIRECTORIES_ONLY, allowAll, fileType);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param fileTypes a lista de tipos a serem aceitos
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleDirectoryInOpenMode(
    final Window window, final Collection<String> fileTypes, final String title,
    final boolean allowAll) {
    return browseSingleInOpenMode(window, title,
      ClientFileChooserSelectionMode.DIRECTORIES_ONLY, allowAll, null,
      fileTypes.toArray(new String[] {}));
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos ou
   * diretrios da rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param title ttulo.
   * @param selectionMode modo de seleo (arquivo ou diretrio).
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param defaultType O tipo padro.
   * @param fileTypes a lista de tipos a serem aceitos
   * @return o arquivo escolhido ou {@code null}.
   */
  public final static OperationResult browseSingleInOpenMode(Window window,
    String title, ClientFileChooserSelectionMode selectionMode,
    boolean allowAll, String defaultType, String... fileTypes) {
    ClientFileChooser chooser =
      buildChooser(window, title, ClientFileChooserType.OPEN, selectionMode,
        ClientFileChooserCardinality.SINGLE_CHOOSE, allowAll, defaultType,
        fileTypes);
    if (chooser == null) {
      return null;
    }
    chooser.setVisible(true);
    List<ClientFile> selected = chooser.getChosenItens();
    if (selected == null || selected.isEmpty()) {
      return null;
    }
    ClientProjectFile first = (ClientProjectFile) selected.get(0);
    OperationResult result = new OperationResult(chooser, first);
    return result;
  }

  private static ClientFileChooser buildChooser(Window window, String title,
    ClientFileChooserType selectionType,
    ClientFileChooserSelectionMode selectionMode,
    ClientFileChooserCardinality cardinality, boolean allowAll,
    String defaultFileType, String... fileTypes) {
    ClientFileChooser chooser = new ClientFileChooser(window);
    DesktopFrame desktop = DesktopFrame.getInstance();
    CommonClientProject project = desktop.getProject();
    if (project == null) {
      StandardDialogs.showErrorDialog(window, null, "No project!");
      return null;
    }
    ClientProjectFile root = project.getRoot();
    chooser.setCurrentDirectory(root);
    chooser.setTitle(title);
    chooser.setSelectionMode(selectionMode);
    chooser.setSelectionType(selectionType);
    chooser.setLocalHomeButtonVisible(false);
    chooser.setLocalRootButtonVisible(false);
    chooser.setProjectHomeButtonVisible(true);
    chooser.setCardinality(cardinality);

    List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();
    ClientFileFilterInterface defaultFilter = null;
    List<String> allFileTypes = new ArrayList<String>();
    if (defaultFileType != null) {
      allFileTypes.add(defaultFileType);
    }
    for (String fileType : fileTypes) {
      if (!allFileTypes.contains(fileType)) {
        allFileTypes.add(fileType);
      }
    }
    // Se no h um tipo padro, ento a primeira opo so todos os aceitos.
    if (defaultFileType == null) {
      if (allFileTypes.size() > 1) {
        filters.add(new ClientFileMultipleTypesFilter(allFileTypes));
      }
    }
    for (final String fileType : allFileTypes) {
      ClientFileSingleTypeFilter filter =
        new ClientFileSingleTypeFilter(fileType);
      filters.add(filter);
      if (defaultFileType != null && defaultFileType.equals(fileType)) {
        defaultFilter = filter;
      }
    }
    // Se no h tipo padro, ento a penltima ou ltima opo so todos os 
    // aceitos.
    if (defaultFileType != null) {
      if (allFileTypes.size() > 1) {
        filters.add(new ClientFileMultipleTypesFilter(allFileTypes));
      }
    }
    //  a ltima opo se for para aceitar qualquer tipo.
    if (allowAll) {
      filters.add(new ClientFileAllFilter());
    }
    chooser.setViewFilters(filters);
    if (defaultFilter != null) {
      chooser.setSelectedFilter(defaultFilter);
    }
    return chooser;
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param application a aplicao.
   * @param fileType tipo a ser aceitos
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInSaveMode(
    final Application application, final String fileType,
    final boolean allowAll, final boolean createChosen) {
    final ApplicationFrame frame = application.getApplicationFrame();
    return browseSingleFileInSaveMode(frame, fileType, application.getName(),
      allowAll, createChosen);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param application a aplicao.
   * @param fileTypes tipos a ser aceitos
   * @param defaultFileType tipos default aceito.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInSaveMode(
    final Application application, final Collection<String> fileTypes,
    String defaultFileType, final boolean allowAll,
    final boolean createChosen) {
    final ApplicationFrame frame = application.getApplicationFrame();
    return browseSingleFileInSaveMode(frame, fileTypes, defaultFileType,
      application.getName(), allowAll, createChosen);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me.
   * @param fileType tipo a ser aceito
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInSaveMode(
    final Window window, final String fileType, final String title,
    final boolean allowAll, final boolean createChosen) {
    final List<String> fileTypes = new ArrayList<String>();
    if (fileType != null) {
      fileTypes.add(fileType);
    }
    return browseSingleFileInSaveMode(window, fileTypes, fileType, title,
      allowAll, createChosen);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me.
   * @param fileTypes a lista de tipos a serem aceitos
   * @param defaultFileType tipo default aceito.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleFileInSaveMode(
    final Window window, final Collection<String> fileTypes,
    String defaultFileType, final String title, final boolean allowAll,
    final boolean createChosen) {
    return browseSingleInSaveMode(window, title,
      ClientFileChooserSelectionMode.FILES_ONLY, createChosen, allowAll,
      defaultFileType, fileTypes.toArray(new String[] {}));
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de diretrios da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me.
   * @param fileType tipo a ser aceito
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleDirectoryInSaveMode(
    final Window window, final String fileType, final String title,
    final boolean allowAll, final boolean createChosen) {
    final List<String> fileTypes = new ArrayList<String>();
    if (fileType != null) {
      fileTypes.add(fileType);
    }
    return browseSingleDirectoryInSaveMode(window, fileTypes, fileType, title,
      allowAll, createChosen);
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de diretrios da
   * rea de projetos com base em lista de tipos aceitveis.
   * 
   * @param window a janela-me.
   * @param fileTypes a lista de tipos a serem aceitos
   * @param defaultFileType tipo default aceito.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @return o arquivo escolhido ou {@code null}.
   */
  final static public OperationResult browseSingleDirectoryInSaveMode(
    final Window window, final Collection<String> fileTypes,
    String defaultFileType, final String title, final boolean allowAll,
    final boolean createChosen) {
    return browseSingleInSaveMode(window, title,
      ClientFileChooserSelectionMode.DIRECTORIES_ONLY, createChosen, allowAll,
      defaultFileType, fileTypes.toArray(new String[] {}));
  }

  /**
   * Mtodo utilitrio para ser usado por aplicaes na escolha de arquivos ou
   * diretrios de sada na rea de projetos.
   * 
   * @param window a janela-me.
   * @param title ttulo.
   * @param selectionMode O modo de selelo.
   * @param createChosen indicativo de criao automtica do novo arquivo
   *        escolhido, se inexistente.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param defaultFileType tipo default aceito.
   * @param fileTypes a lista de tipos a serem aceitos
   * @return o arquivo escolhido ou {@code null}.
   */
  public static final OperationResult browseSingleInSaveMode(Window window,
    String title, ClientFileChooserSelectionMode selectionMode,
    boolean createChosen, boolean allowAll, String defaultFileType,
    String... fileTypes) {
    ClientFileChooser chooser =
      buildChooser(window, title, ClientFileChooserType.SAVE, selectionMode,
        ClientFileChooserCardinality.SINGLE_CHOOSE, allowAll, defaultFileType,
        fileTypes);
    if (chooser == null) {
      return null;
    }
    chooser.setVisible(true);
    String[] selectedSavePath = chooser.getSelectedSavePath();
    if (selectedSavePath == null || selectedSavePath.length <= 0) {
      return null;
    }
    CommonClientProject project = DesktopFrame.getInstance().getProject();
    final OperationResult result =
      new OperationResult(chooser, project, selectedSavePath, createChosen);
    return result;
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de arquivos da projeto com base
   * em uma lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param fileTypes os tipos aceitos.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return lista de arquivos escolhidos ou {@code null}.
   */
  final static public List<OperationResult> browseMultipleFilesInOpenMode(
    final Window window, final Collection<String> fileTypes, final String title,
    final boolean allowAll) {
    return browseMultipleInOpenMode(window, title,
      ClientFileChooserSelectionMode.FILES_ONLY, allowAll, null,
      fileTypes.toArray(new String[] {}));
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de diretrios da projeto com
   * base em um tipo aceitvel.
   * 
   * @param window a janela-me (owner).
   * @param fileType o tipo aceito.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return lista de diretrios escolhidos ou {@code null}.
   */
  final static public List<OperationResult> browseMultipleDirectoriesInOpenMode(
    final Window window, final String fileType, final String title,
    final boolean allowAll) {
    return browseMultipleInOpenMode(window, title,
      ClientFileChooserSelectionMode.DIRECTORIES_ONLY, allowAll, fileType);
  }

  /**
   * Mtodo utilitrio para ser usado na escolha de diretrios da projeto com
   * base em uma lista de tipos aceitveis.
   * 
   * @param window a janela-me (owner).
   * @param fileTypes os tipos aceitos.
   * @param title ttulo.
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @return lista de diretrios escolhidos ou {@code null}.
   */
  final static public List<OperationResult> browseMultipleDirectoriesInOpenMode(
    final Window window, final Collection<String> fileTypes, final String title,
    final boolean allowAll) {
    return browseMultipleInOpenMode(window, title,
      ClientFileChooserSelectionMode.DIRECTORIES_ONLY, allowAll, null,
      fileTypes.toArray(new String[] {}));
  }

  /**
   * Mtodo para ser usado na escolha mtiplos itens de seleo homognicos
   * 
   * @param window a janela-pai (owner).
   * @param title ttulo.
   * @param selectionMode modo de seleo (arquivo ou diretrio).
   * @param allowAll indicativo de incluso de filtro "todos os arquivos".
   * @param defaultFileType O tipo padro.
   * @param fileTypes tipos a ser aceitos
   * @return o arquivo escolhido ou {@code null}.
   */
  public final static List<OperationResult> browseMultipleInOpenMode(
    Window window, String title, ClientFileChooserSelectionMode selectionMode,
    boolean allowAll, String defaultFileType, String... fileTypes) {
    final ClientFileChooser chooser =
      buildChooser(window, title, ClientFileChooserType.OPEN, selectionMode,
        ClientFileChooserCardinality.MULTIPLE_CHOOSE, allowAll, defaultFileType,
        fileTypes);
    if (chooser == null) {
      return null;
    }
    chooser.setVisible(true);
    final List<ClientFile> selected = chooser.getChosenItens();
    if (selected == null || selected.isEmpty()) {
      return null;
    }
    final List<OperationResult> results = new ArrayList<OperationResult>();
    for (ClientFile sel : selected) {
      final OperationResult result =
        new OperationResult(chooser, (ClientProjectFile) sel);
      results.add(result);
    }
    return results;
  }
}
